/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

public abstract class AudioDrawing {

	// Channel は実際にはここで定義する必要はないが、
	// AudioSpectrum, AudioWaveform のどちらでも使用するので、ここで定義している。
	public enum Channel { MONO, LEFT, RIGHT }

	public enum Style { BARS, LINES, DOTS }


	protected final IVideoEffectContext context;

	protected final IVideoRenderSupport support;

	protected final IBlendSupport blendSupport;

	protected AudioDrawing(IVideoEffectContext context, IVideoRenderSupport support, IBlendSupport blendSupport) {
		this.context = context;
		this.support = support;
		this.blendSupport = blendSupport;
		blendSupport.setProgramsClass(blendSupport.getUnmultSrcClass());
	}

	protected interface DataProvider {
		void init();
		void dispose();
		int getDataCount();
		float getDataLower(int index);
		float getDataUpper(int index);
		float getData(int index);
	}

	protected IVideoBuffer draw(
			DataProvider dataProvider, IAnimatableVec2d aStartPoint, IAnimatableVec2d aEndPoint,
			IAnimatableDouble aHeight, IAnimatableDouble aThickness, IAnimatableColor aColor,
			IAnimatableEnum<Style> aStyle, IAnimatableBoolean aSmoothing,
			IAnimatableEnum<BlendMode> aBlendMode, IAnimatableDouble aOpacity) {

		BlendMode blendMode = context.value(aBlendMode);

		IVideoBuffer original;
		VideoBounds bounds;

		if (blendMode == BlendMode.NONE) {
			bounds = context.getPreviousBounds();
			original = context.createVideoBuffer(bounds);
		} else {
			original = context.doPreviousEffect();
			bounds = original.getBounds();
		}

		if (bounds.isEmpty()) {
			return original;
		}

		IVideoBuffer drawing = null;
		try {
			dataProvider.init();

			Resolution resolution = context.getVideoResolution();

			drawing = draw(bounds, dataProvider,
					resolution.scale(context.value(aStartPoint)),
					resolution.scale(context.value(aEndPoint)),
					resolution.scale(context.value(aHeight)),
					resolution.scale(context.value(aThickness)),
					context.value(aColor),
					context.value(aStyle),
					context.value(aSmoothing));

			double opacity = context.value(aOpacity) / 100;

			return blendSupport.blend(drawing, original, null, blendMode, opacity, context);

		} finally {
			if (original != null) original.dispose();
			if (drawing != null) drawing.dispose();
			dataProvider.dispose();
		}
	}

	private IVideoBuffer draw(
			final VideoBounds bounds, final DataProvider dataProvider,
			final Vec2d startPoint, final Vec2d endPoint,
			final double height, final double thickness, final Color color,
			final Style style, final boolean smoothing) {

		IVideoBuffer buffer = null;
		try {
			Runnable operation = new Runnable() {
				public void run() {
					support.ortho2D(bounds);
					draw(dataProvider, startPoint, endPoint, height, thickness, color, style, smoothing);
				}
			};

			int pushAttribs = GL2.GL_HINT_BIT			// GL_LINE_SMOOTH_HINT, GL_POINT_SMOOTH_HINT
							| GL2.GL_CURRENT_BIT		// GL_CURRENT_COLOR
							| GL2.GL_COLOR_BUFFER_BIT	// GL_BLEND, GL_BLEND_SRC_RGB, GL_BLEND_SRC_ALPHA, GL_BLEND_DST_RGB, GL_BLEND_DST_ALPHA, GL_BLEND_EQUATION
							| GL2.GL_POINT_BIT			// GL_POINT_SIZE, GL_POINT_SMOOTH
							| GL2.GL_LINE_BIT			// GL_LINE_WIDTH, GL_LINE_SMOOTH
							| GL2.GL_DEPTH_BUFFER_BIT	// GL_DEPTH_TEST
							| GL2.GL_ENABLE_BIT;		// GL_DEPTH_TEST, GL_BLEND, GL_LINE_SMOOTH, GL_POINT_SMOOTH

			buffer = context.createVideoBuffer(bounds);
			buffer.clear();

			support.useFramebuffer(operation, pushAttribs, buffer);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	private void draw(
			DataProvider dataProvider,
			Vec2d startPoint, Vec2d endPoint,
			double height, double thickness, Color color,
			Style style, boolean smoothing) {

		GL2 gl = context.getGL().getGL2();


		double dx = endPoint.x-startPoint.x;
		double dy = endPoint.y-startPoint.y;

		gl.glMatrixMode(GL2.GL_MODELVIEW);
		gl.glTranslatef((float)startPoint.x, (float)startPoint.y, 0);
		gl.glRotatef((float)Math.toDegrees(Math.atan2(dy, dx)), 0, 0, 1);
		gl.glScalef((float)Math.sqrt(dx*dx+dy*dy), (float)height, 1);


		if (smoothing) {
			thickness = Math.min(thickness, getMaxSmoothThickness());

			gl.glDisable(GL2.GL_DEPTH_TEST);	// もともと無効になっているはずだけど。

			gl.glEnable(GL2.GL_LINE_SMOOTH);
			gl.glEnable(GL2.GL_POINT_SMOOTH);
			gl.glHint(GL2.GL_LINE_SMOOTH_HINT, GL2.GL_NICEST);
			gl.glHint(GL2.GL_POINT_SMOOTH_HINT, GL2.GL_NICEST);

			gl.glEnable(GL2.GL_BLEND);
			gl.glBlendFunc(GL2.GL_ONE, GL2.GL_ONE);
			gl.glBlendEquation(GL2.GL_MAX);
		}

		thickness = Math.max(1e-10, thickness);
		gl.glLineWidth((float)thickness);
		gl.glPointSize((float)thickness);

		color = color.clamp();
		float a = (float)color.a;
		float r = (float)color.r * a;
		float g = (float)color.g * a;
		float b = (float)color.b * a;
		gl.glColor4f(r, g, b, a);


		int dataCount = dataProvider.getDataCount();

		switch (style) {
			case BARS: {
				float adjustment = (float)(smoothing ? 0 : 1/height);

				gl.glBegin(GL2.GL_LINES);
				for (int i = 0; i < dataCount; ++i) {
					gl.glVertex2f((float)i/(dataCount-1), dataProvider.getDataLower(i));
					gl.glVertex2f((float)i/(dataCount-1), dataProvider.getDataUpper(i) - adjustment);
				}
				gl.glEnd();

				if (smoothing) {
					gl.glBegin(GL2.GL_POINTS);
					for (int i = 0; i < dataCount; ++i) {
						gl.glVertex2f((float)i/(dataCount-1), dataProvider.getDataLower(i));
						gl.glVertex2f((float)i/(dataCount-1), dataProvider.getDataUpper(i) - adjustment);
					}
					gl.glEnd();
				}
				break;
			}

			case LINES:
				gl.glBegin(GL2.GL_LINE_STRIP);
				if (dataCount > 1) {
					for (int i = 0; i < dataCount; ++i) {
						gl.glVertex2f((float)i/(dataCount-1), dataProvider.getData(i));
					}
				} else {
					float data = dataProvider.getData(0);
					gl.glVertex2f(0, data);
					gl.glVertex2f(1, data);
				}
				gl.glEnd();

				if (!smoothing) break;
				// fall through

			case DOTS:
				gl.glBegin(GL2.GL_POINTS);
				if (dataCount > 1) {
					for (int i = 0; i < dataCount; ++i) {
						gl.glVertex2f((float)i/(dataCount-1), dataProvider.getData(i));
					}
				} else if (style == Style.LINES) {
					float data = dataProvider.getData(0);
					gl.glVertex2f(0, data);
					gl.glVertex2f(1, data);
				} else {
					gl.glVertex2f(0.5f, dataProvider.getData(0));
				}
				gl.glEnd();
				break;
		}
	}


	private static volatile double maxSmoothThickness;

	private double getMaxSmoothThickness() {
		if (maxSmoothThickness == 0) {
			synchronized (AudioDrawing.class) {
				if (maxSmoothThickness == 0) {
					GL2 gl = context.getGL().getGL2();
					float[] range = new float[2];
					gl.glGetFloatv(GL2.GL_SMOOTH_LINE_WIDTH_RANGE, range, 0);
					double max = range[1];
					gl.glGetFloatv(GL2.GL_SMOOTH_POINT_SIZE_RANGE, range, 0);
					maxSmoothThickness = Math.min(max, range[1]);
				}
			}
		}
		return maxSmoothThickness;
	}

}
