/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.channel;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Invert", category=Categories.CHANNEL)
public class Invert {

	public enum Channel {
		RGB, RED, GREEN, BLUE,
		HSL, HUE, SATURATION, LUMINOSITY,
		ALPHA
	}

	@Property
	private IAnimatableEnum<Channel> channel;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble blendWithOriginal;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final Map<Channel, IShaderProgram> programs = new HashMap<Channel, IShaderProgram>();

	@Inject
	public Invert(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;

		for (Channel ch : Channel.values()) {
			programs.put(ch, shaders.getProgram(Invert.class, ch.name()));
		}
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		if (source.getBounds().isEmpty()) {
			return source;
		}

		Channel ch = context.value(this.channel);
		double blend = context.value(this.blendWithOriginal) / 100;

		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		uniforms.add(new GLUniformData("texture", 0));
		uniforms.add(new GLUniformData("blend", (float)blend));

		try {
			return support.useShaderProgram(programs.get(ch), uniforms, null, source);
		} finally {
			source.dispose();
		}
	}


	private static String[] createRGBASource(Channel channel) {
		String rgbNew, alphaNew;
		switch (channel) {
			case RGB:			rgbNew = "(vec3(1.0)-rgb)";					alphaNew = "src.a";			break;
			case RED:			rgbNew = "vec3(1.0-rgb.r, rgb.gb)";			alphaNew = "src.a";			break;
			case GREEN:			rgbNew = "vec3(rgb.r, 1.0-rgb.g, rgb.b)";	alphaNew = "src.a";			break;
			case BLUE:			rgbNew = "vec3(rgb.rg, 1.0-rgb.b)";			alphaNew = "src.a";			break;
			case ALPHA:			rgbNew = "rgb";								alphaNew = "(1.0-src.a)";	break;
			default: throw new IllegalArgumentException();
		}

		return new String[] {
						"uniform sampler2DRect texture;",
						"uniform float blend;",
						"",
						"void main(void)",
						"{",
						"	vec4 src = texture2DRect(texture, gl_TexCoord[0].st);",
						"	vec3 rgb = (src.a != 0.0) ? src.rgb/src.a : vec3(0.0);",
		String.format(	"	gl_FragColor = vec4(%s, 1.0)*%s*(1.0-blend) + src*blend;", rgbNew, alphaNew),
						"}"
		};
	}

	@ShaderSource
	public static final String[] RGB = createRGBASource(Channel.RGB);

	@ShaderSource
	public static final String[] RED = createRGBASource(Channel.RED);

	@ShaderSource
	public static final String[] GREEN = createRGBASource(Channel.GREEN);

	@ShaderSource
	public static final String[] BLUE = createRGBASource(Channel.BLUE);

	@ShaderSource
	public static final String[] ALPHA = createRGBASource(Channel.ALPHA);


	private static String[] createHSLSource(Channel channel) {
		String hslNew;
		switch (channel) {
			case HSL:			hslNew = "vec3(1.5, 1.0, 1.0) - hsl";		break;
			case HUE:			hslNew = "vec3(1.5-hsl.x, hsl.yz)";			break;
			case SATURATION:	hslNew = "vec3(hsl.x, 1.0-hsl.y, hsl.z)";	break;
			case LUMINOSITY:	hslNew = "vec3(hsl.xy, 1.0-hsl.z)";			break;
			default: throw new IllegalArgumentException();
		}

		return new String[] {
						"uniform sampler2DRect texture;",
						"uniform float blend;",
						"",
						"vec3 rgb2hsl(vec3 rgb);",
						"vec3 hsl2rgb(vec3 rgb);",
						"",
						"void main(void)",
						"{",
						"	vec4 src = texture2DRect(texture, gl_TexCoord[0].st);",
						"	vec3 rgb = (src.a != 0.0) ? src.rgb/src.a : vec3(0.0);",
						"	vec3 hsl = rgb2hsl(rgb);",
		String.format(	"	rgb = hsl2rgb(%s);", hslNew),
						"	gl_FragColor = vec4(rgb, 1.0)*src.a*(1.0-blend) + src*blend;",
						"}"
		};
	}

	// TODO KeyingShadersにあるrgb2hslとFractalNoiseにあるhsl2rgbをどこか適切な場所に移動する。
	private static final String RGB2HSL = "ch.kuramo.javie.effects.keying.KeyingShaders.rgb2hsl";
	private static final String HSL2RGB = "ch.kuramo.javie.effects.noiseGrain.FractalNoise.hsl2rgb";

	@ShaderSource(attach={ RGB2HSL, HSL2RGB })
	public static final String[] HSL = createHSLSource(Channel.HSL);

	@ShaderSource(attach={ RGB2HSL, HSL2RGB })
	public static final String[] HUE = createHSLSource(Channel.HUE);

	@ShaderSource(attach={ RGB2HSL, HSL2RGB })
	public static final String[] SATURATION = createHSLSource(Channel.SATURATION);

	@ShaderSource(attach={ RGB2HSL, HSL2RGB })
	public static final String[] LUMINOSITY = createHSLSource(Channel.LUMINOSITY);

}
