/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.output;

import java.io.File;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.AudioMode.DataType;
import ch.kuramo.javie.core.Composition;

public class MacOSXQTMovieOutput extends PullSourceOutput {

	private static final Logger _logger = LoggerFactory.getLogger(MacOSXQTMovieOutput.class);


	private class VideoRequestWithByteBuffer extends VideoRequest {
		private final ByteBuffer byteBuffer;
		private VideoRequestWithByteBuffer(Time time, ByteBuffer byteBuffer) {
			super(time);
			this.byteBuffer = byteBuffer;
		}
	}

	private class AudioRequestWithByteBuffer extends AudioRequest {
		private final ByteBuffer byteBuffer;
		private AudioRequestWithByteBuffer(Time time, int frameCount, ByteBuffer byteBuffer) {
			super(time, frameCount);
			this.byteBuffer = byteBuffer;
		}
	}


	private VideoCompressorSettings vcSettings;

	private AudioCompressorSettings acSettings;


	public VideoCompressorSettings getVideoCompressorSettings() {
		return vcSettings;
	}

	public void setVideoCompressorSettings(VideoCompressorSettings vcSettings) {
		this.vcSettings = vcSettings;
	}

	public AudioCompressorSettings getAudioCompressorSettings() {
		return acSettings;
	}

	public void setAudioCompressorSettings(AudioCompressorSettings acSettings) {
		this.acSettings = acSettings;
	}

	@Override
	protected void doOutput() {
		Composition comp = getComposition();
		AudioMode audioMode = getAudioMode();
		File file = getFile();

		// QuickTime の書き出しは、既存ファイルを削除しないと容量が増えていく？ので事前に削除する。
		if (file.exists() && !file.delete()) {
			// TODO 削除失敗時は、ユニークな名前の一時ファイルを作ってそこへ書き出し後でリネームする、等を試みる？
			//      というか、既存ファイル云々に関係なく常にそうすべきか。
		}

		Size2i size = comp.getSize();
		Time frameDuration = comp.getFrameDuration();
		Time duration = comp.getDuration();

		if (duration.timeScale != frameDuration.timeScale) {
			duration = Time.fromFrameNumber(duration.toFrameNumber(frameDuration), frameDuration);
		}

		int error;

		boolean videoEnabled = isVideoEnabled();
		boolean audioEnabled = isAudioEnabled();

		if (videoEnabled && audioEnabled) {
			error = doOutput(
						(vcSettings != null ? vcSettings.settings : null),
						(acSettings != null ? acSettings.settings : null),
						file.getAbsolutePath(),
						size.width, size.height, frameDuration.timeScale, frameDuration.timeValue, duration.timeValue,
						audioMode.channels, audioMode.sampleRate, audioMode.sampleSize,
						(audioMode.dataType == DataType.FLOAT), duration.toFrameNumber(audioMode.sampleDuration));

		} else if (videoEnabled) {
			error = doOutput(
						(vcSettings != null ? vcSettings.settings : null), null, file.getAbsolutePath(),
						size.width, size.height, frameDuration.timeScale, frameDuration.timeValue, duration.timeValue,
						0, 0, 0, false, 0);

		} else if (audioEnabled) {
			error = doOutput(
						null, (acSettings != null ? acSettings.settings : null), file.getAbsolutePath(),
						0, 0, 0, 0, 0,
						audioMode.channels, audioMode.sampleRate, audioMode.sampleSize,
						(audioMode.dataType == DataType.FLOAT), duration.toFrameNumber(audioMode.sampleDuration));

		} else {
			// PullSourceOutput内でチェックしているので、ここに来ることは無い。
			throw new Error();
		}

		if (error != 0 && error != -128) {
			_logger.error("doOutput: error=" + error);
		}
	}

	@Override
	protected void finishVideoRequest(VideoRequest request, IVideoBuffer vb) {
		copyVideoBufferToByteBuffer(vb, ((VideoRequestWithByteBuffer) request).byteBuffer);
	}

	@Override
	protected void finishAudioRequest(AudioRequest request, IAudioBuffer ab) {
		copyAudioBufferToByteBuffer(ab, ((AudioRequestWithByteBuffer) request).byteBuffer);
	}

	protected void copyVideoBufferToByteBuffer(IVideoBuffer vb, ByteBuffer bb) {
		// TODO IVideoBufferから配列に一度コピーせず、直接ByteBufferに書き込めるようにする。

		Object array = vb.getArray();

		bb.order(ByteOrder.nativeOrder());

		if (array instanceof byte[]) {
			bb.put((byte[]) array, 0, bb.capacity());
			
		} else if (array instanceof short[]) {
			bb.asShortBuffer().put((short[]) array, 0, bb.capacity() / 2);

		} else if (array instanceof float[]) {
			bb.asFloatBuffer().put((float[]) array, 0, bb.capacity() / 4);

		} else {
			throw new IllegalArgumentException(
					"not array or unsupported array type: "
					+ array.getClass().getName());
		}

		bb.position(bb.capacity());
	}

	protected void copyAudioBufferToByteBuffer(IAudioBuffer ab, ByteBuffer bb) {
		bb.order(ByteOrder.nativeOrder());

		AudioMode audioMode = ab.getAudioMode();
		int dataLen = ab.getDataLength();

		switch (audioMode.dataType) {
			case SHORT:
				bb.asShortBuffer().put((short[])ab.getData(), 0, dataLen);
				break;
			case INT:
				bb.asIntBuffer().put((int[])ab.getData(), 0, dataLen);
				break;
			case FLOAT:
				bb.asFloatBuffer().put((float[])ab.getData(), 0, dataLen);
				break;
			default:
				throw new IllegalArgumentException(
						"unsupported AudioMode: " + audioMode);
		}

		bb.position(dataLen * audioMode.sampleSize);
	}


	@SuppressWarnings("unused")
	private int videoRequest(long timeValue, int timeScale, ByteBuffer buffer) {
		try {
			buffer.clear();
			postVideoRequest(new VideoRequestWithByteBuffer(new Time(timeValue, timeScale), buffer));
			return buffer.position();
		} catch (InterruptedException e) {
			_logger.warn("unexpected interruption", e);
			return -1;
		}
	}

	@SuppressWarnings("unused")
	private int audioRequest(long timeValue, int timeScale, ByteBuffer buffer) {
		// タイムスケールの値と同じだけのフレーム数(=1秒分)に必要なバッファをネイティブコード側で確保している。
		int frameCount = timeScale;

		try {
			buffer.clear();
			postAudioRequest(new AudioRequestWithByteBuffer(new Time(timeValue, timeScale), frameCount, buffer));
			return buffer.position();
		} catch (InterruptedException e) {
			_logger.warn("unexpected interruption", e);
			return -1;
		}
	}

	@SuppressWarnings("unused")
	private boolean updateProgress(short message, int percentDone) {
		if (isCanceled()) return false;

		switch (message) {
			case 0:
				beginTask(0x00010000);
				break;
			case 1:
				currentWork(percentDone);
				break;
			case 2:
				done();
				break;
		}

		return true;
	}


	public static class VideoCompressorSettings {
		private final byte[] settings;
		public final int depth;
		public final double frameRate;
		public final String text;

		private VideoCompressorSettings(byte[] settings, int depth, double frameRate, String text) {
			this.settings = settings;
			this.depth = depth;
			this.frameRate = frameRate;
			this.text = text;
		}
	}

	public static VideoCompressorSettings doVideoCompressorSettings(
			VideoCompressorSettings defaultSettings, double frameRate, boolean showDialog) {

		Object[] newSettings = new Object[4];

		int error = doVideoCompressorSettings(
							defaultSettings != null ? defaultSettings.settings : null,
							frameRate, showDialog, newSettings);

		switch (error) {
			case 0:
				return new VideoCompressorSettings(
						(byte[])newSettings[0], (Integer)newSettings[1],
						(Double)newSettings[2], (String)newSettings[3]);
			case 1:
				// ビデオ圧縮の設定ダイアログは、キャンセル時になぜか-128ではなく1を返してくる。
				return null;
			default:
				_logger.error("doVideoCompressorSettings: error=" + error);
				return null;
		}
	}

	public static class AudioCompressorSettings {
		private final byte[] settings;
		public final double sampleRate;
		public final String text;

		private AudioCompressorSettings(byte[] settings, double sampleRate, String text) {
			this.settings = settings;
			this.sampleRate = sampleRate;
			this.text = text;
		}
	}

	public static AudioCompressorSettings doAudioCompressorSettings(
			AudioCompressorSettings defaultSettings, double sampleRate, boolean showDialog) {

		Object[] newSettings = new Object[3];

		int error = doAudioCompressorSettings(
							defaultSettings != null ? defaultSettings.settings : null,
							sampleRate, showDialog, newSettings);

		switch (error) {
			case 0:
				return new AudioCompressorSettings(
						(byte[])newSettings[0], (Double)newSettings[1], (String)newSettings[2]);
			case -128:
				// userCanceledErr
				return null;
			default:
				_logger.error("doAudioCompressorSettings: error=" + error);
				return null;
		}
	}


	static { System.loadLibrary("QTMovieOutput"); }

	private static native int doVideoCompressorSettings(
			byte[] defaultSettings, double defaultFrameRate, boolean showDialog, Object[] newSettings);

	private static native int doAudioCompressorSettings(
			byte[] defaultSettings, double defaultSampleRate, boolean showDialog, Object[] newSettings);

	private native int doOutput(
			byte[] videoSettings, byte[] audioSettings, String filename,
			int vWidth, int vHeight, int vTimeScale, long vFrameDuration, long vDuration,
			int aChannels, int aSampleRate, int aSampleSize, boolean aFloat, long aDuration);

}
