/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.awt.Graphics2D;
import java.awt.Transparency;
import java.awt.color.ColorSpace;
import java.awt.image.BufferedImage;
import java.awt.image.ComponentColorModel;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.DataBufferUShort;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.imageio.ImageIO;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.FrameDuration;
import ch.kuramo.javie.core.MediaOptions;
import ch.kuramo.javie.core.MediaSource;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.MediaOptions.Option;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;
import com.ibm.icu.text.Normalizer;

public class ImageSequenceSource implements MediaSource {

	private static final Logger _logger = LoggerFactory.getLogger(ImageSequenceSource.class);


	private File[] _files;

	private VideoBounds _videoBounds;


	private final RenderContext _context;

	private final VideoRenderSupport _vrSupport;

	@Inject
	public ImageSequenceSource(RenderContext context, VideoRenderSupport vrSupport) {
		_context = context;
		_vrSupport = vrSupport;
	}

	public boolean initialize(File file) {
		Pattern pattern1 = Pattern.compile("^(.*?)(\\d+)?(\\.[^.]+)?$");
		Matcher matcher = pattern1.matcher(Normalizer.normalize(file.getName(), Normalizer.NFKC));

		// pattern1の正規表現は必ずマッチする。
		matcher.matches();

		String group1 = matcher.group(1);
		String group2 = matcher.group(2);
		String group3 = matcher.group(3);

		if (group2.length() > 0) {
			final Map<Integer, File> fileMap = Util.newMap();
			final int[] minMax = { Integer.MAX_VALUE, Integer.MIN_VALUE };

			final Pattern pattern2 = Pattern.compile(
					String.format("^\\Q%s\\E(\\d+)\\Q%s\\E$", group1, group3), Pattern.CASE_INSENSITIVE);
			file.getParentFile().listFiles(new FileFilter() {
				public boolean accept(File file) {
					Matcher matcher = pattern2.matcher(Normalizer.normalize(file.getName(), Normalizer.NFKC));
					if (matcher.matches()) {
						Integer number = Integer.valueOf(matcher.group(1));
						if (fileMap.containsKey(number)) {
							// TODO 重複していることの警告をダイアログで出す。
							_logger.info("duplicated: " + file.getName());
						} else {
							fileMap.put(number, file);
							minMax[0] = Math.min(minMax[0], number);
							minMax[1] = Math.max(minMax[1], number);
						}
					}
					return false;
				}
			});

			if (!fileMap.isEmpty()) {
				_files = new File[minMax[1] - minMax[0] + 1];

				int lacked = 0;
				for (int i = 0, n = _files.length; i < n; ++i) {
					_files[i] = fileMap.get(i+minMax[0]);
					if (_files[i] == null) {
						++lacked;
						_logger.info("lacked: frameNumber=" + i);
					}
				}
				if (lacked > 0) {
					// TODO 欠落があることの警告をダイアログで出す。
					_logger.info("lacked: " + lacked + (lacked==1 ? " file" : " files"));
				}
			}

		} else if (group3.length() > 0) {
			final Pattern pattern2 = Pattern.compile(
					String.format("^.*?\\Q%s\\E$", group3), Pattern.CASE_INSENSITIVE);
			File[] files = file.getParentFile().listFiles(new FileFilter() {
				public boolean accept(File file) {
					return pattern2.matcher(Normalizer.normalize(file.getName(), Normalizer.NFKC)).matches();
				}
			});

			if (files != null && files.length > 0) {
				Arrays.sort(files, new Comparator<File>() {
					public int compare(File o1, File o2) {
						return o1.getName().compareTo(o2.getName());
					}
				});
				_files = files;
			}
		}

		if (_files == null) {
			return false;
		}

		try {
			BufferedImage image = ImageIO.read(_files[0]);
			if (image == null) {
				_logger.warn("error reading image file: ImageIO.read returns null");
				return false;
			}
			_videoBounds = new VideoBounds(image.getWidth(), image.getHeight());

		} catch (IOException e) {
			_logger.warn("error reading image file", e);
			return false;
		}

		return true;
	}

	public void dispose() {
		// nothing to do
	}

	public Time getDuration(MediaOptions options) {
		if (options == null) {
			return null;
		}

		Time vfd = getVideoFrameDuration(options);
		return new Time(vfd.timeValue*_files.length, vfd.timeScale);
	}

	public Time getVideoFrameDuration(MediaOptions options) {
		if (options == null) {
			return null;
		}

		Time vfd = options.getVideoFrameDuration();
		return (vfd != null) ? vfd : FrameDuration.FPS_29_97;
	}

	public VideoBounds getVideoFrameBounds(MediaOptions options) {
		return _videoBounds;
	}

	public IVideoBuffer getVideoFrame(Time mediaTime, MediaOptions options) {
		Resolution resolution = _context.getVideoResolution();
		Size2i scaledSize = resolution.scale(new Size2i(_videoBounds.width, _videoBounds.height));

		BufferedImage image = null;
		for (long frameNumber = mediaTime.toFrameNumber(getVideoFrameDuration(options));
				image == null && frameNumber >= 0; --frameNumber) {

			File file = _files[(int)frameNumber];
			if (file == null) {
				// TODO 初期化時に検索して以降、ファイルが新たに作成されているかもしれないのでここでも探すべき。
				continue;
			}

			try {
				image = ImageIO.read(file);
				if (image == null) {
					_logger.warn("error reading image file: ImageIO.read returns null");
				}
			} catch (IOException e) {
				_logger.warn("error reading image file", e);
			}
		}

		ColorMode colorMode;
		Object array;
		if (image != null) {
			if (image.getColorModel().getPixelSize() > 32) {
				colorMode = ColorMode.RGBA16;
			} else {
				colorMode = ColorMode.RGBA8;
			}
			array = toInternalFormat(image, colorMode, scaledSize);

		} else {
			// ここに来るのはフレーム0が読み込めないときだけなので、IArrayPoolsを使う必要もないだろう。
			array = new byte[scaledSize.width*scaledSize.height*4];
			colorMode = ColorMode.RGBA8;
		}

		ColorMode contextColorMode = _context.getColorMode();
		IVideoBuffer videoBuffer = _vrSupport.createVideoBuffer(contextColorMode, scaledSize);

		videoBuffer.copyToTexture(array, colorMode);

		return videoBuffer;
	}

	public boolean isVideoAvailable() {
		return true;
	}

	public boolean isAudioAvailable() {
		return false;
	}

	public IAudioBuffer getAudioChunk(Time mediaTime, MediaOptions options) {
		throw new UnsupportedOperationException("audio is not available");
	}

	public MediaOptions validateOptions(MediaOptions options) {
		Option[] availableOptions = {
				// TODO アルファと上下反転も実装する
				Option.VIDEO_FRAME_DURATION
		};
		if (options == null) {
			options = new MediaOptions(availableOptions);
		} else {
			options = options.clone();
			options.setAvailableOptions(availableOptions);
		}
		return options;
	}

	File getFirstFile() {
		return _files[0];
	}

	private Object toInternalFormat(BufferedImage srcImage, ColorMode srcColorMode, Size2i scaleTo) {
		// TODO JavaImageInputにほぼ同じメソッドがある。

		// 一度BufferedImage上でdrawImageして、OpenGLの内部フォーマットに合わせる。

		int dataType = (srcColorMode == ColorMode.RGBA16)
				? DataBuffer.TYPE_USHORT : DataBuffer.TYPE_BYTE;

		WritableRaster wr = Raster.createInterleavedRaster(
				dataType, scaleTo.width, scaleTo.height,
				scaleTo.width*4, 4, new int[] { 2, 1, 0, 3 }, null);
		ComponentColorModel cm = new ComponentColorModel(
				ColorSpace.getInstance(ColorSpace.CS_sRGB), true, true,
				Transparency.TRANSLUCENT, dataType);
		BufferedImage tmpImage = new BufferedImage(cm, wr, true, null);

		Graphics2D g = tmpImage.createGraphics();
		g.drawImage(srcImage, 0, 0, scaleTo.width, scaleTo.height, null);

		DataBuffer dataBuffer = tmpImage.getRaster().getDataBuffer();
		return (dataType == DataBuffer.TYPE_USHORT)
				? ((DataBufferUShort) dataBuffer).getData()
				: ((DataBufferByte) dataBuffer).getData();
	}

}
