/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import org.eclipse.swt.graphics.GC;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.app.project.DisableTimeRemapOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.ProjectOperation;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.TimeCode;

public class TimeRemapElement extends LayerAnimatableValueElement {

	public TimeRemapElement(LayerElement parent) {
		super(parent, parent.layer, "timeRemap", "タイムリマップ");
	}

	@Override
	protected AnimatableValueElementDelegate<?> createDelegate() {
		AnimatableDouble avalue = PropertyUtil.getProperty(layer, property);
		return new TimeRemapElementDelegate(this, name, avalue, (MediaLayer) layer);
	}

	@Override
	protected ProjectOperation createClearKeyframesOperation(ProjectManager pm) {
		return new DisableTimeRemapOperation(pm, (MediaLayer) layer);
	}

}

class TimeRemapElementDelegate extends AnimatableDoubleElementDelegate {

	private final MediaLayer layer;

	private final Time frameDuration;


	public TimeRemapElementDelegate(
			AnimatableValueElement element, String name, AnimatableDouble avalue, MediaLayer layer) {

		super(element, name, avalue);
		this.layer = layer;

		if (LayerNature.isVideoNature(layer) && layer.getMediaInput().getVideoFrameDuration().timeValue > 0) {
			frameDuration = layer.getMediaInput().getVideoFrameDuration();
		} else {
			LayerComposition comp = (LayerComposition) element.viewer.getData(LayerCompositionView.LAYER_COMPOSITION);
			frameDuration = comp.getFrameDuration();
		}
	}

	private String toTimeCode(double sec) {
		Time time = new Time(Math.round(sec*frameDuration.timeScale), frameDuration.timeScale);
		return TimeCode.toTimeCode(time, frameDuration);
	}

	private Time parseTimeCode(String str) {
		long frames = TimeCode.parseTimeCode(str, frameDuration);
		return Time.fromFrameNumber(frames, frameDuration);
	}

	@Override
	protected void drawValue(GC gc, int x, int y, int height, boolean focused) {
		if (LayerNature.isTimeRemapEnabled(layer)) {
			super.drawValue(gc, x, y, height, focused);
		}
	}

	@Override
	protected int drawValue(GC gc, int x, int y, int height,
			boolean focused, Double value, int precision, String unit) {

		x = drawValue(gc, x, y, height, focused, toTimeCode(value));

		if (unit != null) {
			x = drawString(gc, x + 1, y, height, unit);
		}

		return x;
	}

	@Override
	protected Double dragGesture(double dx, double dy) {
		return originalValue + frameDuration.toSecond() * increment * dx;
	}

	@Override
	public Object getCellEditorValue(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			return toTimeCode(value);
		} else {
			return super.getCellEditorValue(columnIndex);
		}
	}

	@Override
	public void setCellEditorValue(int columnIndex, Object value) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			if (!value.equals(toTimeCode(this.value))) {
				modifyValue(parseTimeCode((String) value).toSecond());
			}
		} else {
			super.setCellEditorValue(columnIndex, value);
		}
	}

}
