/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.util.Collections;
import java.util.List;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.widgets.Combo;

import ch.kuramo.javie.app.views.ComboBoxViewerCellEditor;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AnimatableLayerReference;
import ch.kuramo.javie.core.ItemLayer;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Util;

public class AnimatableLayerReferenceElementDelegate extends AnimatableValueElementDelegate<String> {

	private ComboBoxViewerCellEditor editor;


	public AnimatableLayerReferenceElementDelegate(
			AnimatableValueElement element, String name, AnimatableLayerReference avalue) {

		super(element, name, avalue);
	}

	protected void drawValue(GC gc, int x, int y, int height, boolean focused) {
		String value;
		Layer layer = getReferencedLayer();
		if (layer == null) {
			value = "なし";
		} else if (!(layer instanceof ItemLayer<?>) || isLayerNameMode()) {
			value = layer.getName();
		} else {
			value = ((ItemLayer<?>) layer).getItem().getName();
		}
		drawValue(gc, x, y, height, focused, value);
	}

	protected boolean canDragGestureEdit() {
		return false;
	}

	protected String dragGesture(double dx, double dy) {
		throw new UnsupportedOperationException("drag gesture edit is not supported");
	}

	public boolean canEdit(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			return canInlineEdit;
		}
		return false;
	}

	public CellEditor getCellEditor(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			if (editor == null) {
				editor = new ComboBoxViewerCellEditor(element.viewer.getTree(), SWT.READ_ONLY);
				editor.setContenProvider(ArrayContentProvider.getInstance());
				editor.setLabelProvider(new LayerLabelProvider());
				editor.setActivationStyle(ComboBoxViewerCellEditor.DROP_DOWN_ON_PROGRAMMATIC_ACTIVATION);

				((Combo) editor.getControl()).addSelectionListener(new SelectionAdapter() {
					public void widgetSelected(SelectionEvent e) {
						setCellEditorValue(LayerCompositionView.VALUE_COL, editor.getValue());
					}
				});
			}
			editor.setInput(getLayers());
			return editor;
		}
		return null;
	}

	public Object getCellEditorValue(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			Layer layer = getReferencedLayer();
			return (layer != null) ? layer : NO_LAYER;
		}
		return null;
	}

	public void setCellEditorValue(int columnIndex, Object value) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			modifyValue((value != NO_LAYER) ? ((Layer) value).getId() : "");
		}
	}

	private LayerComposition getLayerComposition() {
		return (LayerComposition) element.viewer.getData(LayerCompositionView.LAYER_COMPOSITION);
	}

	private Layer getReferencedLayer() {
		return getLayerComposition().getLayer(value);
	}

	private boolean isLayerNameMode() {
		return Boolean.TRUE.equals(element.viewer.getData(LayerCompositionView.LAYER_NAME_MODE));
	}

	private Object[] getLayers() {
		List<Object> layers = Util.newList();

		for (Layer l :  getLayerComposition().getLayers()) {
			layers.add(l);
		}

		layers.add(NO_LAYER);
		Collections.reverse(layers);

		return layers.toArray();
	}

	private static final Object NO_LAYER = new Object();

	private class LayerLabelProvider extends LabelProvider {
		public String getText(Object element) {
			if (element == NO_LAYER) {
				return "なし";
			} else if (!(element instanceof ItemLayer<?>) || isLayerNameMode()) {
				return ((Layer) element).getName();
			} else {
				return ((ItemLayer<?>) element).getItem().getName();
			}
		}
	}

}
