/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.List;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.AnimatableVec3d;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.TransformableLayer;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class ThreeDSwitchOperation extends ProjectOperation {

	private final String _compId;

	private final String _layerId;

	private final boolean _oldValue;

	private final List<String> _data;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public ThreeDSwitchOperation(ProjectManager projectManager, TransformableLayer layer) {
		super(projectManager, "3Dスイッチの変更");
		InjectorHolder.getInjector().injectMembers(this);

		LayerComposition comp = projectManager.checkLayer(layer);
		_compId = comp.getId();
		_layerId = layer.getId();
		_oldValue = LayerNature.isThreeD(layer);
		_data = save(layer);
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		TransformableLayer layer = getLayer(project);

		if (_oldValue) {
			clear3D(layer);
		} else {
			to3D(layer);
		}

		LayerNature.setThreeD(layer, !_oldValue);

		if (pm != null) {
			pm.fireLayerPropertyChange(layer, "threeD");
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		TransformableLayer layer = getLayer(project);
		restore(layer);

		LayerNature.setThreeD(layer, _oldValue);

		if (pm != null) {
			pm.fireLayerPropertyChange(layer, "threeD");
		}

		return Status.OK_STATUS;
	}

	private TransformableLayer getLayer(Project project) {
		LayerComposition comp = project.getComposition(_compId);
		return (TransformableLayer) comp.getLayer(_layerId);
	}

	private List<String> save(TransformableLayer layer) {
		List<String> data = Util.newList();

		data.add(_encoder.encodeElement(layer.getAnchorPoint()));
		data.add(_encoder.encodeElement(layer.getScale()));
		data.add(_encoder.encodeElement(layer.getOrientation()));
		data.add(_encoder.encodeElement(layer.getRotationX()));
		data.add(_encoder.encodeElement(layer.getRotationY()));
		data.add(_encoder.encodeElement(layer.getPosition()));

		if (layer instanceof MediaLayer) {
			MediaLayer ml = (MediaLayer) layer;
			data.add(_encoder.encodeElement(ml.getDepthBase()));
			data.add(_encoder.encodeElement(ml.getIntersectionGroup()));
		}

		return data;
	}

	private void to3D(TransformableLayer layer) {
		// 3Dレイヤー用の各値は適切にクリアされているはずなので、
		// depthBaseをレイヤーの中心点にするだけでよいが、
		// 念のため他の値も全て初期化する。
		clear3D(layer);

		// スケールのzの値は、キーフレームがなく x==y の場合のみ z も同じ値にする。
		AnimatableVec3d scale = layer.getScale();
		if (!scale.hasKeyframe()) {
			Vec3d value = scale.getStaticValue();
			if (value.x == value.y) {
				setZ(scale, value.x);
			}
		}

		if (layer instanceof MediaLayer) {
			MediaLayer mediaLayer = (MediaLayer) layer;
			VideoBounds bounds = mediaLayer.getMediaInput().getVideoFrameBounds();
			if (bounds != null) {
				mediaLayer.getDepthBase().reset(
						new Vec2d(bounds.width/2d + bounds.x, bounds.height/2d + bounds.y));
			}
		}
	}

	private void setZ(AnimatableVec3d avalue, double z) {
		List<Keyframe<Vec3d>> newKeyframes = Util.newList();

		for (Keyframe<Vec3d> kf : avalue.getKeyframeMap().values()) {
			Vec3d v = kf.value;
			newKeyframes.add(new Keyframe<Vec3d>(kf.time, new Vec3d(v.x, v.y, z), kf.interpolation));
		}

		Vec3d sv = avalue.getStaticValue();
		avalue.clearKeyframes(new Vec3d(sv.x, sv.y, z));

		for (Keyframe<Vec3d> kf : newKeyframes) {
			avalue.putKeyframe(kf);
		}
	}

	private void clear3D(TransformableLayer layer) {
		setZ(layer.getAnchorPoint(), 0);
		setZ(layer.getScale(), 100);
		clear(layer.getOrientation(), new Vec3d(0, 0, 0));
		clear(layer.getRotationX(), 0d);
		clear(layer.getRotationY(), 0d);
		setZ(layer.getPosition(), 0);

		if (layer instanceof MediaLayer) {
			MediaLayer ml = (MediaLayer) layer;
			clear(ml.getDepthBase(), new Vec2d(0, 0));
			clear(ml.getIntersectionGroup(), "");
		}
	}

	private <V> void clear(AnimatableValue<V> avalue, V staticValue) {
		avalue.reset(staticValue);
	}

	private void restore(TransformableLayer layer) throws ExecutionException {
		restore(_data.get(0), layer.getAnchorPoint());
		restore(_data.get(1), layer.getScale());
		restore(_data.get(2), layer.getOrientation());
		restore(_data.get(3), layer.getRotationX());
		restore(_data.get(4), layer.getRotationY());
		restore(_data.get(5), layer.getPosition());

		if (layer instanceof MediaLayer) {
			MediaLayer ml = (MediaLayer) layer;
			restore(_data.get(6), ml.getDepthBase());
			restore(_data.get(7), ml.getIntersectionGroup());
		}
	}

	private <V, A extends AnimatableValue<V>> void restore(String data, A dst) throws ExecutionException {
		try {
			@SuppressWarnings("unchecked")
			A src = _decoder.decodeElement(data, (Class<A>) dst.getClass());

			dst.reset(src.getStaticValue());
			dst.setExpression(src.getExpression());

			for (Keyframe<V> kf : src.getKeyframeMap().values()) {
				dst.putKeyframe(kf);
			}

		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding AnimatableValue data", e);
		}
	}

}
