/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.transition;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.effects.Texture1D;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.LinearWipe", category=Categories.TRANSITION)
public class LinearWipe {

	@Property(min="0", max="100")
	private IAnimatableDouble transitionCompletion;

	@Property("90")
	private IAnimatableDouble wipeAngle;

	@Property(min="0", max="32000")
	private IAnimatableDouble feather;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IArrayPools arrayPools;

	@Inject
	public LinearWipe(IVideoEffectContext context, IVideoRenderSupport support, IArrayPools arrayPools) {
		this.context = context;
		this.support = support;
		this.arrayPools = arrayPools;
	}

	public IVideoBuffer doVideoEffect() {
		double completion = context.value(transitionCompletion)/100;
		if (completion == 1) {
			IVideoBuffer buffer = context.createVideoBuffer(context.getPreviousBounds());
			buffer.clear();
			return buffer;
		}

		IVideoBuffer input = context.doPreviousEffect();

		if (completion == 0) {
			return input;
		}

		VideoBounds bounds = input.getBounds();
		if (bounds.isEmpty()) {
			return input;
		}


		double wipeAngle = context.value(this.wipeAngle) % 360;
		if (wipeAngle < 0) wipeAngle += 360;

		double feather = context.value(this.feather);
		feather = context.getVideoResolution().scale(feather);

		double x = bounds.x;
		double y = bounds.y;
		int w = bounds.width;
		int h = bounds.height;

		double theta = Math.toRadians(wipeAngle);
		double wipeLen = h*Math.abs(Math.cos(theta))+w*Math.abs(Math.sin(theta));
		double wipeWidth = w*Math.abs(Math.cos(theta))+h*Math.abs(Math.sin(theta));


		int wipeTex = 0;
		try {
			final float[] texCoords = new float[2];
			final int tex = wipeTex = createWipeTexture(wipeLen, completion, feather, texCoords);

			final VideoBounds b = bounds;
			final float tx = (float)(x + w/2.0), ty = (float)(y + h/2.0);
			final float rz = (float)wipeAngle;
			final float sx = (float)(wipeWidth/w), sy = (float)(wipeLen/h);
			final float vx0 = (float)x, vy0 = (float)y, vx1 = (float)(x+w), vy1 = (float)(y+h);

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();

					support.ortho2D(b);
					gl.glMatrixMode(GL2.GL_MODELVIEW);
					gl.glTranslatef(tx, ty, 0);
					gl.glRotatef(rz, 0, 0, 1);
					gl.glScalef(sx, sy, 1);
					gl.glTranslatef(-tx, -ty, 0);

					gl.glEnable(GL2.GL_BLEND);
					gl.glBlendFuncSeparate(GL2.GL_ZERO, GL2.GL_SRC_ALPHA, GL2.GL_ZERO, GL2.GL_SRC_ALPHA);

					gl.glActiveTexture(GL2.GL_TEXTURE0);
					gl.glBindTexture(GL2.GL_TEXTURE_1D, tex);
					gl.glEnable(GL2.GL_TEXTURE_1D);

					gl.glBegin(GL2.GL_QUADS);
					gl.glTexCoord1f(texCoords[0]);
					gl.glVertex2f(vx0, vy0);
					gl.glTexCoord1f(texCoords[0]);
					gl.glVertex2f(vx1, vy0);
					gl.glTexCoord1f(texCoords[1]);
					gl.glVertex2f(vx1, vy1);
					gl.glTexCoord1f(texCoords[1]);
					gl.glVertex2f(vx0, vy1);
					gl.glEnd();
				}
			};

			int pushAttribs = GL2.GL_COLOR_BUFFER_BIT | GL2.GL_TEXTURE_BIT | GL2.GL_ENABLE_BIT /*| GL2.GL_TRANSFORM_BIT*/;
			support.useFramebuffer(operation, pushAttribs, input);

			IVideoBuffer result = input;
			input = null;
			return result;

		} finally {
			if (input != null) input.dispose();
			if (wipeTex != 0) deleteTexture(wipeTex);
		}
	}

	private int createWipeTexture(double wipeLen, double completion, double feather, float[] texCoords) {
		int dataLen = (int)Math.ceil(feather)*2 + 2;
		IArray<float[]> data = arrayPools.getFloatArray(dataLen);
		try {
			float[] array = data.getArray();

			if (feather > 0) {
				double sigma = feather / 2.5;
				double twoSigmaSq = 2 * sigma * sigma;
				double k1 = 1.27323954473516276487;
				double k2 = 0.14001228868666648619;

				for (int i = 1; i < dataLen-1; ++i) {
					double d = i-dataLen/2+0.5;

					// Wikipedia の「誤差関数」のページにある「初等関数による近似」の式を用いて計算している。
					// http://ja.wikipedia.org/wiki/%E8%AA%A4%E5%B7%AE%E9%96%A2%E6%95%B0#.E5.88.9D.E7.AD.89.E9.96.A2.E6.95.B0.E3.81.AB.E3.82.88.E3.82.8B.E8.BF.91.E4.BC.BC
					double t2 = d*d/twoSigmaSq;
					double erf = Math.signum(d)*Math.sqrt(1-Math.exp(-t2*(k1+k2*t2)/(1+k2*t2)));

					array[i] = (float)(0.5*(1+erf));
				}
			}

			array[0] = 0;
			array[dataLen-1] = 1;

			double a = 0.5*(wipeLen-dataLen)/dataLen;
			double b = (dataLen*0.5-feather)/dataLen;
			texCoords[0] = (float)((2+2*a-b)*(1-completion)+b*completion);
			texCoords[1] = (float)((1-b)*(1-completion)-(1+2*a-b)*completion);

			return Texture1D.fromArray(data, context);

		} finally {
			data.release();
		}
	}

	private void deleteTexture(int texture) {
		GL2 gl = context.getGL().getGL2();
		gl.glDeleteTextures(1, new int[] { texture }, 0);
	}

}
