/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.stylize;

import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IConvolutionSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Mosaic", category=Categories.STYLIZE)
public class Mosaic {

	@ShaderSource
	public static final String[] MOSAIC_SHARP_COLORS = {
		"uniform sampler2DRect texture;",
		"uniform float hSize;",
		"uniform float vSize;",
		"",
		"void main(void)",
		"{",
		"	float s = (floor(gl_TexCoord[0].s/hSize)+0.5)*hSize;",
		"	float t = (floor(gl_TexCoord[0].t/vSize)+0.5)*vSize;",
		"	gl_FragColor = texture2DRect(texture, vec2(s, t));",
		"}"
	};


	@Property(value="10", min="1")
	private IAnimatableInteger horizontalBlocks;

	@Property(value="10", min="1")
	private IAnimatableInteger verticalBlocks;

	@Property
	private IAnimatableBoolean sharpColors;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IConvolutionSupport convolution;

	private final IShaderProgram sharpColorsProgram;

	@Inject
	public Mosaic(
			IVideoEffectContext context, IVideoRenderSupport support,
			IConvolutionSupport convolution, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.convolution = convolution;
		sharpColorsProgram = shaders.getProgram(Mosaic.class, "MOSAIC_SHARP_COLORS");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer input = context.doPreviousEffect();

		VideoBounds bounds = input.getBounds();
		if (bounds.isEmpty()) {
			return input;
		}

		int hBlocks = context.value(horizontalBlocks);
		int vBlocks = context.value(verticalBlocks);

		int w = bounds.width;
		int h = bounds.height;

		hBlocks = Math.min(hBlocks, w);
		vBlocks = Math.min(vBlocks, h);

		if (hBlocks == w && vBlocks == h) {
			return input;
		}

		boolean sharpColors = context.value(this.sharpColors);

		IVideoBuffer output = null;
		try {
			if (sharpColors) {
				output = doSharpColorsMosaic(input, hBlocks, vBlocks);
			} else {
				output = doAverageColorsMosaic(input, hBlocks, vBlocks);
			}
			return output;

		} finally {
			if (input != output) input.dispose();
		}
	}

	private IVideoBuffer doSharpColorsMosaic(IVideoBuffer input, int hBlocks, int vBlocks) {
		VideoBounds bounds = input.getBounds();
		double hSize = (double)bounds.width/hBlocks;
		double vSize = (double)bounds.height/vBlocks;

		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		uniforms.add(new GLUniformData("texture", 0));
		uniforms.add(new GLUniformData("hSize", (float)hSize));
		uniforms.add(new GLUniformData("vSize", (float)vSize));

		return support.useShaderProgram(sharpColorsProgram, uniforms, null, input);
	}

	private IVideoBuffer doAverageColorsMosaic(IVideoBuffer input, int hBlocks, int vBlocks) {
		// average メソッド内で input の WrapMode が変更されるので、終わったら元に戻す。
		TextureWrapMode wrapMode = input.getTextureWrapMode();
		IVideoBuffer buffer = null;
		try {
			buffer = context.createVideoBuffer(new VideoBounds(hBlocks, vBlocks));
			average(input, buffer);
			scale(buffer, input);
			return input;
		} finally {
			if (buffer != null) buffer.dispose();
			input.setTextureWrapMode(wrapMode);
		}
	}

	private void average(IVideoBuffer src, IVideoBuffer dst) {
		IVideoBuffer tmpBuf = null;
		try {
			VideoBounds srcBounds = src.getBounds();
			VideoBounds dstBounds = dst.getBounds();
			int hBlocks = dstBounds.width;
			int vBlocks = dstBounds.height;

			// こっちだと hBlocks, vBlocks が多いときにぼやけた感じになってしまう。
			//int hSize = (srcBounds.width + hBlocks - 1) / hBlocks;
			//int vSize = (srcBounds.height + vBlocks - 1) / vBlocks;

			// こっちだと、割り切れるとき以外はブロックの端の1ピクセルがサンプリングされない。
			// しかし、見た目ではわからないのでこちらを採用。
			int hSize = srcBounds.width / hBlocks;
			int vSize = srcBounds.height / vBlocks;

			// ブロックサイズが7より大きい場合は複数回にわけて処理する
			// (大きなカーネルで一度に畳み込むと負荷が大きい。また、IConvolutionSupportの実装はカーネルサイズの上限が101(縦横掛けた値で))
			if (hSize > 7 || vSize > 7) {
				tmpBuf = context.createVideoBuffer(new VideoBounds(
						(hSize > 7) ? hBlocks*7 : hBlocks, (vSize > 7) ? vBlocks*7 : vBlocks));
				average(src, tmpBuf);
				src = tmpBuf;

				srcBounds = src.getBounds();
				hSize = srcBounds.width / hBlocks;
				vSize = srcBounds.height / vBlocks;
			}

			int ksize = hSize * vSize;
			float[] kernel = new float[ksize];
			float[] offset = new float[ksize*2];

			for (int j = 0; j < vSize; ++j) {
				for (int i = 0; i < hSize; ++i) {
					int k = j*hSize+i;
					kernel[k] = 1f/ksize;
					offset[k*2  ] = i - hSize/2;
					offset[k*2+1] = j - vSize/2;
				}
			}

			int sw = srcBounds.width;
			int sh = srcBounds.height;
			final VideoBounds bounds = dstBounds;
			final double[][] texCoords = new double[][] { {0, 0}, {sw, 0}, {sw, sh}, {0, sh} };

			Runnable operation = new Runnable() {
				public void run() {
					support.ortho2D(bounds);
					support.quad2D(bounds, texCoords);
				}
			};

			src.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);
			convolution.convolve(src, dst, kernel, offset, operation, 0);

		} finally {
			if (tmpBuf != null) tmpBuf.dispose();
		}
	}

	private void scale(IVideoBuffer src, IVideoBuffer dst) {
		final VideoBounds dstBounds = dst.getBounds();

		VideoBounds srcBounds = src.getBounds();
		int sw = srcBounds.width;
		int sh = srcBounds.height;
		final double[][] texCoords = new double[][] { {0, 0}, {sw, 0}, {sw, sh}, {0, sh} };

		Runnable operation = new Runnable() {
			public void run() {
				support.ortho2D(dstBounds);
				support.quad2D(dstBounds, texCoords);
			}
		};

		support.useFramebuffer(operation, 0, dst, src);
	}

}
