/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.perspective;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IBlurSupport.BlurDimensions;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.DropShadow", category=Categories.PERSPECTIVE)
public class DropShadow {

	@Property("0,0,0")
	private IAnimatableColor shadowColor;

	@Property(value="50", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property("135")
	private IAnimatableDouble direction;

	@Property(value="5", min="0", max="4000")
	private IAnimatableDouble distance;

	@Property(value="0", min="0", max="500")
	private IAnimatableDouble softness;

	@Property("true")
	private IAnimatableBoolean fast;

	@Property
	private IAnimatableBoolean shadowOnly;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlurSupport blurSupport;

	private final IShaderProgram normalProgram;

	private final IShaderProgram shadowOnlyProgram;

	@Inject
	public DropShadow(
			IVideoEffectContext context, IVideoRenderSupport support,
			IBlurSupport blurSupport, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blurSupport = blurSupport;
		this.normalProgram = shaders.getProgram(DropShadow.class, "NORMAL");
		this.shadowOnlyProgram = shaders.getProgram(DropShadow.class, "SHADOW_ONLY");
	}

	public VideoBounds getVideoBounds() {
		VideoBounds bounds = context.getPreviousBounds();
		if (bounds.isEmpty()) {
			return bounds;
		}

		double direction = context.value(this.direction);
		double distance = context.value(this.distance);
		double softness = context.value(this.softness);
		boolean fast = (softness > 50) || context.value(this.fast);

		Resolution resolution = context.getVideoResolution();
		distance = resolution.scale(distance);
		softness = resolution.scale(softness);

		double radians = Math.toRadians(direction);
		double distanceX =  Math.sin(radians) * distance;
		double distanceY = -Math.cos(radians) * distance;

		VideoBounds shadowBounds;

		if (softness > 0) {
			shadowBounds = blurSupport.calcGaussianBlurredBounds(
					bounds, softness, BlurDimensions.BOTH, fast);
		} else {
			shadowBounds = bounds;
		}

		shadowBounds = new VideoBounds(
				shadowBounds.x+distanceX, shadowBounds.y+distanceY,
				shadowBounds.width, shadowBounds.height);

		double x1 = bounds.x + Math.floor(Math.min(bounds.x, shadowBounds.x) - bounds.x);
		double y1 = bounds.y + Math.floor(Math.min(bounds.y, shadowBounds.y) - bounds.y);
		double x2 = Math.max(bounds.x+bounds.width, shadowBounds.x+shadowBounds.width);
		double y2 = Math.max(bounds.y+bounds.height, shadowBounds.y+shadowBounds.height);

		return new VideoBounds(x1, y1, (int)Math.ceil(x2-x1), (int)Math.ceil(y2-y1));
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();

		VideoBounds bounds = source.getBounds();
		if (bounds.isEmpty()) {
			return source;
		}

		Color shadowColor = context.value(this.shadowColor);
		double opacity = context.value(this.opacity) / 100;
		double direction = context.value(this.direction);
		double distance = context.value(this.distance);
		double softness = context.value(this.softness);
		boolean fast = (softness > 50) || context.value(this.fast);
		boolean shadowOnly = context.value(this.shadowOnly);

		Resolution resolution = context.getVideoResolution();
		distance = resolution.scale(distance);
		softness = resolution.scale(softness);

		double radians = Math.toRadians(direction);
		double distanceX =  Math.sin(radians) * distance;
		double distanceY = -Math.cos(radians) * distance;

		IVideoBuffer shadow = null;
		IVideoBuffer buffer = null;
		try {
			if (softness > 0) {
				shadow = blurSupport.gaussianBlur(source, softness, BlurDimensions.BOTH, false, fast);
				shadow.setTextureFilter(TextureFilter.LINEAR);
				//source.setTextureFilter(TextureFilter.LINEAR);
			} else {
				shadow = source;
				shadow.setTextureFilter(TextureFilter.LINEAR);
			}

			VideoBounds shadowBounds = shadow.getBounds();

			shadowBounds = new VideoBounds(
					shadowBounds.x+distanceX, shadowBounds.y+distanceY,
					shadowBounds.width, shadowBounds.height);

			double x1 = bounds.x + Math.floor(Math.min(bounds.x, shadowBounds.x) - bounds.x);
			double y1 = bounds.y + Math.floor(Math.min(bounds.y, shadowBounds.y) - bounds.y);
			double x2 = Math.max(bounds.x+bounds.width, shadowBounds.x+shadowBounds.width);
			double y2 = Math.max(bounds.y+bounds.height, shadowBounds.y+shadowBounds.height);

			buffer = context.createVideoBuffer(new VideoBounds(x1, y1, (int)Math.ceil(x2-x1), (int)Math.ceil(y2-y1)));

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("shadow", 0));
			uniforms.add(new GLUniformData("shadowColor", 4,
					toFloatBuffer(shadowColor.r*opacity, shadowColor.g*opacity, shadowColor.b*opacity, opacity)));
			uniforms.add(new GLUniformData("shadowOffset", 2,
					toFloatBuffer(-distanceX, -distanceY)));

			if (shadowOnly) {
				support.useShaderProgram(shadowOnlyProgram, uniforms, buffer, shadow);
			} else {
				uniforms.add(new GLUniformData("source", 1));
				support.useShaderProgram(normalProgram, uniforms, buffer, shadow, source);
			}

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
			if (shadow != null && shadow != source) shadow.dispose();
			source.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double... values) {
		float[] array = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			array[i] = (float)values[i];
		}
		return FloatBuffer.wrap(array);
	}

	@ShaderSource
	public static final String[] NORMAL = {
		"uniform sampler2DRect shadow;",
		"uniform sampler2DRect source;",
		"uniform vec4 shadowColor;",
		"uniform vec2 shadowOffset;",
		"",
		"void main(void)",
		"{",
		"	float shadowAlpha = texture2DRect(shadow, gl_TexCoord[0].st + shadowOffset).a;",
		"	vec4 sourceColor = texture2DRect(source, gl_TexCoord[1].st);",
		"	gl_FragColor = sourceColor + shadowColor*shadowAlpha*(1.0-sourceColor.a);",
		"}"
	};

	@ShaderSource
	public static final String[] SHADOW_ONLY = {
		"uniform sampler2DRect shadow;",
		"uniform vec4 shadowColor;",
		"uniform vec2 shadowOffset;",
		"",
		"void main(void)",
		"{",
		"	float shadowAlpha = texture2DRect(shadow, gl_TexCoord[0].st + shadowOffset).a;",
		"	gl_FragColor = shadowColor*shadowAlpha;",
		"}"
	};

}
