/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.List;

import ch.kuramo.javie.api.ICamera;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VideoEffect;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoEffectPipeline;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

public class VideoEffectPipelineImpl implements VideoEffectPipeline {

	@Inject
	private RenderContext _context;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private AudioRenderSupport _arSupport;


	public VideoBounds getVideoBounds(List<Effect> effects,
			WrappedOperation<VideoBounds> inputBoundsOperation) {

		VideoEffectContextImpl impl = new VideoEffectContextImpl(
				enabledVideoEffectsOnly(effects), inputBoundsOperation, null);
		VideoEffectContextProxy.push(impl);
		try {
			return impl.getPreviousBounds();
		} finally {
			VideoEffectContextProxy.pop();
		}
	}

	public IVideoBuffer doVideoEffects(List<Effect> effects,
			WrappedOperation<VideoBounds> inputBoundsOperation,
			WrappedOperation<IVideoBuffer> inputBufferOperation) {

		VideoEffectContextImpl impl = new VideoEffectContextImpl(
				enabledVideoEffectsOnly(effects), inputBoundsOperation, inputBufferOperation);
		VideoEffectContextProxy.push(impl);
		try {
			return impl.doPreviousEffect();
		} finally {
			VideoEffectContextProxy.pop();
		}
	}

	private List<VideoEffect> enabledVideoEffectsOnly(List<Effect> effects) {
		List<VideoEffect> videoEffects = Util.newList();
		for (Effect e : effects) {
			if (e instanceof VideoEffect && e.isEnabled()) {
				videoEffects.add((VideoEffect) e);
			}
		}
		return videoEffects;
	}


	private class VideoEffectContextImpl
			extends AbstractEffectContext implements IVideoEffectContext {

		private List<VideoEffect> effects;

		private final WrappedOperation<VideoBounds> inputBoundsOperation;

		private final WrappedOperation<IVideoBuffer> inputBufferOperation;

		private VideoEffect currentEffect;

		private boolean gettingBounds;


		private VideoEffectContextImpl(
				List<VideoEffect> effects,
				WrappedOperation<VideoBounds> inputBoundsOperation,
				WrappedOperation<IVideoBuffer> inputBufferOperation) {

			super(_context, _vrSupport, _arSupport);

			this.effects = effects;
			this.inputBoundsOperation = inputBoundsOperation;
			this.inputBufferOperation = inputBufferOperation;
		}

		public VideoBounds getPreviousBounds() {
			if (inputBoundsOperation == null) {
				throw new IllegalStateException();
			}

			boolean savedGettingBounds = gettingBounds;
			try {
				gettingBounds = true;

				if (effects.isEmpty()) {
					return _context.saveAndExecute(inputBoundsOperation);
				}

				return executePrevious(new WrappedOperation<VideoBounds>() {
					public VideoBounds execute() {
						VideoBounds bounds = currentEffect.getVideoBounds();
						return (bounds != null) ? bounds : getPreviousBounds();
					}
				});
			} finally {
				gettingBounds = savedGettingBounds;
			}
		}
		
		public IVideoBuffer doPreviousEffect() {
			if (inputBufferOperation == null || gettingBounds) {
				throw new IllegalStateException();
			}

			if (effects.isEmpty()) {
				return _context.saveAndExecute(inputBufferOperation);
			}

			return executePrevious(new WrappedOperation<IVideoBuffer>() {
				public IVideoBuffer execute() {
					IVideoBuffer buffer = currentEffect.doVideoEffect();
					if (buffer == null) {
						return doPreviousEffect();
					}
					// TODO エフェクトから返されたバッファのColorModeがコンテキストのColorModeと異なる場合、
					//      コンテキストのColorModeのバッファにコピーして返す。また、警告をログに出す。
					//      ただし、このコメントを書いた時点ではコンテキストのColorModeと異なるバッファを
					//      エフェクト内で作成することはできない。
					//if (buffer.getColorMode() != _context.getColorMode()) {
					//	IVideoBuffer copy = _vrSupport.createVideoBuffer(_context.getColorMode(), buffer.getBounds());
					//	_vrSupport.copy(buffer, copy);
					//	buffer.dispose();
					//	return copy;
					//}
					if (buffer.isTexture()) {
						// TODO テクスチャの現在の補間モードとラッピングモードが NEAREST / CLAMP_TO_BORDER 以外の場合、警告をログに出す。
						buffer.setTextureFilter(TextureFilter.NEAREST);
						buffer.setTextureWrapMode(TextureWrapMode.CLAMP_TO_BORDER);
					}
					return buffer;
				}
			});
		}

		private <T> T executePrevious(final WrappedOperation<T> wop) {
			VideoEffect savedCurrent = currentEffect;
			List<VideoEffect> savedEffects = effects;
			try {

				int lastIndex = effects.size() - 1;
				currentEffect = effects.get(lastIndex);
				effects = effects.subList(0, lastIndex);

				return _context.saveAndExecute(new WrappedOperation<T>() {
					public T execute() {
						return _vrSupport.pushMatrixAndExecute(wop);
					}
				});

			} finally {
				effects = savedEffects;
				currentEffect = savedCurrent;
			}
		}

		public ICamera getCamera() {
			return _context.getCamera();
		}

		public String getEffectName() {
			if (currentEffect == null) {
				throw new IllegalStateException("no current Effect");
			}
			return currentEffect.getName();
		}

	}

}
