/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import static ftgl.FTGL.FT_Err_Invalid_Argument;
import static ftgl.FTGL.FT_Err_Ok;

import java.io.File;
import java.io.FileFilter;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.Font;
import ch.kuramo.javie.core.services.FontList;

import com.sun.jna.Library;
import com.sun.jna.Native;
import com.sun.jna.Pointer;

import ftgl.FTGL;
import ftgl.FT_Face;
import ftgl.FT_Library;
import ftgl.FT_SfntName;

public class FontListImpl implements FontList {

	private static final Logger _logger = LoggerFactory.getLogger(FontListImpl.class);


	private static final List<String> FONT_FILE_EXTENSIONS = Arrays.asList("ttf", "ttc", "dfont", "otf");

	private static final FileFilter FONT_FILE_FILTER = new FileFilter() {
		public boolean accept(File pathname) {
			String name = pathname.getName();
			int lastDot = name.indexOf('.');
			if (lastDot != -1) {
				String extension = name.substring(lastDot+1).toLowerCase();
				return FONT_FILE_EXTENSIONS.contains(extension);
			}
			return false;
		}
	};

	private static final String OS = System.getProperty("os.name").toLowerCase().replaceAll("\\s+", "");
	private static final boolean MACOSX = OS.contains("macosx");
	private static final boolean WINDOWS = OS.contains("windows");
	private static final Locale LOCALE = Locale.getDefault();

	private static final short PLATFORM_UNICODE = 0;
	private static final short PLATFORM_MACINTOSH = 1;
	private static final short PLATFORM_WINDOWS = 3;

	private static final short LANG_MAC_en = 0;
	private static final short LANG_MAC_ja = 11;
	private static final short LANG_WIN_en_US = 0x0409;
	private static final short LANG_WIN_ja_JP = 0x0411;

	private static final short FULL_FONT_NAME = 4;
	private static final short POSTSCRIPT_NAME = 6;


	private static final String[] MACOSX_DEFAULT_FONTS = {
		"HiraKakuProN-W3", "HiraKakuPro-W3", "HiraKakuProN-W6", "HiraKakuPro-W6",
		"Osaka", "Osaka-Mono", "Helvetica", "Monaco"
	};

	private static final String[] WINDOWS_DEFAULT_FONTS = {
		"Meiryo", "Meiryo-Bold", "MS-PGothic", "MS-Gothic", "MicrosoftSansSerif"
	};


	private final Map<String, Font> _fontMap = Util.newMap();

	private List<Font> _fontList;

	private Font _defaultFont;


	public FontListImpl() {
		initialize();
	}

	private List<File> listFontDirectories() {
		String[] dirs;
		if (MACOSX) {
			dirs = new String[] {
					System.getProperty("user.home") + "/Library/Fonts",
					"/Library/Fonts",
					"/Network/Library/Fonts",
					"/System/Library/Fonts"
			};
		} else if (WINDOWS) {
			char[] path = new char[WinShell32.MAX_PATH];
			int hr = WinShell32.instance.SHGetFolderPathW(null, WinShell32.CSIDL_FONTS, null, 0, path);
			if (hr == 0) {
				int i = 0;
				for (i = 0; i < path.length && path[i] != 0; ++i);	// This is OK.
				dirs = new String[] { new String(path, 0, i) };
			} else {
				_logger.warn("no font folder found.");
				return Collections.emptyList();
			}
		} else {
			_logger.warn("font is not supported on this platform.");
			return Collections.emptyList();
		}

		List<File> list = Util.newList();
		for (String dir : dirs) {
			File dirFile = new File(dir);
			if (dirFile.isDirectory()) {
				list.add(dirFile);
			}
		}
		return list;
	}

	private List<File> listFontFiles() {
		List<File> list = Util.newList();
		for (File dir : listFontDirectories()) {
			list.addAll(Arrays.asList(dir.listFiles(FONT_FILE_FILTER)));
		}
		return list;
	}

	private void initialize() {
		FT_Library[] ftlib = new FT_Library[1];
		long error = FTGL.ft_Init_FreeType(ftlib);
		if (error != FT_Err_Ok) {
			_logger.error("FT_Init_FreeType: error=" + error);
			return;
		}
		try {
			for (File fontFile : listFontFiles()) {
				examineFontFile(ftlib[0], fontFile);
			}
		} finally {
			error = FTGL.ft_Done_FreeType(ftlib[0]);
			if (error != FT_Err_Ok) {
				_logger.error("FT_Done_FreeType: error=" + error);
			}
		}

		List<Font> list = Util.newList();
		list.addAll(_fontMap.values());
		Collections.sort(list, new Comparator<Font>() {
			public int compare(Font o1, Font o2) {
				return o1.fullName.compareTo(o2.fullName);
			}
		});

		_fontList = Collections.unmodifiableList(list);


		String[] defaultFonts = MACOSX ? MACOSX_DEFAULT_FONTS :
								WINDOWS ? WINDOWS_DEFAULT_FONTS : new String[0];
		for (String psName : defaultFonts) {
			if ((_defaultFont = _fontMap.get(psName)) != null) break;
		}
		if (_defaultFont == null && _fontList.size() > 0) {
			_defaultFont = _fontList.get(0);
		}


		//for (Font f : _fontList) {
		//	System.out.println(f.fullName);
		//	System.out.printf("  %s,%s,%d%n", f.psName, f.fontFile.getAbsolutePath(), f.faceIndex);
		//}
	}

	private void examineFontFile(FT_Library ftlib, File fontFile) {
		FT_Face[] face = new FT_Face[1];

		for (int i = 0; ; ++i) {
			int error = FTGL.ft_New_Face(ftlib, fontFile.getAbsolutePath(), i, face);
			if (error != FT_Err_Ok) {
				if (error != FT_Err_Invalid_Argument) {
					_logger.error(String.format(
							"FT_New_Face: error=%d (fontFile=%s, faceIndex=%d)",
							error, fontFile.getAbsolutePath(), i));
				}
				return;
			}
			try {
				if (!examineFontFace(fontFile, i, face[0])) {
					return;
				}
			} finally {
				error = FTGL.ft_Done_Face(face[0]);
				if (error != FT_Err_Ok) {
					_logger.error(String.format(
							"FT_Done_Face: error=%d (fontFile=%s, faceIndex=%d)",
							error, fontFile.getAbsolutePath(), i));
				}
			}
		}
	}

	private boolean examineFontFace(File fontFile, int faceIndex, FT_Face face) {
		SfntName psName = null;
		SfntName fullName = null;

		FT_SfntName[] sfntName = new FT_SfntName[1];
		for (int j = 0, n = FTGL.ft_Get_Sfnt_Name_Count(face); j < n; ++j) {
			int error = FTGL.ft_Get_Sfnt_Name(face, j, sfntName);
			if (error != FT_Err_Ok) {
				_logger.error(String.format(
						"FT_Get_Sfnt_Name: error=%d (fontFile=%s, faceIndex=%d)",
						error, fontFile.getAbsolutePath(), faceIndex));
				break;
			}

			SfntName name = new SfntName(sfntName[0]);
			switch (name.name_id) {
				case POSTSCRIPT_NAME:
					psName = preferredName(psName, name);
					break;

				case FULL_FONT_NAME:
					fullName = preferredName(fullName, name);
					break;
			}
		}

		if (psName == null) {
			_logger.debug("no postscript name found: " + fontFile.getAbsolutePath());
			return false;
		} else if (_fontMap.containsKey(psName.string)) {
			_logger.debug("postscript name already exists: " + psName.string);
			return false;
		}

		if (fullName == null) {
			_logger.debug("no full name found, postscript name is used as full name: " + psName.string);
			fullName = psName;
		}
		_fontMap.put(psName.string, new Font(psName.string, fullName.string, fontFile, faceIndex));

		return true;
	}

	private SfntName preferredName(SfntName name1, SfntName name2) {
		if (name1 == null && name2 == null) throw new IllegalArgumentException();
		if (name1 == null) return name2;
		if (name2 == null) return name1;

		short preferedPlatId = MACOSX ? PLATFORM_MACINTOSH : WINDOWS ? PLATFORM_WINDOWS : PLATFORM_UNICODE;

		// TODO en,ja以外にも対応する必要がある。
		short preferedLangId = MACOSX ? (LOCALE.getLanguage().equals("ja") ? LANG_MAC_ja : LANG_MAC_en) :
							   WINDOWS ? (LOCALE.getLanguage().equals("ja") ? LANG_WIN_ja_JP : LANG_WIN_en_US) : -1;

		// platform_id が実行環境と一致する方を選択する。
		if (name1.platform_id == preferedPlatId && name2.platform_id != preferedPlatId) {
			return name1;
		} else if (name1.platform_id != preferedPlatId && name2.platform_id == preferedPlatId) {
			return name2;
		}

		// どちらの platform_id も実行環境と一致しない場合、PLATFORM_UNICODE の方を選択する。
		if (name1.platform_id == PLATFORM_UNICODE && name2.platform_id != PLATFORM_UNICODE) {
			return name1;
		} else if (name1.platform_id != PLATFORM_UNICODE && name2.platform_id == PLATFORM_UNICODE) {
			return name2;
		}

		// language_id が実行環境と一致する方を選択する。
		if (name1.language_id == preferedLangId && name2.language_id != preferedLangId) {
			return name1;
		} else if (name1.language_id != preferedLangId && name2.language_id == preferedLangId) {
			return name2;
		}

		// どちらの language_id も実行環境と一致しない場合、LANG_MAC_en または LANG_WIN_en_US と一致する方を選択する。 
		preferedLangId = MACOSX ? LANG_MAC_en : WINDOWS ? LANG_WIN_en_US : -1;
		if (name1.language_id == preferedLangId && name2.language_id != preferedLangId) {
			return name1;
		} else if (name1.language_id != preferedLangId && name2.language_id == preferedLangId) {
			return name2;
		}

		// どちらも一致度が同じ場合は name1 を選択する。
		return name1;
	}

	private static class SfntName {
		final short platform_id;
		//final short encoding_id;
		final short language_id;
		final short name_id;
		final String string;

		SfntName(FT_SfntName ftSfntName) {
			platform_id = ftSfntName.platform_id;
			//encoding_id = ftSfntName.encoding_id;
			language_id = ftSfntName.language_id;
			name_id = ftSfntName.name_id;
			string = ftSfntName.stringAsJavaString();
		}
	}


	public List<Font> list() {
		return _fontList;
	}

	public Font get(String psName) {
		return _fontMap.get(psName);
	}

	public Font defaultFont() {
		return _defaultFont;
	}

}

interface WinShell32 extends Library {

	static final WinShell32 instance = (WinShell32) Native.loadLibrary("shell32", WinShell32.class);


	int MAX_PATH = 260;
	int CSIDL_FONTS = 0x0014;

	/**
	 * HRESULT SHGetFolderPathW(
	 *		HWND   hwndOwner,
	 *		int    nFolder,
	 *		HANDLE hToken,
	 *		DWORD  dwFlags,
	 *		LPWSTR pszPath)
	 */
	int SHGetFolderPathW(
			Pointer hwndOwner,
			int     nFolder,
			Pointer hToken,
			int     dwFlags,
			char[]  pszPath);

}
