/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IConvolutionSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

public class ConvolutionSupportImpl implements IConvolutionSupport {

	@ShaderSource
	public static final String[] CONVOLUTION = {
		"uniform sampler2DRect texture;",
		"uniform int ksize;",
		"uniform float kernel[101];",
		"uniform vec2 offset[101];",
		"",
		"void main(void)",
		"{",
		"	vec2 texCoord = gl_TexCoord[0].st;",
		"	vec4 sum = vec4(0.0);",
		"	for (int i = 0; i < ksize; ++i) {",
		"		sum += kernel[i] * texture2DRect(texture, texCoord + offset[i]);",
		"	}",
		"	gl_FragColor = sum;",
		"}"
	};


	private final IVideoRenderSupport support;

	private final IShaderProgram program;

	@Inject
	public ConvolutionSupportImpl(IVideoRenderSupport support, IShaderRegistry shaders) {
		this.support = support;
		program = shaders.getProgram(ConvolutionSupportImpl.class, "CONVOLUTION");
	}

	private Set<GLUniformData> uniforms(float[] kernel, float[] offset) {
		int ksize = kernel.length;

		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		uniforms.add(new GLUniformData("texture", 0));
		uniforms.add(new GLUniformData("ksize", ksize));
		uniforms.add(new GLUniformData("kernel[0]", 1, FloatBuffer.wrap(kernel, 0, ksize)));
		uniforms.add(new GLUniformData("offset[0]", 2, FloatBuffer.wrap(offset, 0, ksize*2)));

		return uniforms;
	}

	public IVideoBuffer convolve(
			IVideoBuffer input, IVideoBuffer output,
			float[] kernel, float[] coordOffset,
			Runnable operation, int pushAttribs) {

		return support.useShaderProgram(
				program, uniforms(kernel, coordOffset), operation, pushAttribs, output, input);
	}

	public IVideoBuffer convolve(
			IVideoBuffer input, IVideoBuffer output,
			float[] kernel, float[] coordOffset) {

		return support.useShaderProgram(
				program, uniforms(kernel, coordOffset), output, input);
	}

	private float[] offset2D(int ksize) {
		if ((ksize % 2) == 0) {
			throw new IllegalArgumentException("kernel size must be odd number");
		}

		float[] offset = new float[ksize*ksize*2];
		int half = ksize / 2;

		for (int j = 0; j < ksize; ++j) {
			for (int i = 0; i < ksize; ++i) {
				int k = j*ksize+i;
				offset[k*2  ] = i - half;
				offset[k*2+1] = j - half;
			}
		}

		return offset;
	}

	public IVideoBuffer convolve2D(
			IVideoBuffer input, IVideoBuffer output,
			int kernelSize, float[] kernel,
			Runnable operation, int pushAttribs) {

		return convolve(input, output, kernel, offset2D(kernelSize), operation, pushAttribs);
	}

	public IVideoBuffer convolve2D(
			IVideoBuffer input, IVideoBuffer output,
			int kernelSize, float[] kernel) {

		return convolve(input, output, kernel, offset2D(kernelSize));
	}

	private float[] offset1D(int ksize, ConvolutionDirection direction) {
		if ((ksize % 2) == 0) {
			throw new IllegalArgumentException("kernel size must be odd number");
		}

		float[] offset = new float[ksize*2];

		int half = ksize / 2;
		int j = direction.ordinal();
		int k = 1 - j;

		for (int i = 0; i < ksize; ++i) {
			offset[i*2 + j] = i - half;
			offset[i*2 + k] = 0;
		}

		return offset;
	}

	public IVideoBuffer convolve1D(
			IVideoBuffer input, IVideoBuffer output,
			float[] kernel, ConvolutionDirection direction,
			Runnable operation, int pushAttribs) {

		return convolve(input, output, kernel, offset1D(kernel.length, direction), operation, pushAttribs);
	}

	public IVideoBuffer convolve1D(
			IVideoBuffer input, IVideoBuffer output,
			float[] kernel, ConvolutionDirection direction) {

		return convolve(input, output, kernel, offset1D(kernel.length, direction));
	}

}
