/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.io.File;

import net.arnx.jsonic.JSONHint;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.AbstractItem;
import ch.kuramo.javie.core.ImageSequenceItem;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaOptions;
import ch.kuramo.javie.core.MediaSource;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.MediaSourceFactory;

import com.google.inject.Inject;
import com.google.inject.Injector;

@ProjectElement("imageSequence")
public class ImageSequenceItemImpl extends AbstractItem implements ImageSequenceItem {

	private File _file;

	private MediaSource _mediaSource;

	private MediaOptions _mediaOptions;

	private MediaInput _mediaInput;

	private VideoBounds _placeholderBounds;

	private final MediaSourceFactory _mediaSourceFactory;

	private final Injector _injector;

	@Inject
	public ImageSequenceItemImpl(MediaSourceFactory mediaSourceFactory, Injector injector) {
		super();
		_mediaSourceFactory = mediaSourceFactory;
		_injector = injector;
	}

	public void initialize(File file, Time frameDuration) {
		super.initialize();

		ImageSequenceSource imgSeqSource = _mediaSourceFactory.get(ImageSequenceSource.class, getId(), file);

		if (imgSeqSource != null) {
			_file = imgSeqSource.getFirstFile();
			setName(_file.getName());

			_mediaSource = imgSeqSource;
			_mediaOptions = imgSeqSource.validateOptions(null);
			_mediaOptions.setVideoFrameDuration(frameDuration);
			_mediaInput = new MediaInputImpl();
		}
	}

	@Override
	protected void finalize() throws Throwable {
		dispose();
		super.finalize();
	}

	public void dispose() {
		_mediaInput = null;

		if (_mediaSource != null) {
			_mediaSourceFactory.release(getId());
			_mediaSource = null;
		}
	}

	public File getFile() {
		return _file;
	}

	public void setFile(File file) {
		_file = file;
	}

	@JSONHint(ignore=true)
	public MediaSource getMediaSource() {
		return _mediaSource;
	}

	public MediaOptions getMediaOptions() {
		return (_mediaOptions != null) ? _mediaOptions.clone() : null;
	}

	public void setMediaOptions(MediaOptions options) {
		if (_mediaSource != null) {
			_mediaOptions = _mediaSource.validateOptions(options);

		} else if (_mediaInput != null) {
			// プレースホルダが使用されている場合
			throw new UnsupportedOperationException();

		} else {
			// デコード時
			_mediaOptions = (options != null) ? options.clone() : null;
		}
	}

	@JSONHint(ignore=true)
	public MediaInput getMediaInput() {
		return _mediaInput;
	}

	public VideoBounds getPlaceholderBounds() {
		_placeholderBounds = (_mediaSource != null && _mediaSource.isVideoAvailable())
					? _mediaSource.getVideoFrameBounds(_mediaOptions) : null;
		return _placeholderBounds;
	}

	public void setPlaceholderBounds(VideoBounds placeholderBounds) {
		_placeholderBounds = placeholderBounds;
	}

	@Override
	public void afterDecode(Project p) throws ProjectDecodeException {
		super.afterDecode(p);

		ImageSequenceSource imgSeqSource = _mediaSourceFactory.get(ImageSequenceSource.class, getId(), _file);

		if (imgSeqSource != null) {
			_file = imgSeqSource.getFirstFile();
			_mediaSource = imgSeqSource;
			_mediaOptions = imgSeqSource.validateOptions(_mediaOptions);
			_mediaInput = new MediaInputImpl();
		} else {
			MediaInputPlaceholderImpl placeholder = _injector.getInstance(MediaInputPlaceholderImpl.class);
			placeholder.setVideoFrameBounds(_placeholderBounds);
			_mediaInput = placeholder;
		}
	}

	@JSONHint(ignore=true)
	public String getKind() {
		return "イメージシーケンス";
	}

	private class MediaInputImpl implements MediaInput {

		public boolean isVideoAvailable() {
			return _mediaSource.isVideoAvailable();
		}

		public boolean isAudioAvailable() {
			return _mediaSource.isAudioAvailable();
		}

		public Time getDuration() {
			return _mediaSource.getDuration(_mediaOptions);
		}

		public Time getVideoFrameDuration() {
			return _mediaSource.getVideoFrameDuration(_mediaOptions);
		}

		public VideoBounds getVideoFrameBounds() {
			return _mediaSource.getVideoFrameBounds(_mediaOptions);
		}

		public IVideoBuffer getVideoFrame(Time mediaTime) {
			return _mediaSource.getVideoFrame(mediaTime, _mediaOptions);
		}

		public IAudioBuffer getAudioChunk(Time mediaTime) {
			return _mediaSource.getAudioChunk(mediaTime, _mediaOptions);
		}

	}

}
