/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeSet;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Scale;
import org.eclipse.swt.widgets.Slider;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.swt.widgets.TreeItem;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.ColorUtil;
import ch.kuramo.javie.app.ImageUtil;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.app.project.LayerSlipOperation;
import ch.kuramo.javie.app.project.ModifyKeyframeInterpolationOperation;
import ch.kuramo.javie.app.project.ModifyLayerInPointOperation;
import ch.kuramo.javie.app.project.ModifyLayerOutPointOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.ProjectOperation;
import ch.kuramo.javie.app.project.RemoveKeyframesOperation;
import ch.kuramo.javie.app.project.ShiftKeyframesOperation;
import ch.kuramo.javie.app.project.ShiftLayerTimesOperation;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Interpolation;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.TimeCode;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class TimelineManager {

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");

	private static final int FRAME_WIDTH = 50;
	private static final int LEFT_MARGIN = 30;
	private static final int RIGHT_MARGIN = 40;


	private final ProjectManager projectManager;

	private final LayerComposition composition;

	private final LayerCompositionView view;

	private final Composite meter;

	private final Scale zoomScale;

	private final Slider scrollSlider;

	private final TreeColumn timelineColumn;


	private double zoom;

	private Time newTime;

	private Time currentTime;

	private Time fromTime;

	private Time toTime;

	private boolean seeking;

	private boolean wrapMode = true;

	private final Map<Keyframe<?>, AnimatableValueElement> keyframeSelection = Util.newMap();

	private AbstractDragGestureEditor dragGestureEditor;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public TimelineManager(
			ProjectManager projectManager, LayerComposition composition, LayerCompositionView view,
			Composite meter, Scale zoomScale, Slider scrollSlider, Tree tree) {

		super();
		InjectorHolder.getInjector().injectMembers(this);

		this.projectManager = projectManager;
		this.composition = composition;
		this.view = view;
		this.meter = meter;
		this.zoomScale = zoomScale;
		this.scrollSlider = scrollSlider;
		this.timelineColumn = tree.getColumn(tree.getColumnCount() - 1);

		zoom = Double.NaN;

		meter.setBackground(ColorUtil.tableBackground());
		meter.addPaintListener(new PaintListener() {
			public void paintControl(PaintEvent e) {
				drawMeter(e);
			}
		});

		meter.addMouseListener(new MouseAdapter() {
			public void mouseDown(MouseEvent e) {
				mouseDownOnMeter(e);
			}
		});

		// TODO これは暫定。
		zoomScale.setMinimum(0);
		zoomScale.setMaximum(10);
		zoomScale.setSelection(5);

		zoomScale.setIncrement(1);
		zoomScale.setPageIncrement(1);

		scrollSlider.setMinimum(0);
		scrollSlider.setIncrement(FRAME_WIDTH);

		update(Time.fromFrameNumber(0, composition.getFrameDuration()), true);
	}

	public void update(Time currentTime, boolean reveal) {
		// キーフレームなどをドラッグ中はrevealしない。
		// TODO しかし、ドラッグしたままタイムラインの領域の端まで来たらスクロールするようにはしたい。
		if (dragGestureEditor != null) {
			reveal = false;
		}

		int oldScroll = scrollSlider.getSelection();
		double oldZoom = this.zoom;

		Time compDuration = composition.getDuration();
		Time frameDuration = composition.getFrameDuration();

		// TODO zoomの値の計算方法はもっとよく考える必要がある。
		double zoom = Math.pow(2.0, zoomScale.getSelection() - zoomScale.getMaximum());

		int totalPixels = timeToPixels(compDuration, frameDuration, zoom) + RIGHT_MARGIN;
		int colWidth = timelineColumn.getWidth();

		scrollSlider.setMaximum(totalPixels);
		scrollSlider.setThumb(colWidth);
		scrollSlider.setPageIncrement(colWidth - LEFT_MARGIN - RIGHT_MARGIN);
		scrollSlider.setEnabled(totalPixels > colWidth);

		boolean wrap = wrapMode;

		if (!reveal && zoom != oldZoom && !Double.isNaN(oldZoom)) {
			Time oldFromTime = pixelsToTime(oldScroll, frameDuration, oldZoom);
			Time oldToTime = pixelsToTime(oldScroll + colWidth, frameDuration, oldZoom);

			if (currentTime.before(oldFromTime) || currentTime.after(oldToTime)) {
				Time tmp = oldFromTime.add(oldToTime);
				Time center = new Time(tmp.timeValue/2, tmp.timeScale);
				scrollSlider.setSelection(timeToPixels(center, frameDuration, zoom) - colWidth/2);

			} else {
				int offset = timeToPixels(currentTime, frameDuration, oldZoom) - timeToPixels(oldFromTime, frameDuration, oldZoom);
				scrollSlider.setSelection(timeToPixels(currentTime, frameDuration, zoom) - offset);

				reveal = true;
				wrap = false;
			}
		}

		if (reveal) {
			Time leftTime = pixelsToTime(scrollSlider.getSelection() + LEFT_MARGIN, frameDuration, zoom);
			Time rightTime = pixelsToTime(scrollSlider.getSelection() + colWidth - RIGHT_MARGIN
														- (int)(FRAME_WIDTH * zoom), frameDuration, zoom);

			if (currentTime.before(leftTime) || (wrap && currentTime.after(rightTime))) {
				scrollSlider.setSelection(timeToPixels(currentTime, frameDuration, zoom) - LEFT_MARGIN);

			} else if (!wrap && currentTime.after(rightTime)) {
				Time t = currentTime.add(leftTime).subtract(rightTime);
				scrollSlider.setSelection(timeToPixels(t, frameDuration, zoom) - LEFT_MARGIN);
			}
		}

		this.zoom = zoom;
		this.currentTime = currentTime;
		this.fromTime = pixelsToTime(scrollSlider.getSelection(), frameDuration, zoom);
		this.toTime = pixelsToTime(scrollSlider.getSelection() + colWidth, frameDuration, zoom);

		redraw();

		// キーフレームナビゲータも再描画する。
		redrawShowhideColumn();
	}

	public void redraw() {
		int x = timelineColumnX();
		Tree tree = timelineColumn.getParent();
		Rectangle clientArea = tree.getClientArea();
		tree.redraw(clientArea.x+x, clientArea.y, clientArea.width-x, clientArea.height, true);
		meter.redraw();
	}

	private void redrawShowhideColumn() {
		Tree tree = timelineColumn.getParent();
		int col = LayerCompositionView.SHOWHIDE_COL;
		int x = 0;
		for (int i = 0; i < col; ++i) {
			x += tree.getColumn(i).getWidth();
		}
		int width = tree.getColumn(col).getWidth();
		Rectangle clientArea = tree.getClientArea();
		tree.redraw(x, clientArea.y, width, clientArea.height, true);
	}

	private int timeToPixels(Time time, Time frameDuration, double zoom) {
		return (int)Math.round(FRAME_WIDTH * zoom * time.toSecond() / frameDuration.toSecond()) + LEFT_MARGIN;
	}

	private Time pixelsToTime(int pixels, Time frameDuration, double zoom) {
		double frames = (pixels - LEFT_MARGIN) / (FRAME_WIDTH * zoom);
		return new Time(Math.round(frames * frameDuration.timeValue), frameDuration.timeScale);
	}

	private int timelineColumnX() {
		Tree tree = timelineColumn.getParent();
		int x = 0;
		for (int i = 0, n = tree.getColumnCount()-1; i < n; ++i) {
			TreeColumn column = tree.getColumn(i);
			x += column.getWidth();
		}
		return x;
	}

	public void drawCurrentFrameRegion(Event e) {
		Time frameDuration = composition.getFrameDuration();
		int scroll = scrollSlider.getSelection();

		int x1 = timeToPixels(currentTime, frameDuration, zoom) - scroll;
		int x2 = timeToPixels(currentTime.add(frameDuration), frameDuration, zoom) - scroll;

		e.gc.setBackground(e.display.getSystemColor(SWT.COLOR_BLUE));
		e.gc.setAlpha(48);
		e.gc.fillRectangle(e.x+x1, e.y, x2-x1, e.height);
		e.gc.setAlpha(255);
	}

	public void drawTimeIndicatorLine(PaintEvent e) {
		Time frameDuration = composition.getFrameDuration();
		int scroll = scrollSlider.getSelection();

		int left = timeToPixels(fromTime, frameDuration, zoom) - scroll;

//		if (seeking && newTime != null && !newTime.equals(currentTime)) {
//			int x = timeToPixels(newTime, frameDuration, zoom) - scroll;
//			if (left <= x) {
//				x += timelineColumnX();
//				e.gc.setForeground(e.display.getSystemColor(SWT.COLOR_DARK_RED));
//				e.gc.drawLine(x, e.y, x, e.y+e.height);
//			}
//		}

		int x = timeToPixels(currentTime, frameDuration, zoom) - scroll;
		if (left <= x) {
			x += timelineColumnX();
			e.gc.setForeground(e.display.getSystemColor(SWT.COLOR_RED));
			e.gc.drawLine(x, e.y, x, e.y+e.height);
		}
	}

	public void drawColumnLeftLine(PaintEvent e) {
		int x = timelineColumnX() - 1;
		e.gc.setForeground(ColorUtil.tableRowLine());
		e.gc.drawLine(x, e.y, x, e.y+e.height);
	}

	private void drawMeter(PaintEvent e) {
		Display display = e.display;
		GC gc = e.gc;

		Time frameDuration = composition.getFrameDuration();
		int scroll = scrollSlider.getSelection();
		int h = meter.getBounds().height;

		gc.setForeground(display.getSystemColor(SWT.COLOR_BLACK));

		long step = (long)(1/zoom);
		long i0 = fromTime.toFrameNumber(frameDuration) - step;
		long i1 = toTime.toFrameNumber(frameDuration) + step;
		i0 = Math.max((i0/step)*step, 0);
		for (long i = i0; i <= i1; i += step) {
			Time t = Time.fromFrameNumber(i, frameDuration);
			int x = timeToPixels(t, frameDuration, zoom) - scroll;

			String text = TimeCode.toTimeCode(t, frameDuration, true, false);
			Point extent = gc.textExtent(text);

			gc.drawLine(x, extent.y, x, h);
			if ((i/step)%3 == 0) {
				gc.drawText(text, x-extent.x/2, 0);
			}
		}

		if (seeking && newTime != null && !newTime.equals(currentTime)) {
			int x = timeToPixels(newTime, frameDuration, zoom) - scroll;
			gc.setForeground(display.getSystemColor(SWT.COLOR_DARK_RED));
			gc.drawLine(x, 0, x, h);
		}

		int x = timeToPixels(currentTime, frameDuration, zoom) - scroll;
		gc.setForeground(display.getSystemColor(SWT.COLOR_RED));
		gc.drawLine(x, 0, x, h);
	}

	private void mouseDownOnMeter(final MouseEvent e) {
		final SortedSet<Time> snapTimes = collectSnapTimes(false, true, false, false);

		seeking = true;
		setWrapMode(false);
		updateView(e.x, (e.stateMask & SWT.SHIFT) != 0, snapTimes);

		final Control control = (Control) e.widget;
		final Display display = e.display;

		class MeterMouseTracker implements Runnable, Listener {
			private int x = e.x;
			private boolean snap = (e.stateMask & SWT.SHIFT) != 0;
			private boolean end;

			public void run() {
				if (!end) {
					updateView(x, snap, snapTimes);
					display.timerExec(30, this);
				}
			}

			public void handleEvent(Event e) {
				switch (e.type) {
					case SWT.MouseMove:
						if (!control.isDisposed()) {
							if (COCOA && (getCocoaCurrentButtonState() & 1) == 0) {
								break;
							}

							x = control.toControl(display.getCursorLocation()).x;
							snap = (e.stateMask & SWT.SHIFT) != 0;
							updateView(x, snap, snapTimes);
							break;
						}
						// fall through

					case SWT.MouseUp:
					case SWT.Deactivate:
						end = true;
						display.removeFilter(SWT.MouseMove, this);
						display.removeFilter(SWT.MouseUp, this);
						display.removeFilter(SWT.Deactivate, this);
						setWrapMode(true);
						seeking = false;
						redraw();
						break;
				}
			}
		}

		MeterMouseTracker tracker = new MeterMouseTracker();
		display.timerExec(30, tracker);
		display.addFilter(SWT.MouseMove, tracker);
		display.addFilter(SWT.MouseUp, tracker);
		display.addFilter(SWT.Deactivate, tracker);
	}

	private void updateView(int x, boolean snap, SortedSet<Time> snapTimes) {
		Time frameDuration = composition.getFrameDuration();
		int scroll = scrollSlider.getSelection();

		Time time = pixelsToTime(x + scroll, frameDuration, zoom);

		if (time.timeValue < 0) {
			time = new Time(0, time.timeScale);
		} else if (!time.before(composition.getDuration())) {
			time = composition.getDuration().subtract(frameDuration);
		}

		if (snap) {
			SortedSet<Time> head = snapTimes.headSet(time);
			SortedSet<Time> tail = snapTimes.tailSet(time);

			Time prev = head.isEmpty() ? null : head.last();
			int dxPrev = (prev != null) ? Math.abs(timeToPixels(prev, frameDuration, zoom)-scroll-x) : Integer.MAX_VALUE;
			if (dxPrev < 20) {
				time = prev;
			}

			Time next = tail.isEmpty() ? null : tail.first();
			int dxNext = (next != null) ? Math.abs(timeToPixels(next, frameDuration, zoom)-scroll-x) : Integer.MAX_VALUE;
			if (dxNext < 20 && dxNext < dxPrev) {
				time = next;
			}
		}

		setTime(time);
	}

	public void drawLayer(Event e, Layer layer) {
		Time frameDuration = composition.getFrameDuration();
		int scroll = scrollSlider.getSelection();

		int x = e.x;
		int y = e.y;
		int h = e.height-1;
		GC gc = e.gc;

		boolean hasSlipBar = false;

		if (layer instanceof MediaItemLayer && !LayerNature.isTimeRemapEnabled(layer)) {
			MediaInput input = ((MediaItemLayer) layer).getMediaInput();
			if (input != null) {
				Time duration = input.getDuration();
				if (duration != null) {
					Time startTime = layer.getStartTime();
					Time endTime = startTime.add(new Time((long)(duration.timeValue/Math.abs(layer.getRate())), duration.timeScale));

					int x1 = x + timeToPixels(startTime, frameDuration, zoom) - scroll;
					int x2 = x + timeToPixels(endTime, frameDuration, zoom) - scroll;

					gc.setBackground(e.display.getSystemColor(SWT.COLOR_RED));
					gc.setAlpha(64);
					gc.fillRectangle(x1, y, x2-x1, h);

					hasSlipBar = true;
				}
			}
		}

		Time inPoint = layer.getInPoint();
		Time outPoint = layer.getOutPoint();

		int x1 = x + timeToPixels(inPoint, frameDuration, zoom) - scroll;
		int x2 = x + timeToPixels(outPoint, frameDuration, zoom) - scroll;

		if (!hasSlipBar) {
			gc.setBackground(e.display.getSystemColor(SWT.COLOR_RED));
			gc.setAlpha(64);
			gc.fillRectangle(x1, y, x2-x1, h);
		}
		gc.setBackground(e.display.getSystemColor(SWT.COLOR_DARK_RED));
		gc.setAlpha(128);
		gc.fillRectangle(x1, y, x2-x1, h);
		gc.setAlpha(255);


		drawCurrentFrameRegion(e);
	}

	public void updateCursor(MouseEvent e, Layer layer) {
		if (dragGestureEditor != null) {
			return;
		}

		Cursor cursor = null;

		Time frameDuration = composition.getFrameDuration();

		Time inPoint = layer.getInPoint();
		Time outPoint = layer.getOutPoint();

		int x = e.x - timelineColumnX() + scrollSlider.getSelection();
		int x1 = timeToPixels(inPoint, frameDuration, zoom);
		int x2 = timeToPixels(outPoint, frameDuration, zoom);

		if (x1-3 < x && x < x1+3) {
			cursor = e.display.getSystemCursor(SWT.CURSOR_SIZEE);

		} else if (x2-3 < x && x < x2+3) {
			cursor = e.display.getSystemCursor(SWT.CURSOR_SIZEW);

		} else if (x1 <= x && x < x2) {
			cursor = e.display.getSystemCursor(SWT.CURSOR_HAND);

		} else if (layer instanceof MediaItemLayer && !LayerNature.isTimeRemapEnabled(layer)) {
			MediaInput input = ((MediaItemLayer) layer).getMediaInput();
			if (input != null) {
				Time duration = input.getDuration();
				if (duration != null) {
					Time startTime = layer.getStartTime();
					Time endTime = startTime.add(new Time((long)(duration.timeValue/Math.abs(layer.getRate())), duration.timeScale));

					x1 = timeToPixels(startTime, frameDuration, zoom);
					x2 = timeToPixels(endTime, frameDuration, zoom);

					if (x1 <= x && x < x2) {
						cursor = e.display.getSystemCursor(SWT.CURSOR_HAND);
					}
				}
			}
		}

		Tree tree = timelineColumn.getParent();
		tree.setCursor(cursor);
	}

	public void mouseDown(MouseEvent e, Layer layer) {
		if (dragGestureEditor != null) {
			return;
		}

		Time frameDuration = composition.getFrameDuration();

		Time inPoint = layer.getInPoint();
		Time outPoint = layer.getOutPoint();

		int x = e.x - timelineColumnX() + scrollSlider.getSelection();
		int x1 = timeToPixels(inPoint, frameDuration, zoom);
		int x2 = timeToPixels(outPoint, frameDuration, zoom);

		if (x1-3 < x && x < x1+3) {
			clearKeyframeSelection();
			dragGestureEditor = new DragGestureInPointShifter(e, layer, pixelsToTime(x, frameDuration, zoom));

		} else if (x2-3 < x && x < x2+3) {
			clearKeyframeSelection();
			dragGestureEditor = new DragGestureOutPointShifter(e, layer, pixelsToTime(x, frameDuration, zoom));

		} else if (x1 <= x && x < x2) {
			clearKeyframeSelection();
			dragGestureEditor = new DragGestureLayerTimeShifter(e, layer, pixelsToTime(x, frameDuration, zoom));

		} else if (layer instanceof MediaItemLayer && !LayerNature.isTimeRemapEnabled(layer)) {
			MediaInput input = ((MediaItemLayer) layer).getMediaInput();
			if (input != null) {
				Time duration = input.getDuration();
				if (duration != null) {
					Time startTime = layer.getStartTime();
					Time endTime = startTime.add(new Time((long)(duration.timeValue/Math.abs(layer.getRate())), duration.timeScale));

					x1 = timeToPixels(startTime, frameDuration, zoom);
					x2 = timeToPixels(endTime, frameDuration, zoom);

					if (x1 <= x && x < x2) {
						dragGestureEditor = new DragGestureSlipEditor(e, layer, pixelsToTime(x, frameDuration, zoom));
					}
				}
			}
			if (dragGestureEditor == null) {
				clearKeyframeSelection();			
			}
		} else {
			clearKeyframeSelection();			
		}
	}

	public int drawKeyframes(Event e, AnimatableValue<?> avalue) {
		Time frameDuration = composition.getFrameDuration();
		int scroll = scrollSlider.getSelection();

		int ex = e.x;
		int y = e.y + e.height/2 - 1;
		GC gc = e.gc;

		for (Keyframe<?> kf : avalue.getKeyframeMap().subMap(fromTime, toTime).values()) {
			int x = ex + timeToPixels(kf.time, frameDuration, zoom) - scroll;
			Image icon = ImageUtil.getKeyframeIcon(kf.interpolation, keyframeSelection.containsKey(kf));
			if (icon != null) {
				gc.drawImage(icon, x-5, y-5);
			} else {
				gc.setForeground(e.display.getSystemColor(SWT.COLOR_BLACK));
				gc.drawRectangle(x-4, y-4, 8, 8);
			}
		}

		return y;
	}

	private Keyframe<?> findKeyframe(MouseEvent e, int yKeyframe, AnimatableValue<?> avalue) {
		if (e.y-4 > yKeyframe || e.y+4 < yKeyframe) {
			return null;
		}

		int x = e.x - timelineColumnX();
		int scroll = scrollSlider.getSelection();
		Time frameDuration = composition.getFrameDuration();

		Time time1 = pixelsToTime(x-4+scroll, frameDuration, zoom);
		Time time2 = pixelsToTime(x+5+scroll, frameDuration, zoom);	// headMap は引数の時刻を「含まない」ので
																	// time2 は x+4 ではなく x+5 の時刻で計算する。 
		SortedMap<Time, ?> head = avalue.getKeyframeMap().headMap(time2);
		if (head.isEmpty()) {
			return null;
		}

		Keyframe<?> kf = (Keyframe<?>) head.get(head.lastKey());
		return time1.after(kf.time) ? null : kf;
	}

	public void mouseDown(
			MouseEvent e, int yKeyframe, AnimatableValueElement element, AnimatableValue<?> avalue) {

		Keyframe<?> kf = findKeyframe(e, yKeyframe, avalue);
		if (kf == null) {
			if (keyframeSelection.isEmpty()) return;
			keyframeSelection.clear();

		} else if (!keyframeSelection.containsKey(kf)) {
			if ((e.stateMask & SWT.SHIFT) != SWT.SHIFT) {
				keyframeSelection.clear();
			}
			keyframeSelection.put(kf, element);

		} else if (e.button == 1 && (e.stateMask & SWT.SHIFT) == SWT.SHIFT) {
			keyframeSelection.remove(kf);
		}

		redraw();


		if (kf != null && keyframeSelection.containsKey(kf)
				&& e.button == 1 && dragGestureEditor == null) {

			dragGestureEditor = new DragGestureKeyframeShifter(e, kf);
		}
	}

	public void mouseDown(MouseEvent e) {
		if (keyframeSelection.size() > 0 && e.x >= timelineColumnX()) {
			clearKeyframeSelection();
			redraw();
		}
	}

	public void clearKeyframeSelection() {
		keyframeSelection.clear();
	}

	public void addKeyframeSelection(Keyframe<?> keyframe, AnimatableValueElement element) {
		keyframeSelection.put(keyframe, element);
	}

	public boolean hasKeyframeSelection() {
		return keyframeSelection.size() > 0;
	}

	public Map<Keyframe<?>, AnimatableValueElement> getKeyframeSelection() {
		return Collections.unmodifiableMap(keyframeSelection);
	}

	private Object[][] createSelectedKeyframeData() {
		Object[][] data = new Object[keyframeSelection.size()][];

		int i = 0;
		for (Map.Entry<Keyframe<?>, AnimatableValueElement> e : keyframeSelection.entrySet()) {
			Keyframe<?> kf = e.getKey();
			AnimatableValueElement element = e.getValue();

			if (element instanceof LayerAnimatableValueElement) {
				LayerAnimatableValueElement avalueElem = (LayerAnimatableValueElement) element;
				data[i++] = new Object[] { avalueElem.layer.getId(), -1, avalueElem.property, kf };
			} else {
				EffectAnimatableValueElement avalueElem = (EffectAnimatableValueElement) element;
				data[i++] = new Object[] { avalueElem.layer.getId(),
											avalueElem.layer.getEffects().indexOf(avalueElem.effect),
											avalueElem.descriptor.getName(), kf };
			}
		}

		return data;
	}

	private String[] createBaseAnimatableValues(Object[][] keyframeData) {
		Map<AnimatableValue<?>, AnimatableValue<?>> map = Util.newMap();
		AnimatableValue<?>[] baseAvalues = new AnimatableValue[keyframeData.length];

		for (int i = 0, n = keyframeData.length; i < n; ++i) {
			Layer layer = composition.getLayer((String) keyframeData[i][0]);
			Integer effectIndex = (Integer) keyframeData[i][1];
			String property = (String) keyframeData[i][2];
			Keyframe<?> kf = (Keyframe<?>) keyframeData[i][3];

			AnimatableValue<?> avalue;
			if (effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(effectIndex);
				avalue = PropertyUtil.getProperty(effect, property);
			}

			AnimatableValue<?> baseAvalue = map.get(avalue);
			if (baseAvalue == null) {
				String s = _encoder.encodeElement(avalue);
				try {
					baseAvalue = _decoder.decodeElement(s, avalue.getClass());
				} catch (ProjectDecodeException e) {
					throw new JavieRuntimeException(e);
				}
				map.put(avalue, baseAvalue);
				baseAvalues[i] = baseAvalue;
			}
			baseAvalue.removeKeyframe(kf.time);
		}

		String[] encodedBaseAvalues = new String[baseAvalues.length];
		for (int i = 0, n = baseAvalues.length; i < n; ++i) {
			if (baseAvalues[i] != null) {
				encodedBaseAvalues[i] = _encoder.encodeElement(baseAvalues[i]);
			}
		}
		return encodedBaseAvalues;
	}

	public void removeSelectedKeyframes() {
		projectManager.postOperation(new RemoveKeyframesOperation(
				projectManager, composition, createSelectedKeyframeData()));
	}

	public void modifySelectedKeyframeInterpolation(Interpolation newInterpolation) {
		projectManager.postOperation(new ModifyKeyframeInterpolationOperation(
				projectManager, composition, createSelectedKeyframeData(), newInterpolation));
	}

	public void setWrapMode(boolean wrapMode) {
		this.wrapMode = wrapMode;
	}

	public Time getCurrentTime() {
		return currentTime;
	}

	public void setTime(Time time) {
		Time frameDuration = composition.getFrameDuration();
		newTime = Time.fromFrameNumber(time.toFrameNumber(frameDuration), frameDuration);

		if (!newTime.equals(currentTime)) {
			view.update(newTime);

			if (seeking) {
				//redraw();
				meter.redraw();
			}
		}
	}

	private SortedSet<Time> collectSnapTimes(
			boolean includesCurrentTime, boolean includesKeyframes,
			boolean excludesSelectedLayers, boolean excludesSelectedKeyframes) {

		Set<Layer> layers = Util.newSet();
		final Set<Keyframe<?>> keyframes = Util.newSet();

		Tree tree = timelineColumn.getParent();
		Rectangle clientArea = tree.getClientArea();
		final int clientTop = clientArea.y + tree.getHeaderHeight();
		final int clientBottom = clientArea.y + clientArea.height;

		TreeItem[] items = tree.getItems();
		for (int i = items.length-1; i >= 0; --i) {
			TreeItem item = items[i];
			Rectangle b = item.getBounds(LayerCompositionView.TIMELINE_COL);

			// clientAreaよりも下にある場合何もせずにひとつ上の行に移る。
			if (b.y >= clientBottom) {
				continue;
			}

			// レイヤー行の下辺がclientAreaの下辺よりも上にある場合、レイヤー行の子要素を調べる。
			// ただし、キーフレームにもスナップする場合のみ。
			if (includesKeyframes && b.y+b.height < clientBottom) {
				new Object() {
					void collectKeyframeTimes(TreeItem item) {
						if (item.getExpanded()) {
							for (TreeItem child : item.getItems()) {
								Object element = child.getData();
								if (element instanceof AnimatableValueElement) {
									Rectangle b = child.getBounds(LayerCompositionView.TIMELINE_COL);
									if (b.y < clientBottom && b.y+b.height > clientTop) {
										AnimatableValue<?> avalue = ((AnimatableValueElement) element).getAnimatableValue();
										keyframes.addAll(avalue.getKeyframeMap().values());
									}
								} else {
									collectKeyframeTimes(child);
								}
							}
						}
					}
				}.collectKeyframeTimes(item);
			}

			// レイヤー行の下辺がclientAreaの上辺よりも下にある場合、レイヤーのインポイントとアウトポイントを追加する。
			if (b.y+b.height > clientTop) {
				Object element = item.getData();
				layers.add(((LayerElement) element).layer);
			}

			// レイヤー行の上側がclientAreaの上辺と一致するか上にある場合、残りの行は処理する必要がない。
			if (b.y <= clientTop) {
				break;
			}
		}

		if (excludesSelectedLayers) {
			for (TreeItem item : tree.getSelection()) {
				Object element = item.getData();
				if (element instanceof LayerElement) {
					layers.remove(((LayerElement) element).layer);
				}
			}
		}

		if (excludesSelectedKeyframes) {
			keyframes.removeAll(keyframeSelection.keySet());
		}

		TreeSet<Time> snapTimes = new TreeSet<Time>();

		for (Layer layer : layers) {
			snapTimes.add(layer.getInPoint());
			snapTimes.add(layer.getOutPoint());
		}

		for (Keyframe<?> kf : keyframes) {
			snapTimes.add(kf.time);
		}

		if (includesCurrentTime) {
			snapTimes.add(currentTime);
		}

		snapTimes.add(Time.fromFrameNumber(0, composition.getFrameDuration()));
		snapTimes.add(composition.getDuration());

		return snapTimes;
	}


	// TODO getCocoaCurrentButtonState は AnimatableValueElementDelegate にも同じものがある。

	private static Method cocoaGetCurrentButtonStateMethod;

	private static int getCocoaCurrentButtonState() {
		try {
			if (cocoaGetCurrentButtonStateMethod == null) {
				Class<?> clazz = Class.forName("org.eclipse.swt.internal.cocoa.OS");
				cocoaGetCurrentButtonStateMethod = clazz.getMethod("GetCurrentButtonState");
			}
			return (Integer) cocoaGetCurrentButtonStateMethod.invoke(null);
		} catch (RuntimeException e) {
			throw e;
		} catch (Exception e) {
			throw new JavieRuntimeException(e);
		}
	}


	// TODO AnimatableValueElementDelegate.DragGestureEditor と基本的には同じ。できればうまく纏めたい。
	private abstract class AbstractDragGestureEditor {

		private final Time baseTime;

		private final SortedSet<Time> snapTimes;

		private final String relation = Util.randomId();

		private final long downTime;

		private final int downX;

		private boolean dragDetected;

		private Time prevTime;


		private AbstractDragGestureEditor(MouseEvent event, Time baseTime, SortedSet<Time> snapTimes) {
			Time frameDuration = composition.getFrameDuration();
			this.baseTime = prevTime = Time.fromFrameNumber(baseTime.toFrameNumber(frameDuration), frameDuration);

			this.snapTimes = snapTimes;

			Control control = (Control) event.widget;
			downTime = System.currentTimeMillis();
			downX = event.x;

			init(control);
		}

		private void init(final Control control) {
			final Display display = control.getDisplay();

			Listener listener = new Listener() {
				public void handleEvent(Event e) {
					switch (e.type) {
						case SWT.MouseMove:
							if (!control.isDisposed()) {
								if (COCOA && (getCocoaCurrentButtonState() & 1) == 0) {
									break;
								}

								int x = control.toControl(display.getCursorLocation()).x;

								if (!dragDetected) {
									dragDetected = (System.currentTimeMillis() - downTime > 100) && (Math.abs(x - downX) > 3);
								}

								if (dragDetected) {
									drag(x, (e.stateMask & SWT.SHIFT) != 0);
								}

								break;
							}
							// fall through

						case SWT.MouseUp:
						case SWT.Deactivate:
							dragGestureEditor = null;

							display.removeFilter(SWT.MouseMove, this);
							display.removeFilter(SWT.MouseUp, this);
							display.removeFilter(SWT.Deactivate, this);
							break;
					}
				}
			};

			display.addFilter(SWT.MouseMove, listener);
			display.addFilter(SWT.MouseUp, listener);
			display.addFilter(SWT.Deactivate, listener);
		}

		private void drag(int x, boolean snap) {
			Time frameDuration = composition.getFrameDuration();
			Time time = pixelsToTime(x-timelineColumnX()+scrollSlider.getSelection(), frameDuration, zoom);
			time = Time.fromFrameNumber(time.toFrameNumber(frameDuration), frameDuration);
			if (!time.equals(prevTime)) {
				projectManager.postOperation(createOperation(time, prevTime, baseTime, relation, snap));
				prevTime = time;
			}
		}

		protected abstract ProjectOperation createOperation(
				Time time, Time prevTime, Time baseTime, String relation, boolean snap);


		protected Time adjustDeltaTime(Time deltaTime, boolean snap, Time...oldTimes) {
			Time oldTime = oldTimes[0];
			Time newTime = deltaTime.add(oldTime);	// deltaTimeにoldTimeを加えているのは、deltaTimeのタイムスケールで計算するため。

			Time frameDuration = composition.getFrameDuration();

			if (snap) {
				int dxMin = 20;

				for (Time oldTime2 : oldTimes) {
					Time newTime2 = deltaTime.add(oldTime2);
					int x = timeToPixels(newTime2, frameDuration, zoom);

					SortedSet<Time> head = snapTimes.headSet(newTime2);
					SortedSet<Time> tail = snapTimes.tailSet(newTime2);

					Time prev = head.isEmpty() ? null : head.last();
					int dxPrev = (prev != null) ? Math.abs(timeToPixels(prev, frameDuration, zoom)-x) : Integer.MAX_VALUE;
					if (dxPrev < dxMin) {
						dxMin = dxPrev;
						oldTime = oldTime2;
						newTime = prev;
					}

					Time next = tail.isEmpty() ? null : tail.first();
					int dxNext = (next != null) ? Math.abs(timeToPixels(next, frameDuration, zoom)-x) : Integer.MAX_VALUE;
					if (dxNext < dxMin) {
						dxMin = dxNext;
						oldTime = oldTime2;
						newTime = next;
					}
				}
			}

			Time t = Time.fromFrameNumber(newTime.toFrameNumber(frameDuration), frameDuration);
			if (newTime.subtract(t).toSecond()*2 < frameDuration.toSecond()) {
				newTime = t;
			} else {
				newTime = t.add(frameDuration);
			}

			// TODO この計算で誤差が発生するかも(oldTimeとnewTimeのタイムスケールが異なることによる)
			return newTime.subtract(oldTime);
		}
	}


	private class DragGestureInPointShifter extends AbstractDragGestureEditor {

		private final Time baseInPoint;

		private final Object[][] layersAndInPoints;


		private DragGestureInPointShifter(MouseEvent event, Layer baseLayer, Time baseTime) {
			super(event, baseTime, collectSnapTimes(true, true/*flase*/, true, false));
			baseInPoint = baseLayer.getInPoint();
			layersAndInPoints = getLayersAndInPoints();
		}

		private Object[][] getLayersAndInPoints() {
			List<Object[]> list = Util.newList();
			for (TreeItem treeItem : timelineColumn.getParent().getSelection()) {
				Object element = treeItem.getData();
				if (element instanceof LayerElement) {
					Layer layer = ((LayerElement) element).layer;
					list.add(new Object[] { layer, layer.getInPoint() });
				}
			}
			return list.toArray(new Object[list.size()][]);
		}

		@Override
		protected ProjectOperation createOperation(Time time, Time prevTime, Time baseTime, String relation, boolean snap) {
			Time deltaTime = adjustDeltaTime(time.subtract(baseTime), snap, baseInPoint);
			return new ModifyLayerInPointOperation(
					projectManager, composition, layersAndInPoints, deltaTime, relation);
		}

	}

	private class DragGestureOutPointShifter extends AbstractDragGestureEditor {

		private final Time baseOutPoint;

		private final Object[][] layersAndOutPoints;


		private DragGestureOutPointShifter(MouseEvent event, Layer baseLayer, Time baseTime) {
			super(event, baseTime, collectSnapTimes(true, true/*flase*/, true, false));
			baseOutPoint = baseLayer.getOutPoint();
			layersAndOutPoints = getLayersAndOutPoints();
		}

		private Object[][] getLayersAndOutPoints() {
			List<Object[]> list = Util.newList();
			for (TreeItem treeItem : timelineColumn.getParent().getSelection()) {
				Object element = treeItem.getData();
				if (element instanceof LayerElement) {
					Layer layer = ((LayerElement) element).layer;
					list.add(new Object[] { layer, layer.getOutPoint() });
				}
			}
			return list.toArray(new Object[list.size()][]);
		}

		@Override
		protected ProjectOperation createOperation(Time time, Time prevTime, Time baseTime, String relation, boolean snap) {
			Time deltaTime = adjustDeltaTime(time.subtract(baseTime), snap, baseOutPoint);
			return new ModifyLayerOutPointOperation(
					projectManager, composition, layersAndOutPoints, deltaTime, relation);
		}

	}

	private class DragGestureSlipEditor extends AbstractDragGestureEditor {

		private final Time baseStartTime;

		private final Object[][] layersAndStartTimes;

		private final Object[][] selectedKeyframeData;

		private final String[] baseAnimatableValues;


		private DragGestureSlipEditor(MouseEvent event, Layer baseLayer, Time baseTime) {
			super(event, baseTime, new TreeSet<Time>());
			baseStartTime = baseLayer.getStartTime();
			layersAndStartTimes = getLayersAndStartTimes();
			selectedKeyframeData = createSelectedKeyframeData();
			baseAnimatableValues = createBaseAnimatableValues(selectedKeyframeData);
		}

		private Object[][] getLayersAndStartTimes() {
			List<Object[]> list = Util.newList();
			for (TreeItem treeItem : timelineColumn.getParent().getSelection()) {
				Object element = treeItem.getData();
				if (element instanceof LayerElement) {
					Layer layer = ((LayerElement) element).layer;
					list.add(new Object[] { layer, layer.getStartTime() });
				}
			}
			return list.toArray(new Object[list.size()][]);
		}

		@Override
		protected ProjectOperation createOperation(Time time, Time prevTime, Time baseTime, String relation, boolean snap) {
			Time deltaTime = adjustDeltaTime(time.subtract(baseTime), false, baseStartTime);
			return new LayerSlipOperation(
					projectManager, composition,
					layersAndStartTimes, selectedKeyframeData, baseAnimatableValues,
					deltaTime, relation);
		}

	}

	private class DragGestureLayerTimeShifter extends AbstractDragGestureEditor {

		private final Time baseInPoint;

		private final Time baseOutPoint;

		private final List<Layer> layers = Util.newList();

		private DragGestureLayerTimeShifter(MouseEvent event, Layer baseLayer, Time baseTime) {
			super(event, baseTime, collectSnapTimes(true, true, true, false));
			baseInPoint = baseLayer.getInPoint();
			baseOutPoint = baseLayer.getOutPoint();
			for (TreeItem treeItem : timelineColumn.getParent().getSelection()) {
				Object element = treeItem.getData();
				if (element instanceof LayerElement) {
					layers.add(((LayerElement) element).layer);
				}
			}
		}

		@Override
		protected ProjectOperation createOperation(Time time, Time prevTime, Time baseTime, String relation, boolean snap) {
			Time deltaTime = adjustDeltaTime(time.subtract(baseTime), snap, baseInPoint, baseOutPoint);
			return new ShiftLayerTimesOperation(projectManager, composition, layers, deltaTime, relation);
		}

	}

	private class DragGestureKeyframeShifter extends AbstractDragGestureEditor {

		private final Keyframe<?> baseKeyframe;

		private final Object[][] selectedKeyframeData;

		private final String[] baseAnimatableValues;


		private DragGestureKeyframeShifter(MouseEvent event, Keyframe<?> baseKeyframe) {
			super(event, baseKeyframe.time, collectSnapTimes(true, true, false, true));
			this.baseKeyframe = baseKeyframe;
			selectedKeyframeData = createSelectedKeyframeData();
			baseAnimatableValues = createBaseAnimatableValues(selectedKeyframeData);
		}

		@Override
		protected ProjectOperation createOperation(Time time, Time prevTime, Time baseTime, String relation, boolean snap) {
			Time deltaTime = adjustDeltaTime(time.subtract(baseTime), snap, baseKeyframe.time);
			return new ShiftKeyframesOperation(
					projectManager, composition, selectedKeyframeData, baseAnimatableValues, deltaTime, relation);
		}

	}

}
