/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.player;

import java.awt.Dimension;
import java.awt.EventQueue;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Queue;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicReference;

import javax.media.opengl.DebugGL2;
import javax.media.opengl.GL2;
import javax.media.opengl.GLAutoDrawable;
import javax.media.opengl.GLContext;
import javax.media.opengl.GLEventListener;
import javax.media.opengl.awt.GLCanvas;
import javax.media.opengl.glu.GLU;

import org.eclipse.core.runtime.ListenerList;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.ScrollBar;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.player.GLCanvasFactory.GLCanvasRecord;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.FrameDuration;
import ch.kuramo.javie.core.ImageSequenceItem;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaInputPlaceholder;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.SolidColorItem;
import ch.kuramo.javie.core.TimeCode;
import ch.kuramo.javie.core.misc.AtiIntelLock;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;
import com.sun.opengl.util.gl2.GLUT;

public class VideoCanvas extends Thread implements GLEventListener {

	private static final Logger _logger = LoggerFactory.getLogger(VideoCanvas.class);

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");

	private static final Time TIME0 = Time.fromFrameNumber(0, FrameDuration.FPS_29_97);


	private final MediaItem _mediaItem;

	private final MediaInput _mediaInput;

	private final TimeKeeper _timeKeeper;

	private GLCanvasRecord _glCanvasRecord;

	private final ScrolledComposite _scrolled;

	private final GLCanvas _canvas;

	private GLU _glu;

	private GLUT _glut;

	private final AtomicReference<Object[]> _videoBufferAndTimeRef = new AtomicReference<Object[]>();

	private final Queue<IVideoBuffer> _oldVideoBuffers = new ConcurrentLinkedQueue<IVideoBuffer>();

	private volatile Time _frameDuration;

	private volatile Resolution _resolution = Resolution.FULL;

	private double _zoom;

	private volatile boolean _showInfo;

	private Size2i _canvasSize;

	private int _hScroll;

	private int _vScroll;

	private int _scrollAreaHeight;

	private ControlListener _resizeListener;

	private volatile int[] _viewport = new int[4];

	private final Object _monitor = new Object();

	private boolean _glInitialized;

	private volatile boolean _playing;

	private volatile boolean _forceRender;

	private volatile boolean _finished;

	private volatile boolean _displaying;

	private final ListenerList _playerThreadListeners = new ListenerList();

	private final MovingAverage _delayAverage = new MovingAverage(30);

	private final MovingAverage _frameDurationAvg = new MovingAverage(30);

	@Inject
	private RenderContext _context;

	@Inject
	private GLGlobal _glGlobal;

//	private final boolean _intel;

	private final AtiIntelLock _atiIntelLock;


	public VideoCanvas(Composite parent, MediaItem mediaItem, TimeKeeper timeKeeper) {
		super("VideoCanvas: " + mediaItem.getName());

		MediaInput mediaInput = mediaItem.getMediaInput();
		if (mediaInput == null) {
			throw new IllegalArgumentException("no MediaInput is available");
		}
		if (!mediaInput.isVideoAvailable()) {
			throw new IllegalArgumentException("no video is available");
		}

		InjectorHolder.getInjector().injectMembers(this);

		_mediaItem = mediaItem;
		_mediaInput = mediaInput;
		_timeKeeper = timeKeeper;

		_glCanvasRecord = GLCanvasFactory.getFactory().getGLCanvas(parent);
		_scrolled = _glCanvasRecord.scrolled;
		_canvas = _glCanvasRecord.glCanvas;

		_canvas.addGLEventListener(this);

		if (COCOA) {
			cocoaListenToScrollAndResize();
			_scrollAreaHeight = _scrolled.getClientArea().height;
		} else {
			listenToResize();
		}

		setZoom(1.0);

//		_intel = _glGlobal.isIntel();
		_atiIntelLock = AtiIntelLock.get(_glGlobal);
	}

	public Control getControl() {
		return _scrolled;
	}

	private void cleanup() {
		_canvas.removeGLEventListener(this);

		if (_resizeListener != null) {
			_scrolled.removeControlListener(_resizeListener);
			_resizeListener = null;
		}

		if (_glCanvasRecord != null) {
			GLCanvasFactory.getFactory().releaseGLCanvas(_glCanvasRecord);
			_glCanvasRecord = null;
		}
	}

	private void cocoaListenToScrollAndResize() {
		final ScrollBar hBar = _scrolled.getHorizontalBar();
		final ScrollBar vBar = _scrolled.getVerticalBar();

		hBar.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				int scroll = hBar.getSelection();
				if (_hScroll != scroll) {
					_hScroll = scroll;
					updateViewport();
				}
			}
		});

		vBar.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				int scroll = vBar.getSelection();
				if (_vScroll != scroll) {
					_vScroll = scroll;
					updateViewport();
				}
			}
		});

		_scrolled.addControlListener(new ControlAdapter() {
			public void controlResized(ControlEvent e) {
				if (_zoom == 0) {
					fitCanvasToScrollArea();
				}

				e.display.asyncExec(new Runnable() {
					public void run() {
						if (_scrolled.isDisposed()) {
							return;
						}
						int height = _scrolled.getClientArea().height;
						int hScroll = hBar.getSelection();
						int vScroll = vBar.getSelection();
						if (_scrollAreaHeight != height || _hScroll != hScroll || _vScroll != vScroll) {
							_scrollAreaHeight = height;
							_hScroll = hScroll;
							_vScroll = vScroll;
							updateViewport();
						}
					}
				});
			}
		});
	}

	private void listenToResize() {
		_resizeListener = new ControlAdapter() {
			public void controlResized(ControlEvent e) {
				if (_zoom == 0) {
					fitCanvasToScrollArea();
				}

				e.display.asyncExec(new Runnable() {
					public void run() {
						if (!_scrolled.isDisposed()) {
							_glCanvasRecord.awtFrame.repaint();
						}
					}
				});
			}
		};
		_scrolled.addControlListener(_resizeListener);
	}

	public void setFrameDuration(Time frameDuration) {
		_frameDuration = frameDuration;
	}

	public Resolution getResolution() {
		return _resolution;
	}

	public void setResolution(Resolution resolution) {
		if ((resolution != null && !resolution.equals(_resolution))
				|| (resolution == null && _resolution != null)) {

			_resolution = resolution;
			forceRender();
		}
	}

	public double getZoom() {
		return _zoom;
	}

	public void setZoom(double zoom) {
		if (_zoom != zoom) {
			_zoom = zoom;

			if (zoom == 0) {
				_scrolled.setMinSize(1, 1);
				fitCanvasToScrollArea();

			} else {
				VideoBounds bounds = _mediaInput.getVideoFrameBounds();
				int w = (int) (bounds.width*zoom);
				int h = (int) (bounds.height*zoom);

				_scrolled.setMinSize(w, h);
				setCanvasSize(w, h);
			}
		}
	}

	public boolean isShowInfo() {
		return _showInfo;
	}

	public void setShowInfo(boolean showInfo) {
		if (showInfo != _showInfo) {
			_showInfo = showInfo;
			_canvas.repaint();
		}
	}

	private void fitCanvasToScrollArea() {
		Rectangle scrollArea = _scrolled.getClientArea();
		VideoBounds videoBounds = _mediaInput.getVideoFrameBounds();

		int w, h;
		int availableWidth = scrollArea.width - 10;
		int availableHeight = scrollArea.height - 10;
		double scrollAspect = (double) availableWidth / availableHeight;
		double videoAspect = (double) videoBounds.width / videoBounds.height;
		if (scrollAspect > videoAspect) {
			w = (int) (availableHeight * videoAspect);
			h = availableHeight;
		} else {
			w = availableWidth;
			h = (int) (availableWidth / videoAspect);
		}

		setCanvasSize(w, h);
	}

	private void setCanvasSize(final int width, final int height) {
		_canvasSize = new Size2i(width, height);
		updateViewport();

		EventQueue.invokeLater(new Runnable() {
			public void run() {
				Dimension size = _canvas.getPreferredSize();
				if (size.width != width || size.height != height) {
					_canvas.setPreferredSize(new Dimension(width, height));
					_canvas.invalidate();
					_canvas.getParent().validate();
				}
			}
		});
	}

	private void updateViewport() {
		int[] viewport = new int[] { 0, 0, _canvasSize.width, _canvasSize.height };

		if (COCOA) {
			viewport[0] = -_hScroll;
			if (_scrollAreaHeight < _canvasSize.height) {
				viewport[1] = _vScroll - _canvasSize.height + _scrollAreaHeight;
			}
		}

		if (!Arrays.equals(_viewport, viewport)) {
			_viewport = viewport;
			_canvas.repaint();
		}
	}

	public void init(GLAutoDrawable drawable) {
		_atiIntelLock.lock();
		try {

			GL2 gl = drawable.getGL().getGL2();

			_glu = GLU.createGLU(gl);
			_glut = new GLUT();

			gl.glEnable(GL2.GL_TEXTURE_RECTANGLE);
			gl.glColor4f(1, 1, 1, 1);

			// 背景を黒以外にする場合はアルファブレンドを有効にし、blendFuncを次のように設定する。
			//gl.glEnable(GL2.GL_BLEND);
			//gl.glBlendFunc(GL2.GL_ONE, GL2.GL_ONE_MINUS_SRC_ALPHA);

			gl.glClearColor(0, 0, 0, 1);
			gl.glClear(GL2.GL_COLOR_BUFFER_BIT);

			gl.glMatrixMode(GL2.GL_MODELVIEW);
			gl.glLoadIdentity();

			_glInitialized = true;

		} finally {
			_atiIntelLock.unlock();
		}
	}

	public void display(GLAutoDrawable drawable) {
		if (_finished) {
			return;
		}

		if (!_glInitialized) {
			init(drawable);
		}

		Object[] vbAndTime = null;

		_displaying = true;
		_atiIntelLock.lock();
		try {

			GL2 gl = drawable.getGL().getGL2();

			// 背景を黒以外にする場合に必要。initメソッド内のコメント参照。
			//gl.glClear(GL2.GL_COLOR_BUFFER_BIT);

			vbAndTime = _videoBufferAndTimeRef.getAndSet(null);
			if (vbAndTime == null) {
				return;
			}

			IVideoBuffer vb = (IVideoBuffer) vbAndTime[0];

			int[] vp = _viewport;
			gl.glViewport(vp[0], vp[1], vp[2], vp[3]);

			VideoBounds bounds = vb.getBounds();
			int w = bounds.width;
			int h = bounds.height;

			gl.glMatrixMode(GL2.GL_PROJECTION);
			gl.glLoadIdentity();
			_glu.gluOrtho2D(0, w, h, 0);


			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, vb.getTexture());

			gl.glBegin(GL2.GL_QUADS);
			gl.glTexCoord2f(0, 0); gl.glVertex2f(0, 0);
			gl.glTexCoord2f(w, 0); gl.glVertex2f(w, 0);
			gl.glTexCoord2f(w, h); gl.glVertex2f(w, h);
			gl.glTexCoord2f(0, h); gl.glVertex2f(0, h);
			gl.glEnd();

			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

			if (_showInfo) {
				gl.glPushAttrib(GL2.GL_ENABLE_BIT | GL2.GL_CURRENT_BIT | GL2.GL_COLOR_BUFFER_BIT);

				gl.glLoadIdentity();
				_glu.gluOrtho2D(0, vp[2], vp[3], 0);

				drawInfo(gl, (Time) vbAndTime[1]);

				gl.glPopAttrib();
			}

			gl.glFinish();

		} finally {
			_atiIntelLock.unlock();

			if (vbAndTime != null && !_videoBufferAndTimeRef.compareAndSet(null, vbAndTime)) {
				_oldVideoBuffers.add((IVideoBuffer) vbAndTime[0]);
			}

			_displaying = false;
		}
	}

	private void drawInfo(GL2 gl, Time time) {
		List<String> info = new ArrayList<String>();

		VideoBounds bounds = _mediaInput.getVideoFrameBounds();
		Time frameDuration = _mediaInput.getVideoFrameDuration();

		if (_mediaItem instanceof CompositionItem) {
			String bpc;
			switch (((CompositionItem) _mediaItem).getComposition().getColorMode()) {
				case RGBA8: bpc = "8 bpc"; break;
				case RGBA16: bpc = "16 bpc"; break;
				case RGBA16_FLOAT: bpc = "16 bpc float"; break;
				case RGBA32_FLOAT: bpc = "32 bpc float"; break;
				default: bpc = "unknown bpc"; break;
			}
			info.add(String.format("Composition: %d x %d, %s", bounds.width, bounds.height, bpc));

		} else if (_mediaItem instanceof SolidColorItem) {
			Color color = ((SolidColorItem) _mediaItem).getColor();
			info.add(String.format("Solid Color: %d x %d, R:%d G:%d B:%d", bounds.width, bounds.height,
					(int)(color.r*255), (int)(color.g*255), (int)(color.b*255)));

		} else if (_mediaItem.getMediaInput() instanceof MediaInputPlaceholder) {
			info.add(String.format("Placeholder: %d x %d", bounds.width, bounds.height));

		} else if (_mediaItem instanceof ImageSequenceItem) {
			info.add(String.format("Image Sequence: %d x %d", bounds.width, bounds.height));

		} else if (frameDuration != null) {
			info.add(String.format("Video: %d x %d", bounds.width, bounds.height));

		} else {
			info.add(String.format("Image: %d x %d", bounds.width, bounds.height));
		}

		if (frameDuration != null) {
			if (frameDuration.timeValue > 0) {
				Time duration = _mediaInput.getDuration();
				info.add(String.format("Frame: %d / %d", time.toFrameNumber(frameDuration), duration.toFrameNumber(frameDuration)));
				info.add(String.format("Time: %s / %s", TimeCode.toTimeCode(time, frameDuration), TimeCode.toTimeCode(duration, frameDuration)));
				info.add(String.format("FPS: %.2f / %.2f", 1/_frameDurationAvg.getSampledAverage(), 1/frameDuration.toSecond()));
			} else {
				info.add(String.format("FPS: %.2f", 1/_frameDurationAvg.getSampledAverage()));
			}
		}

		int infoWidth = 0;
		int infoHeight = info.size() * 12;
		for (String s : info) {
			infoWidth = Math.max(infoWidth, _glut.glutBitmapLength(GLUT.BITMAP_HELVETICA_10, s));
		}
		infoWidth += 2;

		gl.glEnable(GL2.GL_BLEND);
		gl.glBlendFunc(GL2.GL_SRC_ALPHA, GL2.GL_ONE_MINUS_SRC_ALPHA);
		gl.glColor4f(0, 0, 0, 0.4f);
		gl.glBegin(GL2.GL_QUADS);
		gl.glVertex2f(10, 10);
		gl.glVertex2f(10+infoWidth, 10);
		gl.glVertex2f(10+infoWidth, 10+infoHeight);
		gl.glVertex2f(10, 10+infoHeight);
		gl.glEnd();

		gl.glDisable(GL2.GL_BLEND);
		gl.glColor4f(0, 1, 0, 1);
		int i = 0;
		for (String s : info) {
			gl.glRasterPos2f(11, 20+(i++)*12);
			_glut.glutBitmapString(GLUT.BITMAP_HELVETICA_10, s);
		}
	}

	public void reshape(GLAutoDrawable drawable, int x, int y, int width, int height) {
		// nothing to do
	}

	public void dispose(GLAutoDrawable drawable) {
		// nothing to do ?
	}

	public void run() {
		AudioMode audioMode = AudioMode.STEREO_48KHZ_INT16;

		_context.activate();
		try {
			// TODO DebugGLの使用をやめる。でも当分はこのままにしておく。
			_context.getGL(); // これをしておかないとRenderContext内でOpenGL関連の初期化が行われず、次のGLContext.getCurrent()がnullになる。
			GLContext glContext = GLContext.getCurrent();
			glContext.setGL(new DebugGL2(glContext.getGL().getGL2()));

			GL2 gl = _context.getGL().getGL2();

			Time duration = _mediaInput.getDuration();
			if (duration == null) {
				Resolution resolution = _resolution;

				_context.reset();
				_context.setVideoResolution(resolution);
				_context.setColorMode(ColorMode.RGBA8);

				IVideoBuffer vb = null;
				try {
					_atiIntelLock.lock();
					try {
						vb = _mediaInput.getVideoFrame(TIME0);
					} finally {
						_atiIntelLock.unlock();
					}
				} catch (Exception e) {
					_logger.error("error getting video frame", e);
				} catch (InternalError e) {
					if (!e.getMessage().contains("glGetError")) throw e;
					_logger.error("error getting video frame", e);
				}

				if (vb != null) {
					_atiIntelLock.lock();
					try {
						setTexParametersForDisplay(vb, resolution, gl);
						glFinish(gl, vb);
					} finally {
						_atiIntelLock.unlock();
					}

					_videoBufferAndTimeRef.getAndSet(new Object[] { vb, TIME0 });
					_canvas.repaint();
					fireThreadRender(TIME0);
				}
				fireThreadEndOfDuration(TIME0);

				synchronized (_monitor) {
					while (!_finished) {
						try { _monitor.wait(); } catch (InterruptedException e) { }
					}
				}
				return;
			}

			_forceRender = true;

			long prevFrame = -1;
			Time[] timeAndBase;
			Time prevBase = _timeKeeper.getTimeAndBase()[1];
			Time delay = TIME0;
			Resolution prevResolution = null;

			while ((timeAndBase = waitForRendering()) != null) {
				boolean forceRender = !_playing && _forceRender;
				_forceRender = false;

				Time frameDuration = _frameDuration;
				if (frameDuration == null) {
					Time fd = _mediaInput.getVideoFrameDuration();
					frameDuration = (fd.timeValue > 0) ? fd : FrameDuration.FPS_59_94;
				}

				boolean seek = (timeAndBase[1] != prevBase);
				prevBase = timeAndBase[1];

				Time compensatedTime = seek ? timeAndBase[0] : timeAndBase[0].add(delay);
				delay = TIME0;

				long frameNumber = compensatedTime.toFrameNumber(frameDuration);
				boolean early = false;
				if (!seek) {
					if (forceRender) {
						frameNumber = Math.max(frameNumber, prevFrame);
					} else if (frameNumber <= prevFrame) {
						frameNumber = prevFrame + 1;
						early = true;
					}
				}

				Time frameTime = Time.fromFrameNumber(frameNumber, frameDuration);

				boolean fireEndOfDuration = false;
				if (!frameTime.before(duration) && !forceRender) {
					Time lastFramePlusOne = duration.add(new Time(frameDuration.timeValue-1, frameDuration.timeScale));
					frameNumber = Math.max(lastFramePlusOne.toFrameNumber(frameDuration)-1, 0);
					frameTime = Time.fromFrameNumber(frameNumber, frameDuration);
					fireEndOfDuration = _playing;
					synchronized (_monitor) {
						_playing = false;
					}
				}

				if (forceRender) {
					_timeKeeper.setTime(frameTime);
				}

				Resolution resolution = _resolution;
				if (resolution != null) {
					prevResolution = null;
				} else if (forceRender) {
					resolution = Resolution.FULL;
					prevResolution = null;
				} else {
					resolution = autoResolution(prevResolution, frameDuration.toSecond()*1.5);
					prevResolution = resolution;
				}

				_context.reset();
				_context.setVideoResolution(resolution);
				_context.setColorMode(ColorMode.RGBA8);
				_context.setVideoFrameDuration(frameDuration);
				_context.setAudioMode(audioMode);
				_context.setAudioAnimationRate(audioMode.sampleRate/100);

				IVideoBuffer vb = null;

				if (_mediaItem instanceof CompositionItem) {
					PlayerLock.readLock().lock();
					try {
						// TODO prepareExpressionはプロジェクトに構造的な変更があった場合のみ行えばよい。
						Composition comp = ((CompositionItem) _mediaItem).getComposition();
						comp.prepareExpression(_context.createInitialExpressionScope(comp));

						_atiIntelLock.lock();
						try {
							vb = _mediaInput.getVideoFrame(frameTime);
						} finally {
							_atiIntelLock.unlock();
						}
					} catch (Exception e) {
						_logger.error("error getting video frame", e);
					} catch (InternalError e) {
						if (!e.getMessage().contains("glGetError")) throw e;
						_logger.error("error getting video frame", e);
					} finally {
						PlayerLock.readLock().unlock();
					}
				} else {
					try {
						_atiIntelLock.lock();
						try {
							vb = _mediaInput.getVideoFrame(frameTime);
						} finally {
							_atiIntelLock.unlock();
						}
					} catch (Exception e) {
						_logger.error("error getting video frame", e);
					} catch (InternalError e) {
						if (!e.getMessage().contains("glGetError")) throw e;
						_logger.error("error getting video frame", e);
					}
				}

				if (vb != null) {
					_atiIntelLock.lock();
					try {
						setTexParametersForDisplay(vb, resolution, gl);
						glFinish(gl, vb);
					} finally {
						_atiIntelLock.unlock();
					}

					delay = calcDelay(timeAndBase);

					if (early && !sleepUntilFrameTime(frameTime, timeAndBase[1])) {
						vb.dispose();
						vb = null;
						delay = TIME0;

					} else {
						Object[] old = _videoBufferAndTimeRef.getAndSet(new Object[] { vb, frameTime });
						_canvas.repaint();
						if (old != null) {
							((IVideoBuffer) old[0]).dispose();
						}

						fireThreadRender(frameTime);

						if (_logger.isTraceEnabled() && frameNumber > prevFrame + 1) {
							_logger.trace(frameNumber - prevFrame - 1 + " frames dropped");
						}

						prevFrame = frameNumber;

						if (forceRender) {
							_delayAverage.reset();
							_frameDurationAvg.reset();
						} else {
							_delayAverage.add(delay.toSecond());
							_frameDurationAvg.add((delay.before(frameDuration) ? frameDuration : delay).toSecond());
						}
					}
				}

				disposeOldVideoBuffers();

				if (fireEndOfDuration) {
					fireThreadEndOfDuration(frameTime);
				}
			}

		} finally {
			while (_displaying) {
				sleep(FrameDuration.FPS_29_97);
			}

			Object[] vbAndTime = _videoBufferAndTimeRef.getAndSet(null);
			if (vbAndTime != null) {
				((IVideoBuffer) vbAndTime[0]).dispose();
			}
			disposeOldVideoBuffers();

			_context.deactivate();
		}
	}

	public void forceRender() {
		synchronized (_monitor) {
			_forceRender = true;
			_monitor.notify();
		}
	}

	public void play(boolean play) {
		if (_playing == play) {
			return;
		}

		synchronized (_monitor) {
			_playing = play;
			_monitor.notify();
		}
	}

	public void finish() {
		if (_finished) {
			return;
		}

		synchronized (_monitor) {
			_finished = true;
			_monitor.notify();
		}

		try {
			join();
		} catch (InterruptedException e) {
			// ignore
		}

		cleanup();
	}

	private void glFinish(GL2 gl, IVideoBuffer vb) {
//		if (!_intel) {
//			gl.glFinish();
//			return;
//		}

		// Intel GMAの場合、カラーバッファに何もアタッチされていないとglFinishに失敗する。
		// (GMA X4500で確認) 

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, vb.getTexture(), 0);

		gl.glFinish();

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
	}

	private void setTexParametersForDisplay(IVideoBuffer vb, Resolution resolution, GL2 gl) {
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, vb.getTexture());

		// フル解像度のときは、拡大したときにテクスチャの補間が効かないようにする。
		// フル解像度でないときは補間を有効にするが、補間によってエッジ部分が
		// 黒とブレンドされないようにラッピングモードを GL_CLAMP_TO_EDGE に設定する。

		if (Resolution.FULL.equals(resolution)) {
			// GL_TEXTURE_MAG_FILTER だけ設定すればいいはずだが、
			// ATIではなぜかダメなようなので MIN/MAG 両方を設定している。
			gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_NEAREST);
			gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_NEAREST);

		} else {
			gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_EDGE);
			gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_EDGE);
		}

		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
	}

	private Resolution autoResolution(Resolution prevResolution, double avgThreshold) {
		if (prevResolution == null) {
			return Resolution.FULL;

		// avgThreshold 以下のときは現在の解像度を維持する。
		// MovingAverage.getAverage() は要素数が少ないとNaNを返すので不等号の向きに注意。
		} else if (_delayAverage.getAverage() > avgThreshold) {

			if (prevResolution.equals(Resolution.FULL)) {
				_delayAverage.reset();
				return Resolution.HALF;

			} else if (prevResolution.equals(Resolution.HALF)) {
				_delayAverage.reset();
				return Resolution.ONETHIRD;

			} else {
				return Resolution.QUARTER;
			}

		} else {
			return prevResolution;
		}
	}

	private Time[] waitForRendering() {
		if (!_playing && !_forceRender) {
			synchronized (_monitor) {
				while (!_playing && !_forceRender && !_finished) {
					try {
						_monitor.wait();
					} catch (InterruptedException e) {
						// ignore
					}
				}
			}
		}
		return _finished ? null : _timeKeeper.getTimeAndBase();
	}

	private void sleep(Time time) {
		if (time.timeValue > 0) {
			try {
				long sleep = (long)(time.toSecond() * 1000000000);
				Thread.sleep(sleep / 1000000, (int) (sleep % 1000000));
			} catch (InterruptedException e) {
			}
		}
	}

	private Time calcDelay(Time[] timeAndBase) {
		Time[] curTimeAndBase = _timeKeeper.getTimeAndBase();
		if (curTimeAndBase[1] != timeAndBase[1]) {
			// シークした場合。
			return TIME0;
		}

		Time delay = curTimeAndBase[0].subtract(timeAndBase[0]);
		return (delay.timeValue > 0) ? delay : TIME0;
	}

	private boolean sleepUntilFrameTime(Time frameTime, Time tkBase) {
		while (_playing && !_finished) {
			Time[] curTimeAndBase = _timeKeeper.getTimeAndBase();
			if (curTimeAndBase[1] != tkBase) {
				// シークした場合。
				return false;
			}

			Time sleepTime = frameTime.subtract(curTimeAndBase[0]);
			if (sleepTime.timeValue <= 0) {
				return true;
			}

			sleep(sleepTime);
		}

		return false;
	}

	private void disposeOldVideoBuffers() {
		for (Iterator<IVideoBuffer> it = _oldVideoBuffers.iterator(); it.hasNext(); ) {
			IVideoBuffer old = it.next();
			old.dispose();
			it.remove();
		}
	}

	public void addPlayerThreadListener(PlayerThreadListener listener) {
		_playerThreadListeners.add(listener);
	}

	public void removePlayerThreadListener(PlayerThreadListener listener) {
		_playerThreadListeners.remove(listener);
	}

	private void fireThreadRender(Time time) {
		PlayerThreadEvent event = new PlayerThreadEvent(this, time);
		for (Object l : _playerThreadListeners.getListeners()) {
			((PlayerThreadListener) l).threadRender(event);
		}
	}

	private void fireThreadEndOfDuration(Time time) {
		PlayerThreadEvent event = new PlayerThreadEvent(this, time);
		for (Object l : _playerThreadListeners.getListeners()) {
			((PlayerThreadListener) l).threadEndOfDuration(event);
		}
	}

}

class MovingAverage {

	@SuppressWarnings("serial")
	private final LinkedHashMap<Object, Double> map = new LinkedHashMap<Object, Double>() {

		protected boolean removeEldestEntry(Entry<Object, Double> eldest) {
			if (size() > numEntries) {
				sum -= eldest.getValue();
				return true;
			}
			return false;
		}
	};


	private final int numEntries;

	private final int sampleInterval;

	private double sum;

	private volatile double average;

	private int counter;

	private volatile double sampledAverage;


	MovingAverage(int numEntries, int sampleInterval) {
		this.numEntries = numEntries;
		this.sampleInterval = sampleInterval;
	}

	MovingAverage(int numEntries) {
		this(numEntries, 10);
	}

	void add(double value) {
		map.put(new Object(), value);

		sum += value;

		int size = map.size();
		average = (size < numEntries) ? Double.NaN : sum/size;

		if ((counter++ % sampleInterval) == 0) {
			sampledAverage = average;
		}
	}

	double getAverage() {
		return average;
	}

	double getSampledAverage() {
		return sampledAverage;
	}

	void reset() {
		map.clear();
		sum = 0;
		average = Double.NaN;
		counter = 0;
		sampledAverage = Double.NaN;
	}

}
