/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api;

/**
 * <p><code>Color</code>クラスは色を表現します。
 * <code>Color</code>オブジェクトは、色のRGB成分およびアルファ値をdouble値で保持します。
 * 通常、値の範囲は0.0から1.0ですが、範囲外の値を使用して<code>Color</code>オブジェクトを生成することも可能です。</p>
 * 
 * <p><code>Color</code>オブジェクトは不変です。
 * <code>Color</code>オブジェクトは作成したあとに変更できないため、共用することができます。</p>
 */
public final class Color {

	/**
	 * 無色透明を表します。
	 */
	public static final Color COLORLESS_TRANSPARENT = new Color(0, 0, 0, 0);

	/**
	 * 黒を表します。
	 */
	public static final Color BLACK = new Color(0, 0, 0, 1);

	/**
	 * 白を表します。
	 */
	public static final Color WHITE = new Color(1, 1, 1, 1);

	/**
	 * 50%のグレーを表します。
	 */
	public static final Color GRAY = new Color(0.5, 0.5, 0.5, 1);


	/**
	 * この色の赤成分です。
	 */
	public final double r;

	/**
	 * この色の緑成分です。
	 */
	public final double g;

	/**
	 * この色の青成分です。
	 */
	public final double b;

	/**
	 * この色のアルファ値です。
	 */
	public final double a;

	/**
	 * この<code>Color</code>オブジェクトのRGB成分およびアルファ値を
	 * 0.0から1.0の範囲にクランプ処理した<code>Color</code>オブジェクトです。
	 * 
	 * @see #clamp()
	 */
	private final Color clampedColor;


	/**
	 * RGB成分およびアルファ値を使って<code>Color</code>オブジェクトを生成します。
	 * 
	 * @param red   赤成分
	 * @param green 緑成分
	 * @param blue  青成分
	 * @param alpha アルファ値
	 */
	public Color(double red, double green, double blue, double alpha) {
		super();
		r = red;
		g = green;
		b = blue;
		a = alpha;

		if ((r >= 0 && r <= 1) && (g >= 0 && g <= 1) && (b >= 0 && b <= 1) && (a >= 0 && a <= 1)) {
			clampedColor = this;
		} else {
			clampedColor = new Color(
					Double.isNaN(r) ? 0 : Math.max(0, Math.min(1, r)),
					Double.isNaN(g) ? 0 : Math.max(0, Math.min(1, g)),
					Double.isNaN(b) ? 0 : Math.max(0, Math.min(1, b)),
					Double.isNaN(a) ? 1 : Math.max(0, Math.min(1, a))); 
		}
	}

	/**
	 * RGB成分を使ってアルファ値1.0の<code>Color</code>オブジェクトを生成します。
	 * 
	 * @param red   赤成分
	 * @param green 緑成分
	 * @param blue  青成分
	 */
	public Color(double red, double green, double blue) {
		this(red, green, blue, 1);
	}

	/**
	 * この<code>Color</code>オブジェクトのRGB成分およびアルファ値を、
	 * 0.0から1.0の範囲にクランプ処理した<code>Color</code>オブジェクトを返します。
	 * クランプ処理の必要がない場合はこのオブジェクト自身を返します。
	 * RGB成分のいずれかが<code>NaN</code>の場合、その成分をクランプ処理した結果は0.0となります。
	 * アルファ値が<code>NaN</code>の場合、クランプ処理した結果のアルファ値は1.0となります。
	 * 
	 * @return	クランプ処理の必要がない場合はこのオブジェクト自身、
	 * 			そうでない場合はこのオブジェクトのRGB成分およびアルファ値を
	 * 			0.0から1.0の範囲にクランプ処理した<code>Color</code>オブジェクト。
	 */
	public Color clamp() {
		return clampedColor;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		long temp;
		temp = Double.doubleToLongBits(a);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(b);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(g);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(r);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		return result;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Color other = (Color) obj;
		if (Double.doubleToLongBits(a) != Double.doubleToLongBits(other.a))
			return false;
		if (Double.doubleToLongBits(b) != Double.doubleToLongBits(other.b))
			return false;
		if (Double.doubleToLongBits(g) != Double.doubleToLongBits(other.g))
			return false;
		if (Double.doubleToLongBits(r) != Double.doubleToLongBits(other.r))
			return false;
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return "Color [r=" + r + ", g=" + g + ", b=" + b + ", a=" + a + "]";
	}

}
