/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.time;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.GLProgram;
import ch.kuramo.javie.api.annotations.GLShader;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.GLShader.ShaderType;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Echo", category=Effect.TIME)
public class Echo {

	@Property("-0.0333")
	private IAnimatableDouble echoTime;

	@Property(value="1", min="0")
	private IAnimatableInteger numberOfEchoes;

	@Property(value="1", min="0", max="1")
	private IAnimatableDouble startingIntensity;

	@Property(value="1", min="0")
	private IAnimatableDouble decay;

	@Property
	private IAnimatableEnum<EchoOperator> echoOperator;


	private final IVideoEffectContext context;

	private final IShaderRegistry shaders;

	@Inject
	public Echo(IVideoEffectContext context, IShaderRegistry shaders) {
		this.context = context;
		this.shaders = shaders;
	}

	public VideoBounds getVideoBounds() {
		Time time = context.getTime();
		double echoTime = context.value(this.echoTime);
		int numberOfEchoes = context.value(this.numberOfEchoes);

		VideoBounds bounds = null;
		for (int i = 0; i <= numberOfEchoes; ++i) {
			Time t = time.add(new Time(Math.round(i*echoTime*time.timeScale), time.timeScale));
			context.setTime(t);

			VideoBounds b = context.getPreviousBounds();

			if (bounds == null) {
				bounds = b;
			} else {
				double left = Math.min(b.x, bounds.x);
				double top = Math.min(b.y, bounds.y);
				double right = Math.max(b.x+b.width, bounds.x+bounds.width);
				double bottom = Math.max(b.y+b.height, bounds.y+bounds.height);
				bounds = new VideoBounds(left, top, (int)Math.ceil(right-left), (int)Math.ceil(bottom-top));
			}
		}
		return bounds;
	}

	public IVideoBuffer doVideoEffect() {
		Time time = context.getTime();
		double echoTime = context.value(this.echoTime);
		int numberOfEchoes = context.value(this.numberOfEchoes);
		double startingIntensity = context.value(this.startingIntensity);
		double decay = context.value(this.decay);
		EchoOperator op = context.value(this.echoOperator);
		IShaderProgram program = shaders.getProgram(Echo.class,
				(op == EchoOperator.COMPOSITE_IN_BACK || op == EchoOperator.COMPOSITE_IN_FRONT)
					? "COMPOSITE" : op.name());

		IVideoBuffer vb1 = null;
		try {
			GL gl = context.getGL();
			GLU glu = context.getGLU();

			for (int i = 0; i <= numberOfEchoes; ++i) {
				int echoIndex = (op == EchoOperator.COMPOSITE_IN_FRONT) ? numberOfEchoes-i : i;
				double intensity = Math.min(1.0, startingIntensity * Math.pow(decay, echoIndex));
				Time t = time.add(new Time(Math.round(echoIndex*echoTime*time.timeScale), time.timeScale));
				context.setTime(t);
	
				IVideoBuffer vb2 = null;
				IVideoBuffer vb3 = null;
				try {
					vb2 = context.doPreviousEffect();

					if (vb1 == null) {
						vb1 = context.createVideoBuffer(vb2.getBounds());
						VideoEffectUtil.clearTexture(vb1, gl);
					}

					VideoBounds b1 = vb1.getBounds();
					VideoBounds b2 = vb2.getBounds();
					double left = Math.min(b1.x, b2.x);
					double top = Math.min(b1.y, b2.y);
					double right = Math.max(b1.x+b1.width, b2.x+b2.width);
					double bottom = Math.max(b1.y+b1.height, b2.y+b2.height);
					VideoBounds bounds = new VideoBounds(left, top, (int)Math.ceil(right-left), (int)Math.ceil(bottom-top));

					vb3 = context.createVideoBuffer(bounds);
					VideoEffectUtil.clearTexture(vb3, gl);


					VideoEffectUtil.ortho2D(gl, glu, bounds.width, bounds.height);

					gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
							GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, vb3.getTexture(), 0);
					gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

					gl.glActiveTexture(GL.GL_TEXTURE0);
					gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, vb1.getTexture());
					gl.glActiveTexture(GL.GL_TEXTURE1);
					gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, vb2.getTexture());

					synchronized (program) {
						gl.glUseProgram(program.getProgram());
						gl.glUniform1i(program.getUniformLocation("texDst"), 0);
						gl.glUniform1i(program.getUniformLocation("texSrc"), 1);
						gl.glUniform1f(program.getUniformLocation("intensity"), (float)intensity);

						gl.glBegin(GL.GL_QUADS);
						gl.glMultiTexCoord2f(GL.GL_TEXTURE0, (float)(bounds.x-b1.x), (float)(bounds.y-b1.y));
						gl.glMultiTexCoord2f(GL.GL_TEXTURE1, (float)(bounds.x-b2.x), (float)(bounds.y-b2.y));
						gl.glVertex2f(0, 0);
						gl.glMultiTexCoord2f(GL.GL_TEXTURE0, (float)(bounds.x-b1.x+bounds.width), (float)(bounds.y-b1.y));
						gl.glMultiTexCoord2f(GL.GL_TEXTURE1, (float)(bounds.x-b2.x+bounds.width), (float)(bounds.y-b2.y));
						gl.glVertex2f(bounds.width, 0);
						gl.glMultiTexCoord2f(GL.GL_TEXTURE0, (float)(bounds.x-b1.x+bounds.width), (float)(bounds.y-b1.y+bounds.height));
						gl.glMultiTexCoord2f(GL.GL_TEXTURE1, (float)(bounds.x-b2.x+bounds.width), (float)(bounds.y-b2.y+bounds.height));
						gl.glVertex2f(bounds.width, bounds.height);
						gl.glMultiTexCoord2f(GL.GL_TEXTURE0, (float)(bounds.x-b1.x), (float)(bounds.y-b1.y+bounds.height));
						gl.glMultiTexCoord2f(GL.GL_TEXTURE1, (float)(bounds.x-b2.x), (float)(bounds.y-b2.y+bounds.height));
						gl.glVertex2f(0, bounds.height);
						gl.glEnd();

						gl.glFinish();
						gl.glUseProgram(0);
					}

					gl.glActiveTexture(GL.GL_TEXTURE1);
					gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);
					gl.glActiveTexture(GL.GL_TEXTURE0);
					gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);

					gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
							GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

				} finally {
					if (vb1 != null) { vb1.dispose(); vb1 = null; }
					if (vb2 != null) { vb2.dispose(); vb2 = null; }
					vb1 = vb3;
					vb3 = null;
				}
			}

			VideoEffectUtil.premultiply(vb1, gl, glu);

			IVideoBuffer result = vb1;
			vb1 = null;
			return result;

		} finally {
			if (vb1 != null) vb1.dispose();
		}
	}

	private static final String[] createProgramSource(String name) {
		return new String[] {
								"uniform sampler2DRect texDst;",
								"uniform sampler2DRect texSrc;",
								"uniform float intensity;",
								"",
				String.format(	"vec4 blend_%s(vec4 pDst, vec4 pSrc, float intensity);", name),
								"",
								"void main(void)",
								"{",
								"	vec4 dst = texture2DRect(texDst, gl_TexCoord[0].st);",
								"	vec4 src = texture2DRect(texSrc, gl_TexCoord[1].st);",
				String.format(	"	dst = blend_%s(vec4(dst.rgb*dst.a, dst.a), src, intensity);", name),
								"	gl_FragColor = (dst.a != 0.0) ? vec4(dst.rgb/dst.a, dst.a) : vec4(0.0);",
								"}"
		};
	}

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions")
	public static final String[] ADD = createProgramSource("add");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions")
	public static final String[] MAXIMUM = createProgramSource("lighten");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions")
	public static final String[] MINIMUM = createProgramSource("darken");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions")
	public static final String[] SCREEN = createProgramSource("screen");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions")
	public static final String[] COMPOSITE = createProgramSource("normal");

}
