/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#pragma once


class CVideoPushPin : public CSourceStream
{
public:
	DECLARE_IUNKNOWN

	CVideoPushPin(HRESULT* hr, CSource* filter,
		long width, long height, LONGLONG frameDuration, BOOL hasAlpha);

	virtual ~CVideoPushPin();

	HRESULT GetMediaType(CMediaType* mediaType);
	HRESULT DecideBufferSize(IMemAllocator* alloc, ALLOCATOR_PROPERTIES* request);
	HRESULT FillBuffer(IMediaSample* sample);

	STDMETHODIMP Notify(IBaseFilter *pSelf, Quality q) { return E_FAIL; }

	void WriteBuffer(LONGLONG timeStart, LONGLONG timeEnd, void* buffer);

private:
	long mWidth;
	long mHeight;
	LONGLONG mFrameDuration;
	BOOL mHasAlpha;

	LONGLONG mTimeStart;
	LONGLONG mTimeEnd;
	void* mBuffer;
	HANDLE mWaitEvent1;
	HANDLE mWaitEvent2;
};


class CVideoSource : public CSource
{
public:
	DECLARE_IUNKNOWN

	CVideoSource(LPUNKNOWN unk, HRESULT* hr,
		long width, long height, LONGLONG frameDuration, BOOL hasAlpha);

	virtual ~CVideoSource();

	CVideoPushPin* GetPushPin() { return mPushPin; }

private:
	CVideoPushPin* mPushPin;
};


class CAudioPushPin : public CSourceStream
{
public:
	DECLARE_IUNKNOWN

	CAudioPushPin(HRESULT* hr, CSource* filter,
		long channels, long sampleRate, long sampleSize, BOOL float_);

	virtual ~CAudioPushPin();

	HRESULT GetMediaType(CMediaType* mediaType);
	HRESULT DecideBufferSize(IMemAllocator* alloc, ALLOCATOR_PROPERTIES* request);
	HRESULT FillBuffer(IMediaSample* sample);

	STDMETHODIMP Notify(IBaseFilter *pSelf, Quality q) { return E_FAIL; }

	void WriteBuffer(LONGLONG time, long audioFrameCount, void* buffer);

private:
	long mChannels;
	long mSampleRate;
	long mSampleSize;
	BOOL mFloat;

	LONGLONG mTime;
	long mAudioFrameCount;
	void* mBuffer;
	HANDLE mWaitEvent1;
	HANDLE mWaitEvent2;
};


class CAudioSource : public CSource
{
public:
	DECLARE_IUNKNOWN

	CAudioSource(LPUNKNOWN unk, HRESULT* hr,
		long channels, long sampleRate, long sampleSize, BOOL float_);

	virtual ~CAudioSource();

	CAudioPushPin* GetPushPin() { return mPushPin; }

private:
	CAudioPushPin* mPushPin;
};


class DirectShowOutput
{
public:
	DirectShowOutput();
	~DirectShowOutput();

	HRESULT Initialize(LPCWSTR file);
	HRESULT AddVideoSource(long width, long height, LONGLONG frameDuration, BOOL hasAlpha,
						LPCWSTR vcName, double vcQuality, long vcKeyFrameRate, long vcPFramesPerKey,
						LONGLONG vcBitRate, LONGLONG vcWindowSize, void* vcConfig, size_t vcConfigSize);
	HRESULT AddAudioSource(long channels, long sampleRate, long sampleSize, BOOL float_);
	HRESULT SetMasterStreamAndInterleaving(REFERENCE_TIME interleave);
	HRESULT RunGraph();

	void WriteVideo(LONGLONG timeStart, LONGLONG timeEnd, void* buffer)
	{
		mVideoSource->GetPushPin()->WriteBuffer(timeStart, timeEnd, buffer);
	}

	void WriteAudio(LONGLONG time, long audioFrameCount, void* buffer)
	{
		mAudioSource->GetPushPin()->WriteBuffer(time, audioFrameCount, buffer);
	}

private:
	ICaptureGraphBuilder2* mCapBuilder2;
	IBaseFilter* mAVIMux;
	IGraphBuilder* mGraphBuilder;
	IMediaControl* mMediaControl;
	CVideoSource* mVideoSource;
	CAudioSource* mAudioSource;

	HRESULT GetAudioStreamIndex(int* index);

	void AddToRot();
	void RemoveFromRot();
	DWORD mRotRegister;
};

class VideoCompressorEnumerator
{
public:
	VideoCompressorEnumerator(long width, long height, LONGLONG frameDuration, BOOL hasAlpha);
	~VideoCompressorEnumerator();

	HRESULT Next();

	LPCWSTR GetDisplayName()		{ return mDisplayName; }
	LPCWSTR GetFriendlyName()		{ return mFriendlyName.bstrVal; }
	long GetCapabilities()			{ return mCapabilities; }
	BOOL GetCanConfigDialog()		{ return mCanConfigDialog; }
	BOOL GetCanAboutDialog()		{ return mCanAboutDialog; }
	double GetDefaultQuality()		{ return mDefaultQuality; }
	long GetDefaultKeyFrameRate()	{ return mDefaultKeyFrameRate; }
	long GetDefaultPFramesPerKey()	{ return mDefaultPFramesPerKey; }
	LONGLONG GetDefaultBitRate()	{ return mDefaultBitRate; }

private:
	const long mWidth;
	const long mHeight;
	const LONGLONG mFrameDuration;
	const BOOL mHasAlpha;

	IEnumMoniker* mEnumMoniker;
	IMalloc* mMalloc;

	LPOLESTR mDisplayName;
	VARIANT mFriendlyName;
	long mCapabilities;
	double mDefaultQuality;
	long mDefaultKeyFrameRate;
	long mDefaultPFramesPerKey;
	LONGLONG mDefaultBitRate;
	BOOL mCanConfigDialog;
	BOOL mCanAboutDialog;

	HRESULT ReleaseCurrent();
};

class VideoCompressorUtil
{
public:
	static HRESULT BuildDummyGraph(
		long width, long height, LONGLONG frameDuration, BOOL hasAlpha,
		IBaseFilter* vcompFilter, IGraphBuilder** dummyGraph);

	static HRESULT GetPin(IBaseFilter* filter, PIN_DIRECTION pindir, IPin** pin);

	static HRESULT GetBitRate(IPin* pin, LONGLONG* bitRate);

	static HRESULT SetBitRate(IPin* pin, LONGLONG bitRate);

	static HRESULT OpenConfigDialog(
		long width, long height, LONGLONG frameDuration, BOOL hasAlpha,
		LPCWSTR displayName, void** ioConfig, size_t* ioConfigSize, HWND hwnd);

	static HRESULT OpenAboutDialog(LPCWSTR displayName, HWND hwnd);

private:
	VideoCompressorUtil() {}
	~VideoCompressorUtil() {}
};
