/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.shaders;

import ch.kuramo.javie.api.annotations.GLProgram;
import ch.kuramo.javie.api.annotations.GLShader;
import ch.kuramo.javie.api.annotations.GLShader.ShaderType;

public class BlendModeShaders {

	private static final String[] createProgramSource(String name, boolean temporal) {
		return new String[] {
								"uniform sampler2DRect texDst;",
								"uniform sampler2DRect texSrc;",
								"uniform float opacity;",
				temporal ?		"uniform float time;" : "",
								"",
				String.format(	"vec4 blend_%s(vec4 pDst, vec4 pSrc, float opacity%s);", name, temporal ? ", float time" : ""),
								"",
								"void main(void)",
								"{",
								"	vec2 texCoord = gl_TexCoord[0].st;",
								"	vec4 dst = texture2DRect(texDst, texCoord);",
								"	vec4 src = texture2DRect(texSrc, texCoord);",
				String.format(	"	dst = blend_%s(vec4(dst.rgb*dst.a, dst.a), src, opacity%s);", name, temporal ? ", time" : ""),
								"	gl_FragColor = (dst.a != 0.0) ? vec4(dst.rgb/dst.a, dst.a) : vec4(0.0);",
								"}"
		};
	}

	private static final String[] createProgramSource(String name) {
		return createProgramSource(name, false);
	}


	@GLShader(ShaderType.FRAGMENT_SHADER)
	public static final String[] blend_functions = {
		/**
		 * 輝度算出用定数
		 */
		"const vec3 lumaVec = vec3(0.299, 0.587, 0.114);",

		/**
		 * ディザ合成用乱数
		 */
		"const vec2 randVec = vec2(12.9898, 78.233);",
		"",
		"float rand1(vec2 coord, float time)",
		"{",
		"	return fract(sin(dot(coord, randVec + time)) * 43758.5453);",
		"}",

		/**
		 * 通常
		 */
		"vec4 blend_normal(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	pSrc *= opacity;",
		"	return pSrc + pDst*(1.0-pSrc.a);",
		"}",

		/**
		 * ディザ合成
		 */
		"vec4 blend_dissolve(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float rand = rand1(gl_TexCoord[0].st, 0.0);",
		"	float a = pSrc.a*opacity;",
		"	return (rand < a || a >= 1.0) ? vec4(pSrc.rgb/pSrc.a, 1.0) : pDst;",
		"}",

		/**
		 * ダイナミックディザ合成
		 */
		"vec4 blend_dancing_dissolve(vec4 pDst, vec4 pSrc, float opacity, float time)",
		"{",
		"	float rand = rand1(gl_TexCoord[0].st, time);",
		"	float a = pSrc.a*opacity;",
		"	return (rand < a || a >= 1.0) ? vec4(pSrc.rgb/pSrc.a, 1.0) : pDst;",
		"}",

		/**
		 * 比較(暗)
		 */
		"vec4 blend_darken(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	return vec4(min(uSrc3, pDst3 + min(uSrc3, cda))*sa + pDst3*csa, a);",
		"}",

		/**
		 * 乗算
		 */
		"vec4 blend_multiply(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 pSrc3 = pSrc.rgb;",
		"",
//		"	return vec4(uSrc3*(pDst3 + cda)*sa + pDst3*csa, a);",		// 数式的にはこっちだけど、等価な下の式の方が計算量が少ない。
		"	return vec4(pSrc3*(pDst3 + cda)*opacity + pDst3*csa, a);",
		"}",

		/**
		 * 焼き込みカラー
		 */
		"vec4 blend_color_burn(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	float r = uSrc3.r+uDst3.r<1.0 ? 0.0 : uSrc3.r>0.0 ? 1.0-(1.0-uDst3.r)/uSrc3.r : 1.0;",
		"	float g = uSrc3.g+uDst3.g<1.0 ? 0.0 : uSrc3.g>0.0 ? 1.0-(1.0-uDst3.g)/uSrc3.g : 1.0;",
		"	float b = uSrc3.b+uDst3.b<1.0 ? 0.0 : uSrc3.b>0.0 ? 1.0-(1.0-uDst3.b)/uSrc3.b : 1.0;",
		"",
		"	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * 焼き込みリニア
		 */
		"vec4 blend_linear_burn(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	return vec4((max(uSrc3+uDst3-1.0, 0.0)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * カラー比較(暗)
		 */
		"vec4 blend_darker_color(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	float dy = dot(uDst3, lumaVec);",
		"	float sy = dot(uSrc3, lumaVec);",
		"",
		"	return vec4(((dy < sy ? uDst3 : uSrc3)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * 加算
		 */
		"vec4 blend_add(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float a = pSrc.a*opacity + pDst.a*(1.0 - pSrc.a*opacity);",
		"	return vec4(pSrc.rgb*opacity + pDst.rgb, a);",
		"}",

		/**
		 * 比較(明)
		 */
		"vec4 blend_lighten(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
//		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	return vec4(max(uSrc3, pDst3)*sa + pDst3*csa, a);",
		"}",

		/**
		 * スクリーン
		 */
		"vec4 blend_screen(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	return vec4(((1.0-(1.0-uDst3)*(1.0-uSrc3))*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * 覆い焼きカラー
		 */
		"vec4 blend_color_dodge(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	float r = uSrc3.r+uDst3.r>1.0 ? 1.0 : uDst3.r>0.0 ? uDst3.r/(1.0-uSrc3.r) : 0.0;",
		"	float g = uSrc3.g+uDst3.g>1.0 ? 1.0 : uDst3.g>0.0 ? uDst3.g/(1.0-uSrc3.g) : 0.0;",
		"	float b = uSrc3.b+uDst3.b>1.0 ? 1.0 : uDst3.b>0.0 ? uDst3.b/(1.0-uSrc3.b) : 0.0;",
		"",
		"	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * 覆い焼きリニア
		 */
		"vec4 blend_linear_dodge(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	return vec4((min(uSrc3+uDst3, 1.0)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * カラー比較(明)
		 */
		"vec4 blend_lighter_color(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	float dy = dot(uDst3, lumaVec);",
		"	float sy = dot(uSrc3, lumaVec);",
		"",
		"	return vec4(((dy > sy ? uDst3 : uSrc3)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * オーバーレイ
		 */
		"vec4 blend_overlay(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	float r = uDst3.r<0.5 ? 2.0*uSrc3.r*uDst3.r : 1.0-2.0*(1.0-uSrc3.r)*(1.0-uDst3.r);",
		"	float g = uDst3.g<0.5 ? 2.0*uSrc3.g*uDst3.g : 1.0-2.0*(1.0-uSrc3.g)*(1.0-uDst3.g);",
		"	float b = uDst3.b<0.5 ? 2.0*uSrc3.b*uDst3.b : 1.0-2.0*(1.0-uSrc3.b)*(1.0-uDst3.b);",
		"",
		"	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * ソフトライト
		 */
		"vec4 blend_soft_light(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",

		// TODO AEと完全には一致しない。

		"	float r = uSrc3.r<0.5           ? uDst3.r+(uDst3.r-uDst3.r*uDst3.r)*(2.0*uSrc3.r-1.0) :",
		"			  uDst3.r<=(32.0/255.0) ? uDst3.r+(uDst3.r-uDst3.r*uDst3.r)*(2.0*uSrc3.r-1.0)*(3.0-8.0*uDst3.r) :",
		"			                          uDst3.r+(sqrt(uDst3.r)-uDst3.r)*(2.0*uSrc3.r-1.0);",
		"	float g = uSrc3.g<0.5           ? uDst3.g+(uDst3.g-uDst3.g*uDst3.g)*(2.0*uSrc3.g-1.0) :",
		"			  uDst3.g<=(32.0/255.0) ? uDst3.g+(uDst3.g-uDst3.g*uDst3.g)*(2.0*uSrc3.g-1.0)*(3.0-8.0*uDst3.g) :",
		"			                          uDst3.g+(sqrt(uDst3.g)-uDst3.g)*(2.0*uSrc3.g-1.0);",
		"	float b = uSrc3.b<0.5           ? uDst3.b+(uDst3.b-uDst3.b*uDst3.b)*(2.0*uSrc3.b-1.0) :",
		"			  uDst3.b<=(32.0/255.0) ? uDst3.b+(uDst3.b-uDst3.b*uDst3.b)*(2.0*uSrc3.b-1.0)*(3.0-8.0*uDst3.b) :",
		"			                          uDst3.b+(sqrt(uDst3.b)-uDst3.b)*(2.0*uSrc3.b-1.0);",

//		"	float r = uSrc3.r<0.5 ? 2.0*uDst3.r*uSrc3.r+uDst3.r*uDst3.r*(1.0-2.0*uSrc3.r) : sqrt(uDst3.r)*(2.0*uSrc3.r-1.0)+2.0*uDst3.r*(1.0-uSrc3.r);",
//		"	float g = uSrc3.g<0.5 ? 2.0*uDst3.g*uSrc3.g+uDst3.g*uDst3.g*(1.0-2.0*uSrc3.g) : sqrt(uDst3.g)*(2.0*uSrc3.g-1.0)+2.0*uDst3.g*(1.0-uSrc3.g);",
//		"	float b = uSrc3.b<0.5 ? 2.0*uDst3.b*uSrc3.b+uDst3.b*uDst3.b*(1.0-2.0*uSrc3.b) : sqrt(uDst3.b)*(2.0*uSrc3.b-1.0)+2.0*uDst3.b*(1.0-uSrc3.b);",

		// AEではこの成分が加えられているようである。これは上の式でuDst3=0.5のもの。
		"	float r2 = uSrc3.r<0.5 ? 0.5*uSrc3.r+0.25 : 0.414214*uSrc3.r+0.292893;",
		"	float g2 = uSrc3.g<0.5 ? 0.5*uSrc3.g+0.25 : 0.414214*uSrc3.g+0.292893;",
		"	float b2 = uSrc3.b<0.5 ? 0.5*uSrc3.b+0.25 : 0.414214*uSrc3.b+0.292893;",

		"",
		"	return vec4((vec3(r, g, b)*da + vec3(r2, g2, b2)*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * ハードライト
		 */
		"vec4 blend_hard_light(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	float r = uSrc3.r<0.5 ? 2.0*uDst3.r*uSrc3.r : 1.0-2.0*(1.0-uDst3.r)*(1.0-uSrc3.r);",
		"	float g = uSrc3.g<0.5 ? 2.0*uDst3.g*uSrc3.g : 1.0-2.0*(1.0-uDst3.g)*(1.0-uSrc3.g);",
		"	float b = uSrc3.b<0.5 ? 2.0*uDst3.b*uSrc3.b : 1.0-2.0*(1.0-uDst3.b)*(1.0-uSrc3.b);",
		"",
		"	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * リニアライト
		 */
		"vec4 blend_linear_light(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	float r = uSrc3.r<0.5 ? (uDst3.r<1.0-uSrc3.r*2.0 ? 0.0 : uSrc3.r*2.0+uDst3.r-1.0)",
		"						  : (uDst3.r<2.0-uSrc3.r*2.0 ? uSrc3.r*2.0+uDst3.r-1.0 : 1.0);",
		"	float g = uSrc3.g<0.5 ? (uDst3.g<1.0-uSrc3.g*2.0 ? 0.0 : uSrc3.g*2.0+uDst3.g-1.0)",
		"						  : (uDst3.g<2.0-uSrc3.g*2.0 ? uSrc3.g*2.0+uDst3.g-1.0 : 1.0);",
		"	float b = uSrc3.b<0.5 ? (uDst3.b<1.0-uSrc3.b*2.0 ? 0.0 : uSrc3.b*2.0+uDst3.b-1.0)",
		"						  : (uDst3.b<2.0-uSrc3.b*2.0 ? uSrc3.b*2.0+uDst3.b-1.0 : 1.0);",
		"",
		"	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * ビビッドライト
		 */
		"vec4 blend_vivid_light(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	float r = uSrc3.r<0.5 ? (uDst3.r<=1.0-uSrc3.r*2.0 ? 0.0 : (uDst3.r-(1.0-uSrc3.r*2.0))/(uSrc3.r*2.0))",
		"						  : (uDst3.r< 2.0-uSrc3.r*2.0 ? uDst3.r/(2.0-uSrc3.r*2.0) : 1.0);",
		"	float g = uSrc3.g<0.5 ? (uDst3.g<=1.0-uSrc3.g*2.0 ? 0.0 : (uDst3.g-(1.0-uSrc3.g*2.0))/(uSrc3.g*2.0))",
		"						  : (uDst3.g< 2.0-uSrc3.g*2.0 ? uDst3.g/(2.0-uSrc3.g*2.0) : 1.0);",
		"	float b = uSrc3.b<0.5 ? (uDst3.b<=1.0-uSrc3.b*2.0 ? 0.0 : (uDst3.b-(1.0-uSrc3.b*2.0))/(uSrc3.b*2.0))",
		"						  : (uDst3.b< 2.0-uSrc3.b*2.0 ? uDst3.b/(2.0-uSrc3.b*2.0) : 1.0);",
		"",
		"	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * ピンライト
		 */
		"vec4 blend_pin_light(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	float r = uSrc3.r<0.5 ? min(uSrc3.r*2.0, uDst3.r) : max(uSrc3.r*2.0-1.0, uDst3.r);",
		"	float g = uSrc3.g<0.5 ? min(uSrc3.g*2.0, uDst3.g) : max(uSrc3.g*2.0-1.0, uDst3.g);",
		"	float b = uSrc3.b<0.5 ? min(uSrc3.b*2.0, uDst3.b) : max(uSrc3.b*2.0-1.0, uDst3.b);",
		"",
		"	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * ハードミックス
		 */
		"vec4 blend_hard_mix(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	float r = (uSrc3.r+uDst3.r>1.0 || uDst3.r >= 1.0) ? 1.0 : 0.0;",
		"	float g = (uSrc3.g+uDst3.g>1.0 || uDst3.g >= 1.0) ? 1.0 : 0.0;",
		"	float b = (uSrc3.b+uDst3.b>1.0 || uDst3.b >= 1.0) ? 1.0 : 0.0;",
		"",
		"	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * 差
		 */
		"vec4 blend_difference(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	return vec4((abs(uSrc3-uDst3)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * 除外
		 */
		"vec4 blend_exclusion(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	float r = (1.0-uDst3.r)*uSrc3.r+(1.0-uSrc3.r)*uDst3.r;",
		"	float g = (1.0-uDst3.g)*uSrc3.g+(1.0-uSrc3.g)*uDst3.g;",
		"	float b = (1.0-uDst3.b)*uSrc3.b+(1.0-uSrc3.b)*uDst3.b;",
		"",
		"	return vec4((vec3(r, g, b)*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * RGBからHSLへ変換する関数
		 */
		"vec3 rgb2hsl(vec3 rgb)",
		"{",
		"	float min = min(min(rgb.r, rgb.g), rgb.b);",
		"	float max = max(max(rgb.r, rgb.g), rgb.b);",
		"	float mmDelta = max - min;",
		"	float mmSum = max + min;",
		"	float luma = mmSum * 0.5;",
		"	float hue, sat;",
		"",
		"	if (mmDelta == 0.0) {",
		"		hue = 0.0;",
		"		sat = 0.0;",
		"	} else {",
		"		sat = (luma < 0.5) ? mmDelta / mmSum",
		"						   : mmDelta / (2.0 - mmSum);",
		"",
		"		vec3 rgbDelta = ((max-rgb)/6.0 + max*0.5) / max;",
		"",
		"		hue = rgb.r == max ? rgbDelta.b - rgbDelta.g :",
		"			  rgb.g == max ? rgbDelta.r - rgbDelta.b + 1.0/3.0 :",
		"			                 rgbDelta.g - rgbDelta.r + 2.0/3.0;",
		"",
		"		if (hue < 0.0) hue += 1.0;",
		"		else if (hue > 1.0) hue -= 1.0;",
		"	}",
		"	return vec3(hue, sat, luma);",
		"}",

		/**
		 * 色相からRGBの各成分に変換する関数 (hsl2rgbから呼び出す)
		 */
		"float hue2rgb(float t1, float t2, float hue)",
		"{",
		"	if (hue < 0.0) hue += 1.0;",
		"	else if (hue > 1.0) hue -= 1.0;",
		"",
		"	return (hue*6.0 < 1.0) ? t1 + (t2-t1)*6.0*hue :",
		"		   (hue*2.0 < 1.0) ? t2 :",
		"		   (hue*3.0 < 2.0) ? t1 + (t2-t1)*(2.0/3.0-hue)*6.0 :",
		"		                     t1;",
		"}",

		/**
		 * HSLからRGBへ変換する関数
		 */
		"vec3 hsl2rgb(vec3 hsl)",
		"{",
		"	float hue = hsl.x;",
		"	float sat = hsl.y;",
		"	float luma = hsl.z;",
		"	vec3 rgb;",
		"",
		"	if (sat == 0.0) {",
		"		rgb = vec3(luma);",
		"	} else {",
		"		float t2 = (luma < 0.5) ? luma*(1.0+sat) : luma+sat-luma*sat;",
		"		float t1 = luma*2.0 - t2;",
		"",
		"		rgb = vec3(",
		"				hue2rgb(t1, t2, hue+1.0/3.0),",
		"				hue2rgb(t1, t2, hue),",
		"				hue2rgb(t1, t2, hue-1.0/3.0));",
		"	}",
		"	return rgb;",
		"}",

		/**
		 * RGBからLabに変換する関数
		 */
		"vec3 rgb2lab(vec3 rgb)",
		"{",
		"	rgb.r = (rgb.r > 0.04045) ? pow((rgb.r+0.055)/1.055, 2.4) : rgb.r/12.92;",
		"	rgb.g = (rgb.g > 0.04045) ? pow((rgb.g+0.055)/1.055, 2.4) : rgb.g/12.92;",
		"	rgb.b = (rgb.b > 0.04045) ? pow((rgb.b+0.055)/1.055, 2.4) : rgb.b/12.92;",
		"",
		"	float x = dot(rgb, vec3(41.24, 35.76, 18.05)) /  95.047;",
		"	float y = dot(rgb, vec3(21.26, 71.52,  7.22)) / 100.000;",
		"	float z = dot(rgb, vec3( 1.93, 11.92, 95.05)) / 108.883;",
		"",
		"	x = (x > 0.008856) ? pow(x, 1.0/3.0) : (7.787*x)+(16.0/116.0);",
		"	y = (y > 0.008856) ? pow(y, 1.0/3.0) : (7.787*y)+(16.0/116.0);",
		"	z = (z > 0.008856) ? pow(z, 1.0/3.0) : (7.787*z)+(16.0/116.0);",
		"",
		"	return vec3(116.0*y-16.0, 500.0*(x-y), 200.0*(y-z));",
		"}",

		/**
		 * LabからRGBに変換する関数
		 */
		"vec3 lab2rgb(vec3 lab)",
		"{",
		"	float y = (lab.s + 16.0) / 116.0;",
		"	float x = lab.t / 500.0 + y;",
		"	float z = y - lab.p / 200.0;",
		"",
		"	vec3 xyz = pow(vec3(x,y,z), vec3(3.0));",
		"",
		"	y = (xyz.y > 0.008856) ? xyz.y : (y - 16.0 / 116.0) / 7.787;",
		"	x = (xyz.x > 0.008856) ? xyz.x : (x - 16.0 / 116.0) / 7.787;",
		"	z = (xyz.z > 0.008856) ? xyz.z : (z - 16.0 / 116.0) / 7.787;",
		"",
		"	xyz = vec3(x,y,z) * vec3(0.95047, 1.00000, 1.08883);",
		"",
		"	float r = dot(xyz, vec3( 3.2406, -1.5372, -0.4986));",
		"	float g = dot(xyz, vec3(-0.9689,  1.8758,  0.0415));",
		"	float b = dot(xyz, vec3( 0.0557, -0.2040,  1.0570));",
		"",
		"	r = (r > 0.0031308) ? 1.055 * pow(r, 1.0/2.4) - 0.055 : 12.92 * r;",
		"	g = (g > 0.0031308) ? 1.055 * pow(g, 1.0/2.4) - 0.055 : 12.92 * g;",
		"	b = (b > 0.0031308) ? 1.055 * pow(b, 1.0/2.4) - 0.055 : 12.92 * b;",
		"",
		"	return vec3(r,g,b);",
		"}",

		/**
		 * 色相
		 */
		"vec4 blend_hue(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	vec3 newDst3 = (uSrc3.r == uSrc3.g && uSrc3.r == uSrc3.b)",
		"					? uDst3 : hsl2rgb(vec3(rgb2hsl(uSrc3).x, rgb2hsl(uDst3).yz));",
		"",
		"	return vec4((newDst3*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * 彩度
		 */
		"vec4 blend_saturation(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	vec3 newDst3;",
		"",
		"	vec3 dstLab = rgb2lab(uDst3);",
		"	if (all(equal(dstLab.tp, vec2(0.0)))) {",
		"		newDst3 = uDst3;",
		"	} else {",
		"		vec3 srcLab = rgb2lab(uSrc3);",
		"		float ratio = distance(srcLab.tp, vec2(0.0)) / distance(dstLab.tp, vec2(0.0));",
		"		newDst3 = lab2rgb(vec3(dstLab.s, dstLab.tp*ratio));",
		"	}",
		"",
		"	return vec4((newDst3*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * カラー
		 */
		"vec4 blend_color(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	vec3 dstLab = rgb2lab(uDst3);",
		"	vec3 srcLab = rgb2lab(uSrc3);",
		"	vec3 newDst3 = lab2rgb(vec3(dstLab.s, srcLab.tp));",
		"",
		"	return vec4((newDst3*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * 輝度
		 */
		"vec4 blend_luminosity(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	float da = pDst.a;",
		"	float sa = pSrc.a*opacity;",
		"	float cda = 1.0 - da;",
		"	float csa = 1.0 - sa;",
		"	float a = sa + da*csa;",
		"",
		"	vec3 pDst3 = pDst.rgb;",
		"	vec3 uDst3 = (pDst.a != 0.0) ? pDst.rgb/pDst.a : vec3(0.0);",
		"	vec3 uSrc3 = (pSrc.a != 0.0) ? pSrc.rgb/pSrc.a : vec3(0.0);",
		"",
		"	vec3 dstLab = rgb2lab(uDst3);",
		"	vec3 srcLab = rgb2lab(uSrc3);",
		"	vec3 newDst3 = lab2rgb(vec3(srcLab.s, dstLab.tp));",
		"",
		"	return vec4((newDst3*da + uSrc3*cda)*sa + pDst3*csa, a);",
		"}",

		/**
		 * ステンシルアルファ
		 */
		"vec4 blend_stencil_alpha(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	return pDst * (pSrc.a * opacity);",
		"}",

		/**
		 * ステンシルルミナンス
		 */
		"vec4 blend_stencil_luma(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	return pDst * (dot(pSrc.rgb, lumaVec) * opacity);",
		"}",

		/**
		 * シルエットアルファ
		 */
		"vec4 blend_silhouette_alpha(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	return pDst * (1.0 - pSrc.a * opacity);",
		"}",

		/**
		 * シルエットルミナンス
		 */
		"vec4 blend_silhouette_luma(vec4 pDst, vec4 pSrc, float opacity)",
		"{",
		"	return pDst * (1.0 - dot(pSrc.rgb, lumaVec) * opacity);",
		"}",

		/**
		 * 交差グループ用ブレンド関数
		 */
		"vec4 blend(int mode, vec4 pDst, vec4 pSrc, float opacity, float time)",
		"{",
		"	return (mode == 0) ? blend_normal(pDst, pSrc, opacity) :",
		"		   (mode == 1) ? blend_dissolve(pDst, pSrc, opacity) :",
		"		   (mode == 2) ? blend_dancing_dissolve(pDst, pSrc, opacity, time) :",

		"		   (mode == 3) ? blend_darken(pDst, pSrc, opacity) :",
		"		   (mode == 4) ? blend_multiply(pDst, pSrc, opacity) :",
		"		   (mode == 5) ? blend_color_burn(pDst, pSrc, opacity) :",
		"		   (mode == 6) ? blend_linear_burn(pDst, pSrc, opacity) :",
		"		   (mode == 7) ? blend_darker_color(pDst, pSrc, opacity) :",

		"		   (mode == 8) ? blend_add(pDst, pSrc, opacity) :",
		"		   (mode == 9) ? blend_lighten(pDst, pSrc, opacity) :",
		"		   (mode == 10) ? blend_screen(pDst, pSrc, opacity) :",
		"		   (mode == 11) ? blend_color_dodge(pDst, pSrc, opacity) :",
		"		   (mode == 12) ? blend_linear_dodge(pDst, pSrc, opacity) :",
		"		   (mode == 13) ? blend_lighter_color(pDst, pSrc, opacity) :",

		"		   (mode == 14) ? blend_overlay(pDst, pSrc, opacity) :",
		"		   (mode == 15) ? blend_soft_light(pDst, pSrc, opacity) :",
		"		   (mode == 16) ? blend_hard_light(pDst, pSrc, opacity) :",
		"		   (mode == 17) ? blend_linear_light(pDst, pSrc, opacity) :",
		"		   (mode == 18) ? blend_vivid_light(pDst, pSrc, opacity) :",
		"		   (mode == 19) ? blend_pin_light(pDst, pSrc, opacity) :",
		"		   (mode == 20) ? blend_hard_mix(pDst, pSrc, opacity) :",

		"		   (mode == 21) ? blend_difference(pDst, pSrc, opacity) :",
		"		   (mode == 22) ? blend_exclusion(pDst, pSrc, opacity) :",

		"		   (mode == 23) ? blend_hue(pDst, pSrc, opacity) :",
		"		   (mode == 24) ? blend_saturation(pDst, pSrc, opacity) :",
		"		   (mode == 25) ? blend_color(pDst, pSrc, opacity) :",
		"		   (mode == 26) ? blend_luminosity(pDst, pSrc, opacity) :",

// TODO ステンシルアルファとステンシルルミナンスは、次のふたつの理由で3Dレイヤーでは正しく動作しない。
//      1. レイヤー矩形の外側はアルファ値がゼロなので、ノンゼロアルファ(nza)のテストで落とされてしまう（これはシェーダの修正だけで解決可能）
//      2. レイヤー矩形の外側はデプス値が最大なので、ソートしたときに一番奥に配置されてしまう（これはシェーダの修正だけでは解決できない）
//
//		"		   (mode == 27) ? blend_stencil_alpha(pDst, pSrc, opacity) :",
//		"		   (mode == 28) ? blend_stencil_luma(pDst, pSrc, opacity) :",
		"		   (mode == 29) ? blend_silhouette_alpha(pDst, pSrc, opacity) :",
		"		   (mode == 30) ? blend_silhouette_luma(pDst, pSrc, opacity) :",

		"		   blend_normal(pDst, pSrc, opacity);",
		"}"
	};

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] NORMAL = createProgramSource("normal");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach={"blend_functions"})
	public static final String[] DISSOLVE = createProgramSource("dissolve");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach={"blend_functions"})
	public static final String[] DANCING_DISSOLVE = createProgramSource("dancing_dissolve", true);

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] DARKEN = createProgramSource("darken");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] MULTIPLY = createProgramSource("multiply");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] COLOR_BURN = createProgramSource("color_burn");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] LINEAR_BURN = createProgramSource("linear_burn");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] DARKER_COLOR = createProgramSource("darker_color");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] ADD = createProgramSource("add");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] LIGHTEN = createProgramSource("lighten");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] SCREEN = createProgramSource("screen");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] COLOR_DODGE = createProgramSource("color_dodge");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] LINEAR_DODGE = createProgramSource("linear_dodge");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] LIGHTER_COLOR = createProgramSource("lighter_color");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] OVERLAY = createProgramSource("overlay");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] SOFT_LIGHT = createProgramSource("soft_light");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] HARD_LIGHT = createProgramSource("hard_light");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] LINEAR_LIGHT = createProgramSource("linear_light");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] VIVID_LIGHT = createProgramSource("vivid_light");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] PIN_LIGHT = createProgramSource("pin_light");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] HARD_MIX = createProgramSource("hard_mix");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] DIFFERENCE = createProgramSource("difference");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] EXCLUSION = createProgramSource("exclusion");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] HUE = createProgramSource("hue");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] SATURATION = createProgramSource("saturation");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] COLOR = createProgramSource("color");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] LUMINOSITY = createProgramSource("luminosity");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] STENCIL_ALPHA = createProgramSource("stencil_alpha");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] STENCIL_LUMA = createProgramSource("stencil_luma");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] SILHOUETTE_ALPHA = createProgramSource("silhouette_alpha");

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] SILHOUETTE_LUMA = createProgramSource("silhouette_luma");



	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="blend_functions")
	public static final String[] INTERSECTING_BLEND = {
		// 以下のような変な書き方をしているのは、普通に書いたらMacで動かなかったから。
		// (sampler2DRectの配列をuniform変数にしても、配列のインデックスを定数にしないと動かない)

		// あと、シェーダ内で配列を使うと遅くなるようなので、ord0-ord6は配列にしていない。
		// TODO texture2DRectでサンプリングした値は変数に保持しておいた方がいいかも。
		//      ただし配列ではなくord0-ord6と同じような形で。

		"const int maxLayers = 7;",
		"",
		"uniform int nLayers;",
		"uniform sampler2DRect texDst;",
		"uniform sampler2DRect texSrc0;",
		"uniform sampler2DRect texSrc1;",
		"uniform sampler2DRect texSrc2;",
		"uniform sampler2DRect texSrc3;",
		"uniform sampler2DRect texSrc4;",
		"uniform sampler2DRect texSrc5;",
		"uniform sampler2DRect texSrc6;",
		"uniform sampler2DRect texDep0;",
		"uniform sampler2DRect texDep1;",
		"uniform sampler2DRect texDep2;",
		"uniform sampler2DRect texDep3;",
		"uniform sampler2DRect texDep4;",
		"uniform sampler2DRect texDep5;",
		"uniform sampler2DRect texDep6;",
		"uniform int blendMode[maxLayers];",
		"uniform float opacity[maxLayers];",
		"uniform float time;",
		"",
		"",
		"vec2 texCoord = gl_TexCoord[0].st;",
		"",
		"vec4 getSrc(int i) {",
		"	return ",
		"		(i == 0) ? texture2DRect(texSrc0, texCoord) : ",
		"		(i == 1) ? texture2DRect(texSrc1, texCoord) : ",
		"		(i == 2) ? texture2DRect(texSrc2, texCoord) : ",
		"		(i == 3) ? texture2DRect(texSrc3, texCoord) : ",
		"		(i == 4) ? texture2DRect(texSrc4, texCoord) : ",
		"		(i == 5) ? texture2DRect(texSrc5, texCoord) : ",
		"		(i == 6) ? texture2DRect(texSrc6, texCoord) : ",
		"		vec4(0.0);",
		"}",
		"",
		"float getDep(int i) {",
		"	return ",
		"		(i == 0) ? texture2DRect(texDep0, texCoord).r : ",
		"		(i == 1) ? texture2DRect(texDep1, texCoord).r : ",
		"		(i == 2) ? texture2DRect(texDep2, texCoord).r : ",
		"		(i == 3) ? texture2DRect(texDep3, texCoord).r : ",
		"		(i == 4) ? texture2DRect(texDep4, texCoord).r : ",
		"		(i == 5) ? texture2DRect(texDep5, texCoord).r : ",
		"		(i == 6) ? texture2DRect(texDep6, texCoord).r : ",
		"		-1.0;",
		"}",
		"",
		"int ord0 = 0;",
		"int ord1 = 1;",
		"int ord2 = 2;",
		"int ord3 = 3;",
		"int ord4 = 4;",
		"int ord5 = 5;",
		"int ord6 = 6;",
		"",
		"int getOrd(int i) {",
		"	return ",
		"		(i == 0) ? ord0 : ",
		"		(i == 1) ? ord1 : ",
		"		(i == 2) ? ord2 : ",
		"		(i == 3) ? ord3 : ",
		"		(i == 4) ? ord4 : ",
		"		(i == 5) ? ord5 : ",
		"		(i == 6) ? ord6 : ",
		"		-1;",
		"}",
		"",
		"void setOrd(int i, int ord) {",
		"	if (i == 0) ord0 = ord; else ",
		"	if (i == 1) ord1 = ord; else ",
		"	if (i == 2) ord2 = ord; else ",
		"	if (i == 3) ord3 = ord; else ",
		"	if (i == 4) ord4 = ord; else ",
		"	if (i == 5) ord5 = ord; else ",
		"	if (i == 6) ord6 = ord; else ",
		"	{}",
		"}",
		"",
		"",
		"vec4 blend(int mode, vec4 pDst, vec4 pSrc, float opacity, float time);",
		"",
		"void main(void)",
		"{",
		"	vec4 dst = texture2DRect(texDst, texCoord);",
		"",
		// texSrcNのアルファ値が全てゼロ(nzaCount == 0)の場合、texDstの値をそのまま使えばよい。
		// texSrcNのうちひとつだけがゼロ以外のアルファを持つ(nzaCount == 1)場合、そのテクスチャだけをtexDstとブレンドすればよい。
		"	int nzaCount = 0;",
		"	int nzaLayer;",
		"	vec4 nzaColor;",
		"	for (int i = 0; i < nLayers; ++i) {",
		"		vec4 s = getSrc(i);",
		"		if (s.a > 0.0) {",
		"			if (++nzaCount > 1) {",
		"				break;",
		"			}",
		"			nzaLayer = i;",
		"			nzaColor = s;",
		"		}",
		"	}",
		"	if (nzaCount == 1) {",
		"		dst = vec4(dst.rgb*dst.a, dst.a);",
		"		dst = blend(blendMode[nzaLayer], dst, nzaColor, opacity[nzaLayer], time);",
		"		dst = (dst.a != 0.0) ? vec4(dst.rgb/dst.a, dst.a) : vec4(0.0);",
		"",
		"	} else if (nzaCount > 1) {",
		"		dst = vec4(dst.rgb*dst.a, dst.a);",
		"		",
		"		for (int i = nLayers - 2; i >= 0; --i) {",
		"			int ord = getOrd(i);",
		"			float dep = getDep(ord);",
		"			int j, ord2;",
		"			for (j = i + 1; j < nLayers && getDep(ord2 = getOrd(j)) > dep; ++j) {",
		"				setOrd(j - 1, ord2);",
		"			}",
		"			setOrd(j - 1, ord);",
		"		}",
		"		",
		"		for (int i = 0; i < nLayers; ++i) {",
		"			int ord = getOrd(i);",
		"			dst = blend(blendMode[ord], dst, getSrc(ord), opacity[ord], time);",
		"		}",
		"		dst = (dst.a != 0.0) ? vec4(dst.rgb/dst.a, dst.a) : vec4(0.0);",
		"	}",
		"",
		"	gl_FragColor = dst;",
		"}"
	};


	private BlendModeShaders() { }

}
