/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.output;

import java.io.File;
import java.util.Arrays;

import javax.media.opengl.GL;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.services.AudioRenderContext;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

public abstract class PushSourceOutput {

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private AudioRenderContext _arContext;

	@Inject
	private AudioRenderSupport _arSupport;


	public void doOutput(
			final Composition comp, final AudioMode audioMode, File file, ProgressMonitor progMon) {

		initialize(comp, audioMode, file);

		final ProgressMonitor monitor = (progMon != null) ? progMon : ProgressMonitor.NULL;

		Thread audioThread = new Thread() { public void run() { audioLoop(comp, audioMode, monitor); } };
		audioThread.start();

		try {
			videoLoop(comp, monitor);
		} finally {
			// TODO writeAudioでブロックされているときは永久に待ち続けるかもしれない。
			try { audioThread.join(); } catch (InterruptedException e) { }

			finish();
		}
	}

	protected abstract void initialize(Composition comp, AudioMode audioMode, File file);

	protected abstract void finish();


	protected void videoLoop(Composition comp, ProgressMonitor monitor) {
		Time frameDuration = comp.getFrameDuration();
		long numFrames = comp.getDuration().toFrameNumber(frameDuration);

		monitor.beginTask((int)numFrames);

		_vrContext.activate();
		try {
			GL gl = _vrContext.getGL();

			comp.prepareExpression(_vrContext.createInitialExpressionScope(comp));

			for (long i = 0; i < numFrames && !monitor.isCanceled(); ++i) {
				Time time = Time.fromFrameNumber(i, frameDuration);
				_vrContext.reset();
				_vrContext.setRenderResolution(RenderResolution.FULL);
				_vrContext.setTime(time);

				VideoBuffer vb = comp.renderVideoFrame();
				try {
					// TODO アルファ無し／ストレートアルファ／乗算済みアルファを選択できるよにする
					_vrSupport.premultiply(vb);

					gl.glFinish();

					writeVideo(time, vb);
				} finally {
					vb.dispose();
				}

				monitor.currentWork((int)i);
			}
		} finally {
			_vrContext.deactivate();
		}

		monitor.done();
	}

	protected abstract void writeVideo(Time time, VideoBuffer vb);


	protected void audioLoop(Composition comp, AudioMode audioMode, ProgressMonitor monitor) {
		long totalFrames = comp.getDuration().toFrameNumber(audioMode.sampleDuration);

		// １回あたりの処理量を１秒分のフレーム数（＝サンプルレートの値と同じ数）とする。
		int chunkFrames = audioMode.sampleRate;

		_arContext.activate();
		try {
			comp.prepareExpression(_arContext.createInitialExpressionScope(comp));

			for (long i = 0; i < totalFrames && !monitor.isCanceled(); ) {
				Time time = Time.fromFrameNumber(i, audioMode.sampleDuration);
				_arContext.reset();
				_arContext.setAudioMode(audioMode);
				_arContext.setTime(time);
				_arContext.setFrameCount((int) Math.min(chunkFrames, totalFrames-i));
				_arContext.setEvaluationResolution(audioMode.sampleRate/100);

				AudioBuffer ab = comp.renderAudioChunk();
				try {
					if (ab == null) {
						ab = createAudioBufferClear();
					}
					writeAudio(time, ab);
					i+= ab.getFrameCount();
				} finally {
					ab.dispose();
				}
			}
		} finally {
			_arContext.deactivate();
		}
	}

	protected abstract void writeAudio(Time time, AudioBuffer ab);


	private AudioBuffer createAudioBufferClear() {
		// このメソッドは AudioRenderSupport にあってもいいかもしれない。
		// PullSourceOutputにも同じメソッドがあるし。

		AudioBuffer ab = _arSupport.createAudioBuffer();

		switch (ab.getAudioMode().dataType) {
			case SHORT:
				Arrays.fill((short[])ab.getData(), 0, ab.getDataLength(), (short)0);
				break;
			case INT:
				Arrays.fill((int[])ab.getData(), 0, ab.getDataLength(), 0);
				break;
			case FLOAT:
				Arrays.fill((float[])ab.getData(), 0, ab.getDataLength(), 0);
				break;
			default:
				throw new UnsupportedOperationException(
						"unsupported AudioMode.DataType: " + ab.getAudioMode().dataType);
		}

		return ab;
	}

}
