/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.output;

import java.io.File;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.AudioMode.DataType;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.VideoBuffer;

public class MacOSXQTMovieOutput extends PullSourceOutput {

	private static final Logger _logger = LoggerFactory.getLogger(MacOSXQTMovieOutput.class);


	private class VideoRequestWithByteBuffer extends VideoRequest {
		private final ByteBuffer byteBuffer;
		private VideoRequestWithByteBuffer(Time time, ByteBuffer byteBuffer) {
			super(time);
			this.byteBuffer = byteBuffer;
		}
	}

	private class AudioRequestWithByteBuffer extends AudioRequest {
		private final ByteBuffer byteBuffer;
		private AudioRequestWithByteBuffer(Time time, int frameCount, ByteBuffer byteBuffer) {
			super(time, frameCount);
			this.byteBuffer = byteBuffer;
		}
	}


	private byte[] _settings;


	public byte[] getSettings() {
		return _settings;
	}

	public void setSettings(byte[] settings) {
		_settings = settings;
	}

	@Override
	protected void doOutput(Composition comp, AudioMode audioMode, File file) {
		if (_settings == null) throw new IllegalStateException("settings is not set");

		Size2i size = comp.getSize();
		Time frameDuration = comp.getFrameDuration();
		Time duration = comp.getDuration();

		if (duration.timeScale != frameDuration.timeScale) {
			duration = Time.fromFrameNumber(duration.toFrameNumber(frameDuration), frameDuration);
		}

		int error = doOutput(_settings, file.getAbsolutePath(),
				size.width, size.height, frameDuration.timeScale, frameDuration.timeValue, duration.timeValue,
				audioMode.channels, audioMode.sampleRate, audioMode.sampleSize,
				(audioMode.dataType == DataType.FLOAT), duration.toFrameNumber(audioMode.sampleDuration));

		if (error != 0 && error != -128) {
			_logger.error("doOutput: error=" + error);
		}
	}

	@Override
	protected void finishVideoRequest(VideoRequest request, VideoBuffer vb) {
		copyVideoBufferToByteBuffer(vb, ((VideoRequestWithByteBuffer) request).byteBuffer);
	}

	@Override
	protected void finishAudioRequest(AudioRequest request, AudioBuffer ab) {
		copyAudioBufferToByteBuffer(ab, ((AudioRequestWithByteBuffer) request).byteBuffer);
	}

	protected void copyVideoBufferToByteBuffer(VideoBuffer vb, ByteBuffer bb) {
		// TODO VideoBufferから配列に一度コピーせず、直接ByteBufferに書き込めるようにする。

		Object array = vb.getArray();

		bb.order(ByteOrder.nativeOrder());

		if (array instanceof byte[]) {
			bb.put((byte[]) array, 0, bb.capacity());
			
		} else if (array instanceof short[]) {
			bb.asShortBuffer().put((short[]) array, 0, bb.capacity() / 2);

		} else if (array instanceof float[]) {
			bb.asFloatBuffer().put((float[]) array, 0, bb.capacity() / 4);

		} else {
			throw new IllegalArgumentException(
					"not array or unsupported array type: "
					+ array.getClass().getName());
		}

		bb.position(bb.capacity());
	}

	protected void copyAudioBufferToByteBuffer(AudioBuffer ab, ByteBuffer bb) {
		bb.order(ByteOrder.nativeOrder());

		switch (ab.getAudioMode().dataType) {
			case SHORT:
				bb.asShortBuffer().put((short[])ab.getData(), 0, ab.getDataLength());
				break;
			case INT:
				bb.asIntBuffer().put((int[])ab.getData(), 0, ab.getDataLength());
				break;
			case FLOAT:
				bb.asFloatBuffer().put((float[])ab.getData(), 0, ab.getDataLength());
				break;
			default:
				throw new IllegalArgumentException(
						"unsupported AudioMode: " + ab.getAudioMode());
		}

		bb.position(ab.getDataLengthInBytes());
	}


	@SuppressWarnings("unused")
	private int videoRequest(long timeValue, int timeScale, ByteBuffer buffer) {
		try {
			buffer.clear();
			postVideoRequest(new VideoRequestWithByteBuffer(new Time(timeValue, timeScale), buffer));
			return buffer.position();
		} catch (InterruptedException e) {
			_logger.warn("unexpected interruption", e);
			return -1;
		}
	}

	@SuppressWarnings("unused")
	private int audioRequest(long timeValue, int timeScale, ByteBuffer buffer) {
		// タイムスケールの値と同じだけのフレーム数(=1秒分)に必要なバッファをネイティブコード側で確保している。
		int frameCount = timeScale;

		try {
			buffer.clear();
			postAudioRequest(new AudioRequestWithByteBuffer(new Time(timeValue, timeScale), frameCount, buffer));
			return buffer.position();
		} catch (InterruptedException e) {
			_logger.warn("unexpected interruption", e);
			return -1;
		}
	}

	@SuppressWarnings("unused")
	private boolean updateProgress(short message, int percentDone) {
		if (isCanceled()) return false;

		switch (message) {
			case 0:
				beginTask(0x00010000);
				break;
			case 1:
				currentWork(percentDone);
				break;
			case 2:
				done();
				break;
		}

		return true;
	}


	public static byte[] doUserDialogAndGetSettings(byte[] defaultSettings) {
		byte[][] ioSettings = new byte[][] { defaultSettings };
		int error = doUserDialogAndGetSettings(ioSettings);
		switch (error) {
			case 0:
				return ioSettings[0];
			case -128:
				// userCanceledErr
				return null;
			default:
				_logger.error("doUserDialogAndGetSettings: error=" + error);
				return null;
		}
	}


	static { System.loadLibrary("QTMovieOutput"); }

	private static native int doUserDialogAndGetSettings(byte[][] ioSettings);

	private native int doOutput(
			byte[] settings, String filename,
			int vWidth, int vHeight, int vTimeScale, long vFrameDuration, long vDuration,
			int aChannels, int aSampleRate, int aSampleSize, boolean aFloat, long aDuration);

}
