/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import javax.media.opengl.GL;
import javax.media.opengl.GLContext;
import javax.media.opengl.glu.GLU;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.VideoRenderContext;

import com.google.inject.Inject;

public class VideoRenderContextImpl extends AbstractRenderContext implements VideoRenderContext {

	private static final Logger _logger = LoggerFactory.getLogger(VideoRenderContextImpl.class);

	@Inject
	private GLGlobal _glGlobal;


	private GLContext _glContext;

	private GLU _glu;

	private int _fboId;

	private RenderResolution _resolution;

	private ColorMode _colorMode;


	protected void finalize() throws Throwable {
		if (_glContext != null) {
			// TODO 解放忘れの場合どうする？
			//		finalize は別のスレッドから呼ばれるので、ここで deactivate するわけにはいかない。
			_logger.warn("finalizing a VideoRenderContextImpl object, but the object is not deactivated.");
		}
		super.finalize();
	}

	public void activate() {
		if (_glContext != null) {
			throw new IllegalStateException("already activated");
		}

		super.activate();

		synchronized (_glGlobal.getLockObject()) {
			_glContext = _glGlobal.createContext();
			_glContext.makeCurrent();
		}

		//_glContext.setGL(new DebugGL(_glContext.getGL()));
		GL gl = _glContext.getGL();

		_glu = new GLU();

		int[] fboId = new int[1];
		gl.glGenBuffers(1, fboId, 0);
		_fboId = fboId[0];

		gl.glBindFramebufferEXT(GL.GL_FRAMEBUFFER_EXT, _fboId);

		FontManagerProxy.createImpl();
	}

	public void deactivate() {
		FontManagerProxy.disposeImpl();

		if (_glContext != null) {
			GL gl = _glContext.getGL();

			gl.glBindFramebufferEXT(GL.GL_FRAMEBUFFER_EXT, 0);
			gl.glDeleteBuffers(1, new int[] { _fboId }, 0);
			_fboId = 0;

			synchronized (_glGlobal.getLockObject()) {
				_glContext.release();
				_glGlobal.destroyContext(_glContext);
			}

			_glContext = null;
			_glu = null;
		}

		_resolution = null;
		_colorMode = null;

		super.deactivate();
	}

	public void reset() {
		super.reset();
		_resolution = null;
		_colorMode = null;
	}

	public <T> T saveAndExecute(WrappedOperation<T> operation) {
		ColorMode colorMode = _colorMode;
		try {
			return super.saveAndExecute(operation);
		} finally {
			_colorMode = colorMode;
		}
	}

	public GL getGL() {
		return _glContext.getGL();
	}

	public GLU getGLU() {
		return _glu;
	}

	public RenderResolution getRenderResolution() {
		if (_resolution == null) {
			throw new IllegalStateException("renderResolution is not set.");
		}
		return _resolution;
	}

	public void setRenderResolution(RenderResolution resolution) {
		if (_resolution != null) {
			throw new IllegalStateException("renderResolution has already been set.");
		}
		_resolution = resolution;
	}

	public ColorMode getColorMode() {
		return _colorMode;
	}

	public void setColorMode(ColorMode colorMode) {
		_colorMode = colorMode;
	}

}
