/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.LinkedList;
import java.util.Queue;

import javax.media.opengl.GLCanvas;
import javax.media.opengl.GLCapabilities;
import javax.media.opengl.GLContext;
import javax.media.opengl.GLDrawableFactory;
import javax.media.opengl.GLPbuffer;
import javax.media.opengl.Threading;

import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.services.GLGlobal;

public class GLGlobalImpl implements GLGlobal {

	private final Object _lock = new Object();

	private final GLPbuffer _pbuffer;

	private final GLContext _sharedContext;

	private Queue<GLContext> _pool;

	private GLCapabilities _canvasCapabilities;


	public GLGlobalImpl() {
		GLCapabilities glcap = new GLCapabilities();
		glcap.setHardwareAccelerated(true);
		glcap.setDoubleBuffered(false);

		_pbuffer = GLDrawableFactory.getFactory().createGLPbuffer(glcap, null, 1, 1, null);
		_sharedContext = _pbuffer.getContext();

		_canvasCapabilities = glcap;
	}

	public Object getLockObject() {
		return _lock;
	}

	public GLContext getSharedContext() {
		return _sharedContext;
	}

	public void enterPoolMode(final int poolSize) {
		if (_pool != null) {
			throw new IllegalStateException("already in pool mode");
		}
		if (poolSize <= 0) {
			throw new IllegalArgumentException("poolSize must be one or greater");
		}

		_pool = new LinkedList<GLContext>();

		Threading.invokeOnOpenGLThread(new Runnable() {
			public void run() {
				for (int i = 0; i < poolSize; ++i) {
					GLContext context = _pbuffer.createContext(_sharedContext);
					context.makeCurrent();
					context.release();
					_pool.offer(context);
				}
			}
		});
	}

	public GLContext createContext() {
		if (_pool != null) {
			GLContext context = _pool.poll();
			if (context == null) {
				throw new JavieRuntimeException("no GLContext is available");
			}
			return context;
		}
		return _pbuffer.createContext(_sharedContext);
	}

	public void destroyContext(GLContext context) {
		if (_pool != null) {
			_pool.offer(context);
		} else {
			context.destroy();
		}
	}

	public GLCapabilities getCanvasCapabilities() {
		return (GLCapabilities) _canvasCapabilities.clone();
	}

	public void setCanvasCapabilities(GLCapabilities capabilities) {
		_canvasCapabilities = (GLCapabilities) capabilities.clone();
	}

	public GLCanvas createCanvas() {
		return new GLCanvas(_canvasCapabilities, null, _sharedContext, null);
	}

}
