/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.io.File;
import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.services.FontManager;
import ftgl.FTGL;
import ftgl.FTGLfont;

public class FontManagerImpl implements FontManager {

	private static final Logger logger = LoggerFactory.getLogger(FontManagerImpl.class);

	private static final int MAX_FONTS = 8;


	@SuppressWarnings("serial")
	private final Map<String, FTGLfont> _fontCache = new LinkedHashMap<String, FTGLfont>(16, 0.75f, true) {
		@Override
		protected boolean removeEldestEntry(Entry<String, FTGLfont> eldest) {
			if (size() > MAX_FONTS) {
				String key = eldest.getKey();
				FTGL.ftglDestroyFont(eldest.getValue());
				remove(key);
				logger.info("font removed from cache: " + key);
			}
			return false;
		}
	};

	void dispose() {
		for (Entry<String, FTGLfont> e : _fontCache.entrySet()) {
			FTGL.ftglDestroyFont(e.getValue());
		}
		_fontCache.clear();
	}

	public FTGLfont getPolygonFont(String filename, int faceIndex, int size) {
		return getFont(FontType.Polygon, filename, faceIndex, size);
	}

	public FTGLfont getOutlineFont(String filename, int faceIndex, int size) {
		return getFont(FontType.Outline, filename, faceIndex, size);
	}

	public FTGLfont getExtrudeFont(String filename, int faceIndex, int size) {
		return getFont(FontType.Extrude, filename, faceIndex, size);
	}

	public FTGLfont getTextureFont(String filename, int faceIndex, int size) {
		return getFont(FontType.Texture, filename, faceIndex, size);
	}

	public FTGLfont getBufferFont(String filename, int faceIndex, int size) {
		return getFont(FontType.Buffer, filename, faceIndex, size);
	}


	private enum FontType { Polygon, Outline, Extrude, Texture, Buffer }

	private FTGLfont getFont(FontType type, String filename, int faceIndex, int size) {
		try {
			filename = new File(filename).getCanonicalPath();
		} catch (IOException e) {
			logger.warn("cannot get canonical path for font: " + filename);
		}

		filename = String.format("%d,%s", faceIndex, filename);

		String key = String.format("%s,%s,%d", type.name(), filename, size);

		FTGLfont font = _fontCache.get(key);
		if (font != null) {
			return font;
		}

		switch (type) {
			case Polygon:
				font = FTGL.ftglCreatePolygonFont(filename);
				break;

			case Outline:
				font = FTGL.ftglCreateOutlineFont(filename);
				break;

			case Extrude:
				font = FTGL.ftglCreateExtrudeFont(filename);
				break;

			case Texture:
				font = FTGL.ftglCreateTextureFont(filename);
				break;

			case Buffer:
				font = FTGL.ftglCreateBufferFont(filename);
				break;

			default:
				throw new JavieRuntimeException("unsupported font type");
		}

		int error = FTGL.ftglGetFontError(font);
		if (error != 0) {
			FTGL.ftglDestroyFont(font);
			throw new JavieRuntimeException(
					String.format("ftglCreate%sFont failed: error=%d", type.name(), error));
		}
		FTGL.ftglSetFontCharMap(font, FTGL.FT_ENCODING_UNICODE);
		FTGL.ftglSetFontFaceSize(font, size, 72);

		_fontCache.put(key, font);
		return font;
	}

}
