/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.lang.reflect.Array;
import java.util.Arrays;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IObjectArray;
import ch.kuramo.javie.api.services.IArrayPools;

public class ArrayPoolsImpl implements IArrayPools {

	private static final Logger _logger = LoggerFactory.getLogger(ArrayPoolsImpl.class);


	private final ArrayPool<byte[]> _bytePool = new ArrayPool<byte[]>(byte.class);

	private final ArrayPool<short[]> _shortPool = new ArrayPool<short[]>(short.class);

	private final ArrayPool<int[]> _intPool = new ArrayPool<int[]>(int.class);

	private final ArrayPool<float[]> _floatPool = new ArrayPool<float[]>(float.class);

	private final ArrayPool<Object[]> _objectPool = new ArrayPool<Object[]>(Object.class);


	public IArray<byte[]> getByteArray(int length) {
		return new ByteArray(length);
	}

	public IArray<short[]> getShortArray(int length) {
		return new ShortArray(length);
	}

	public IArray<int[]> getIntArray(int length) {
		return new IntArray(length);
	}

	public IArray<float[]> getFloatArray(int length) {
		return new FloatArray(length);
	}

	public <T> IObjectArray<T> getObjectArray(int length) {
		return new ObjectArray<T>(length);
	}


	private abstract static class AbstractArray<T> implements IArray<T> {

		private final ArrayPool<T> pool;

		protected final int length;

		protected T array;


		protected AbstractArray(ArrayPool<T> pool, int length) {
			this.pool = pool;
			this.length = length;
			this.array = pool.get(length);
		}

		protected void finalize() throws Throwable {
			if (array != null) {
				_logger.warn(String.format("ARRAY IS NOT RELEASED: %s[%d/%d]",
						array.getClass().getComponentType().getName(), length, Array.getLength(array)));
				release();
			}
			super.finalize();
		}

		public T getArray() {
			return array;
		}

		public int getLength() {
			return length;
		}

		public void release() {
			if (array != null) {
				pool.put(array);
				array = null;
			}
		}
	}

	private class ByteArray extends AbstractArray<byte[]> {

		private ByteArray(int length) {
			super(_bytePool, length);
		}

		public void clear() {
			Arrays.fill(array, 0, length, (byte)0);
		}

	}

	private class ShortArray extends AbstractArray<short[]> {

		private ShortArray(int length) {
			super(_shortPool, length);
		}

		public void clear() {
			Arrays.fill(array, 0, length, (short)0);
		}

	}

	private class IntArray extends AbstractArray<int[]> {

		private IntArray(int length) {
			super(_intPool, length);
		}

		public void clear() {
			Arrays.fill(array, 0, length, 0);
		}

	}

	private class FloatArray extends AbstractArray<float[]> {

		private FloatArray(int length) {
			super(_floatPool, length);
		}

		public void clear() {
			Arrays.fill(array, 0, length, 0);
		}

	}

	private class ObjectArray<T> extends AbstractArray<Object[]> implements IObjectArray<T> {

		private ObjectArray(int length) {
			super(_objectPool, length);
		}

		public void clear() {
			Arrays.fill(array, 0, length, null);
		}

		public void release() {
			// オブジェクト配列の場合、クリアしておかないと配列内にオブジェクトが保持されたままになる。
			clear();
			super.release();
		}

		@SuppressWarnings("unchecked")
		public T get(int i) {
			return (T) array[i];
		}

		public void set(int i, T value) {
			array[i] = value;
		}

	}

}
