/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.regex.Pattern;

import net.arnx.jsonic.JSONHint;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.AnimatableColor;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableHorizontalAlignment;
import ch.kuramo.javie.core.AnimatableInteger;
import ch.kuramo.javie.core.AnimatableString;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.DepthBuffer;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.HorizontalAlignment;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.RenderContext;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.VectorMediaInput;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.Font;
import ch.kuramo.javie.core.services.FontList;
import ch.kuramo.javie.core.services.FontManager;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

import ftgl.FTGL;
import ftgl.FTGLfont;

@ProjectElement("textLayer")
public class TextLayerImpl extends AbstractMediaLayer implements TextLayer {

	private final TextInput _input = new TextInput();

	private boolean _ctcr;

	private AnimatableString _sourceText = new AnimatableString("テキスト");

	private AnimatableString _font = new AnimatableString("");

	private AnimatableInteger _fontSize = new AnimatableInteger(36, 1, Integer.MAX_VALUE);

	private AnimatableColor _fillColor = new AnimatableColor(Color.WHITE);

	private AnimatableHorizontalAlignment _horizontalAlignment = new AnimatableHorizontalAlignment(HorizontalAlignment.LEFT);

	private AnimatableDouble _leading = new AnimatableDouble(-1.0);

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private FontList _fontList;

	@Inject
	private FontManager _fontManager;


	@Override
	protected void initialize(boolean videoAvailable, boolean audioAvailable) {
		throw new UnsupportedOperationException("Use initialize() method instead.");
	}

	@Override
	public void initialize() {
		super.initialize(true, false);
	}

	@JSONHint(ignore=true)
	public MediaInput getMediaInput() {
		return _input;
	}

	public boolean isCTCR() {
		return _ctcr;
	}

	public void setCTCR(boolean ctcr) {
		_ctcr = ctcr;
	}

	public AnimatableString getSourceText() {
		return _sourceText;
	}

	public void setSourceText(AnimatableString sourceText) {
		sourceText.copyConfigurationFrom(_sourceText);
		_sourceText = sourceText;
	}

	public AnimatableString getFont() {
		return _font;
	}

	public void setFont(AnimatableString font) {
		font.copyConfigurationFrom(_font);
		_font = font;
	}

	public AnimatableInteger getFontSize() {
		return _fontSize;
	}

	public void setFontSize(AnimatableInteger fontSize) {
		fontSize.copyConfigurationFrom(_fontSize);
		_fontSize = fontSize;
	}

	public AnimatableColor getFillColor() {
		return _fillColor;
	}

	public void setFillColor(AnimatableColor fillColor) {
		fillColor.copyConfigurationFrom(_fillColor);
		_fillColor = fillColor;
	}

	public AnimatableHorizontalAlignment getHorizontalAlignment() {
		return _horizontalAlignment;
	}

	public void setHorizontalAlignment(
			AnimatableHorizontalAlignment horizontalAlignment) {
		horizontalAlignment.copyConfigurationFrom(_horizontalAlignment);
		_horizontalAlignment = horizontalAlignment;
	}

	public AnimatableDouble getLeading() {
		return _leading;
	}

	public void setLeading(AnimatableDouble leading) {
		leading.copyConfigurationFrom(_leading);
		_leading = leading;
	}

	@Override
	public void prepareExpression(ExpressionScope scope) {
		super.prepareExpression(scope);
		scope.assignTo(_sourceText);
		scope.assignTo(_font);
		scope.assignTo(_fontSize);
		scope.assignTo(_fillColor);
		scope.assignTo(_horizontalAlignment);
		scope.assignTo(_leading);
	}

	@Override
	public Object createExpressionElement(RenderContext renderContext) {
		return new TextLayerExpressionElement(renderContext);
	}

	public class TextLayerExpressionElement extends MediaLayerExpressionElement {

		public TextLayerExpressionElement(RenderContext renderContext) {
			super(renderContext);
		}

		public Object getSourceText()			{ return elem(_sourceText); }
		public Object getFont()					{ return elem(_font); }
		public Object getFontSize()				{ return elem(_fontSize); }
		public Object getFillColor()			{ return elem(_fillColor); }
		public Object getHorizontalAlignment()	{ return elem(_horizontalAlignment); }
		public Object getLeading()				{ return elem(_leading); }
	}


	private static final Pattern LF = Pattern.compile("\n");

	private class TextInput implements VectorMediaInput {

		public void dispose() {
			// nothing to do
		}

		public boolean isVideoAvailable() {
			return true;
		}
		
		public boolean isAudioAvailable() {
			return false;
		}

		public Time getDuration() {
			return null;
		}

		public Time getVideoFrameDuration() {
			return null;
		}

		public VideoBounds getVideoFrameBounds() {
			if (!_vrContext.isActive()) {
				return null;
			}

			String sourceText = _sourceText.value(_vrContext);
			String psName = _font.value(_vrContext);
			int fontSize = _fontSize.value(_vrContext);
			FTGLfont font = getFont(psName, fontSize);

			// TODO サイズ0のVideoBoundsを返すとエラーになるので
			//      暫定的な対処として (1, 1) を返しているが、もっと適切な方法を考える。
			//      (プレビュー解像度がFULL以外の場合はこの対処ではエラーになるなど、不完全)
			if (sourceText.length() == 0 || font == null) {
				return new VideoBounds(1, 1);
			}

			HorizontalAlignment halign = _horizontalAlignment.value(_vrContext);

			double leading = _leading.value(_vrContext);
			if (leading < 0) leading = fontSize * 1.2;	// AEは「自動」のときにはフォントサイズの1.2倍になってるっぽい。

			double upper = 0, lower = Double.MAX_VALUE, advanced = 0;
			String[] texts = LF.split(sourceText);

			// [left, lower, near, right, upper, far]
			float[] box = new float[6];
			for (int i = 0; i < texts.length; ++i) {
				FTGL.ftglGetFontBBox(font, texts[i], -1, box);
				upper = Math.max(upper, box[4]);
				lower = Math.min(lower, box[1]);
				advanced = Math.max(advanced, FTGL.ftglGetFontAdvance(font, texts[i]));
			}

			upper = Math.ceil(upper)+1;
			lower = Math.floor(lower)-1;

			double x;
			int width;

			switch (halign) {
				default:
				case LEFT:
					x = 0;
					width = (int)Math.ceil(advanced);
					break;

				case CENTER:
					x = -Math.ceil(advanced/2);
					width = (int)-x*2;
					break;

				case RIGHT:
					x = -Math.ceil(advanced);
					width = (int)-x+1;
					break;
			}

			return new VideoBounds(x, -upper, width, (int)(upper + Math.ceil(leading*(texts.length-1)) - lower));
		}

		public VideoBuffer getVideoFrameImage(Time mediaTime) {
			RenderResolution resolution = _vrContext.getRenderResolution();
			VideoBounds bounds = resolution.scale(getVideoFrameBounds());

			VideoBuffer vb = _vrSupport.createVideoBuffer(_vrContext.getColorMode(), bounds);
			vb.allocateAsTexture();
			vb.clear();

			String psName = _font.value(_vrContext);
			int fontSize = _fontSize.value(_vrContext);
			FTGLfont font = getFont(psName, fontSize);
			if (font != null) {
				String sourceText = _sourceText.value(_vrContext);
				Color fillColor = _fillColor.value(_vrContext);
				HorizontalAlignment halign = _horizontalAlignment.value(_vrContext);
				double leading = _leading.value(_vrContext);

				String[] texts = LF.split(sourceText);

				if (leading < 0) leading = fontSize * 1.2;
				leading = resolution.scale(leading);

				double[][] offsets = calcTextOffsets(font, texts, halign, leading, resolution);

				_vrSupport.renderText(font, fillColor, texts, offsets, vb);
			}

			return vb;
		}

		public void rasterize(
				VideoBuffer resultBuffer, DepthBuffer depthBuffer,
				double[] mvMatrix, double[] prjMatrix) {

			RenderResolution resolution = _vrContext.getRenderResolution();

			String psName = _font.value(_vrContext);
			int fontSize = _fontSize.value(_vrContext);
			FTGLfont font = getFont(psName, fontSize);
			if (font != null) {
				String sourceText = _sourceText.value(_vrContext);
				Color fillColor = _fillColor.value(_vrContext);
				HorizontalAlignment halign = _horizontalAlignment.value(_vrContext);
				double leading = _leading.value(_vrContext);

				String[] texts = LF.split(sourceText);

				if (leading < 0) leading = fontSize * 1.2;
				leading = resolution.scale(leading);

				double[][] offsets = calcTextOffsets(font, texts, halign, leading, resolution);

				_vrSupport.renderText(
						font, fillColor, texts, offsets, resultBuffer, depthBuffer, mvMatrix, prjMatrix);
			}
		}

		private FTGLfont getFont(String psName, int fontSize) {
			Font font = _fontList.get(psName);
			if (font == null && (font = _fontList.defaultFont()) == null) {
				return null;
			}

			String fontPath = font.fontFile.getAbsolutePath();
			int faceIndex = font.faceIndex;

			if (_ctcr) {
				return _fontManager.getPolygonFont(fontPath, faceIndex, fontSize);
			} else {
				return _fontManager.getBufferFont(fontPath, faceIndex, fontSize);
			}
		}

		private double[][] calcTextOffsets(
				FTGLfont font, String[] texts,
				HorizontalAlignment halign, double leading,
				RenderResolution resolution) {

			double[][] offsets = new double[texts.length][2];

			switch (halign) {
				default:
				case LEFT:
					for (int i = 0; i < texts.length; ++i) {
						offsets[i] = new double[] { 0, leading*i };
					}
					break;

				case CENTER:
					for (int i = 0; i < texts.length; ++i) {
						double advance = resolution.scale(FTGL.ftglGetFontAdvance(font, texts[i]));
						offsets[i] = new double[] { -advance/2, leading*i };
					}
					break;

				case RIGHT:
					for (int i = 0; i < texts.length; ++i) {
						double advance = resolution.scale(FTGL.ftglGetFontAdvance(font, texts[i]));
						offsets[i] = new double[] { -advance, leading*i };
					}
					break;
			}

			return offsets;
		}

		public AudioBuffer getAudioChunk(Time mediaTime) {
			throw new UnsupportedOperationException("audio is not available");
		}

	}

}
