/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core;

import java.util.Collection;
import java.util.Iterator;
import java.util.Set;
import java.util.SortedMap;
import java.util.Map.Entry;

import net.arnx.jsonic.JSONHint;
import ch.kuramo.javie.api.IObjectArray;
import ch.kuramo.javie.api.Time;

public abstract class ArithmeticalAnimatableValue<V> extends AbstractAnimatableValue<V> {

	protected V _minValue;

	protected V _maxValue;


	public ArithmeticalAnimatableValue(
			V staticValue, Collection<Keyframe<V>> keyframes, String expression,
			V minValue, V maxValue) {

		super(staticValue, keyframes, expression);
		_minValue = minValue;
		_maxValue = maxValue;
	}

	public ArithmeticalAnimatableValue(V defaultValue, V minValue, V maxValue) {
		super(defaultValue);
		_minValue = minValue;
		_maxValue = maxValue;
	}

	@Override
	public void copyConfigurationFrom(AbstractAnimatableValue<V> src) {
		super.copyConfigurationFrom(src);

		ArithmeticalAnimatableValue<V> src2 = (ArithmeticalAnimatableValue<V>) src;
		_minValue = src2._minValue;
		_maxValue = src2._maxValue;
	}

	public abstract V clamp(V value);

	@Override
	public V value(RenderContext renderContext) {
		return clamp(super.value(renderContext));
	}

	@Override
	public V valueAtTime(Time time, RenderContext renderContext) {
		return clamp(super.valueAtTime(time, renderContext));
	}

	@Override
	public void valuesAtTime(
			final IObjectArray<V> values, final Time sampleDuration, final int evalResolution,
			final Time time, final RenderContext renderContext) {

		final Object[] array = values.getArray();
		final int len = values.getLength();

		if (_expression != null && !renderContext.checkCyclicEvaluation(this)) {
			try {
				renderContext.saveAndExecute(new WrappedOperation<Object>() {
					public Object execute() {
						Time t1 = null, t2 = time;
						double[] v1 = null, v2;

						renderContext.setTime(t2);
						v2 = toArray(renderContext.evaluateExpression(ArithmeticalAnimatableValue.this));

						for (int i = 0; i < len; ++i) {
							if ((i % evalResolution) == 0) {
								t1 = t2;
								v1 = v2;
								t2 = time.add(new Time(sampleDuration.timeValue*(i+evalResolution), sampleDuration.timeScale));

								renderContext.setTime(t2);
								v2 = toArray(renderContext.evaluateExpression(ArithmeticalAnimatableValue.this));
							}
							array[i] = valueOf(linear(v1, v2, t1, t2,
									time.add(new Time(sampleDuration.timeValue*i, sampleDuration.timeScale))));
						}
						return null;
					}
				});
				return;
			} catch (Exception e) {
				// TODO エラーが発生しているかどうかを示すフラグを立てる。_expressionを変更した後はそのフラグはリセットする。
				// TODO そのフラグを立てたとき、つまり初回だけエクスプレッションエラーをGUIに通知する仕組みを作る
				e.printStackTrace();
			}
		}

		Time t1 = null, t2 = time;
		double[] v1 = null, v2 = toArray(interpolate(t2));
		for (int i = 0; i < len; ++i) {
			if ((i % evalResolution) == 0) {
				t1 = t2;
				v1 = v2;
				t2 = time.add(new Time(sampleDuration.timeValue*(i+evalResolution), sampleDuration.timeScale));
				v2 = toArray(interpolate(t2));
			}
			array[i] = valueOf(linear(v1, v2, t1, t2,
					time.add(new Time(sampleDuration.timeValue*i, sampleDuration.timeScale))));
		}
	}

	@Override
	protected V interpolate(
			Time time, Keyframe<V> k1, Keyframe<V> k2, Iterator<Entry<Time, Keyframe<V>>> tail) {

		switch (k1.interpolation) {
			case LINEAR:
				return valueOf(linear(toArray(k1.value), toArray(k2.value), k1.time, k2.time, time));

			case CATMULL_ROM: {
				SortedMap<Time, Keyframe<V>> headMap = _keyframes.headMap(k1.time);
				Keyframe<V> k0 = !headMap.isEmpty() ? headMap.get(headMap.lastKey()) : k1;
				Keyframe<V> k3 = tail.hasNext() ? tail.next().getValue() : k2;
				return valueOf(catmullRom(toArray(k0.value), toArray(k1.value),
						toArray(k2.value), toArray(k3.value), k1.time, k2.time, time));
			}

			default:
				return super.interpolate(time, k1, k2, tail);
		}
	}

	protected abstract double[] toArray(V value);

	protected abstract V valueOf(double[] d);

	protected double[] linear(double[] p1, double[] p2, Time time1, Time time2, Time time) {
		double[] result = new double[p1.length];
		for (int i = 0, n = result.length; i < n; ++i) {
			double s1 = time1.toSecond();
			double s2 = time2.toSecond();
			double s = time.toSecond();

			double t = (s - s1) / (s2 - s1);

			result[i] = (p2[i] - p1[i]) * t + p1[i];
		}
		return result;
	}

	protected double[] catmullRom(
			double[] p0, double[] p1, double[] p2, double[] p3, Time time1, Time time2, Time time) {

		double[] result = new double[p0.length];
		for (int i = 0, n = result.length; i < n; ++i) {
			double s1 = time1.toSecond();
			double s2 = time2.toSecond();
			double s = time.toSecond();

			double t = (s - s1) / (s2 - s1);
			double tt = t*t;
			double ttt = tt*t;

			double v0 = (p2[i] - p0[i]) * 0.5;
			double v1 = (p3[i] - p1[i]) * 0.5;
			result[i] = (2*p1[i] - 2*p2[i] + v0 + v1)*ttt + (-3*p1[i] + 3*p2[i] - 2*v0 - v1)*tt + v0*t + p1[i];
		}
		return result;
	}

	private static final Set<Interpolation> INTERPOLATIONS = Util.unmodifiableSet(
			Interpolation.HOLD, Interpolation.LINEAR, Interpolation.CATMULL_ROM);

	@Override
	@JSONHint(ignore=true)
	public Set<Interpolation> getAvailableInterpolations() {
		return INTERPOLATIONS;
	}

	@Override
	@JSONHint(ignore=true)
	public Interpolation getDefaultInterpolation() {
		return Interpolation.LINEAR;
	}

}
