/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core;

import java.util.Collection;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.NativeJavaArray;
import org.mozilla.javascript.ScriptRuntime;
import org.mozilla.javascript.Scriptable;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.core.exprelems.DoubleProperty;

public class AnimatableColor extends ArithmeticalAnimatableValue<Color> implements IAnimatableColor {

	public AnimatableColor(Color staticValue, Collection<Keyframe<Color>> keyframes, String expression) {
		super(staticValue, keyframes, expression, Color.COLORLESS_TRANSPARENT, Color.WHITE);
	}

	public AnimatableColor(Color defaultValue) {
		super(defaultValue, Color.COLORLESS_TRANSPARENT, Color.WHITE);
	}

	@Override
	public Color clamp(Color value) {
		// Colorクラスは自身のコンストラクタで値の範囲をクランプしているのでそのまま返す。
		return value;
	}

	public Color jsToJava(Object jsValue) {
		if (jsValue instanceof NativeJavaArray) {
			// anchorPoint や position などをそのまま参照した場合 (例: thisComp.layer(3).position)

			Object[] value = (Object[]) ((NativeJavaArray) jsValue).unwrap();
			double[] array = new double[] { 0, 0, 0, 1 };
			for (int i = 0, n = Math.min(4, value.length); i < n; ++i) {
				array[i] = (Double) Context.jsToJava(value[i], double.class);
			}
			return valueOf(array);

		} else /*if (jsValue instanceof NativeArray)*/ {
			// 配列表現を用いた場合 (例: [time*10, 200, 0])

			double[] array = (double[]) Context.jsToJava(jsValue, double[].class);
			if (array.length < 4) {
				double[] tmp = array;
				array = new double[] { 0, 0, 0, 1 };
				for (int i = 0; i < tmp.length; ++i) {
					array[i] = tmp[i];
				}
			}
			return valueOf(array);
		}
	}

	@Override
	protected double[] toArray(Color value) {
		return new double[] { value.r, value.g, value.b, value.a };
	}

	@Override
	protected Color valueOf(double[] d) {
		return new Color(d[0], d[1], d[2], d[3]);
	}

	public Scriptable[] createExpressionElement(final RenderContext renderContext) {
		DoubleProperty r = new DoubleProperty() {
			public double getValue()			{ return value(renderContext).r; }
			public double valueAtTime(double t)	{ return AnimatableColor.this.valueAtTime(t, renderContext).r; }
		};

		DoubleProperty g = new DoubleProperty() {
			public double getValue()			{ return value(renderContext).g; }
			public double valueAtTime(double t)	{ return AnimatableColor.this.valueAtTime(t, renderContext).g; }
		};

		DoubleProperty b = new DoubleProperty() {
			public double getValue()			{ return value(renderContext).b; }
			public double valueAtTime(double t)	{ return AnimatableColor.this.valueAtTime(t, renderContext).b; }
		};

		DoubleProperty a = new DoubleProperty() {
			public double getValue()			{ return value(renderContext).a; }
			public double valueAtTime(double t)	{ return AnimatableColor.this.valueAtTime(t, renderContext).a; }
		};

		return new Scriptable[] {
				renderContext.toNativeJavaObject(r, ScriptRuntime.NumberClass),
				renderContext.toNativeJavaObject(g, ScriptRuntime.NumberClass),
				renderContext.toNativeJavaObject(b, ScriptRuntime.NumberClass),
				renderContext.toNativeJavaObject(a, ScriptRuntime.NumberClass)
		};
	}

}
