/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views;

import java.io.File;
import java.io.IOException;

import javax.swing.filechooser.FileSystemView;

import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.ImageUtil;
import ch.kuramo.javie.core.FileItem;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.TimeCode;

public class ItemLabelProvider extends LabelProvider implements ITableLabelProvider {

	public Image getColumnImage(Object element, int columnIndex) {
		switch (columnIndex) {
			case 0:
				return ImageUtil.getItemIcon((Item) element);

			case 1:
				// TODO ラベルの色で四角く塗りつぶしたアイコン
				return null;

			default:
				return null;
		}
	}

	public String getColumnText(Object element, int columnIndex) {
		Item item = (Item) element;
		switch (columnIndex) {
			case 0: return item.getName();
			case 2: return getKind(item);
			case 3: return getSize(item);
			case 4: return getDuration(item);
			case 5: return getFilePath(item);
			default: return null;
		}
	}

	private String getKind(Item item) {
		String kind = item.getKind();
		if (kind != null) {
			return kind;
		}

		if (!(item instanceof FileItem)) {
			return null;
		}

		File file = ((FileItem) item).getFile();
		if (file.exists()) {
			// TODO このAPIはWindowsでしか実装されていない？
			return FileSystemView.getFileSystemView().getSystemTypeDescription(file);
		}
		return null;
	}

	private String getSize(Item item) {
		if (!(item instanceof FileItem)) {
			return null;
		}

		File file = ((FileItem) item).getFile();
		double length = file.length();

		String[] suffixes = { "B", "KB", "MB", "GB" };
		int i = 0;
		while (i < suffixes.length && length >= 1024) {
			length = length / 1024;
			++i;
		}
		String str = String.format("%.1f", Math.floor(length * 10) / 10);
		return (str.endsWith(".0") ? str.substring(0, str.length()-2) : str)
					+ ((i < suffixes.length) ? suffixes[i] : "TB");
	}

	private String getDuration(Item item) {
		if (!(item instanceof MediaItem)) {
			return null;
		}

		MediaInput input = ((MediaItem) item).getMediaInput();
		Time duration = input.getDuration();
		if (duration == null) {
			return null;
		}

		if (input.isVideoAvailable()) {
			return TimeCode.toTimeCode(duration, input.getVideoFrameDuration());
		} else {
			double s0 = duration.toSecond();
			int h = (int) s0 / 3600; s0 -= h*3600;
			int m = (int) s0 / 60; s0 -= m*60;
			int s = (int) s0;
			int ss = (int) ((s0 - s) * 100);
			return String.format("%d:%02d:%02d.%02d", h, m, s, ss);
		}
	}

	private String getFilePath(Item item) {
		if (!(item instanceof FileItem)) {
			return null;
		}

		File file = ((FileItem) item).getFile();
		try {
			return file.getCanonicalPath();
		} catch (IOException e) {
			return file.getAbsolutePath();
		}
	}

}
