/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.SortedMap;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Interpolation;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;

public class ModifyEffectKeyframeOperation<V> extends ProjectOperation {

	private final String _compId;

	private final String _layerId;

	private final int _effectIndex;

	private final String _property;

	private final Keyframe<V> _oldKeyframe;

	private volatile Keyframe<V> _newKeyframe;


	public ModifyEffectKeyframeOperation(
			ProjectManager projectManager, EffectableLayer layer, int effectIndex,
			String property, Time time, V value, String relation, String name) {

		super(projectManager, name + "の変更", relation);

		LayerComposition comp = projectManager.checkLayer(layer);
		Effect effect = layer.getEffects().get(effectIndex);
		AnimatableValue<V> avalue = PropertyUtil.getProperty(effect, property);
		Keyframe<V> oldKeyframe = avalue.getKeyframe(time);

		setLabel(avalue.hasKeyframe(), (oldKeyframe != null), (value != null));

		_compId = comp.getId();
		_layerId = layer.getId();
		_effectIndex = effectIndex;
		_property = property;
		_oldKeyframe = oldKeyframe;

		if (value != null) {
			_newKeyframe = new Keyframe<V>(time, value, defaultInterpolation(avalue, oldKeyframe, time));
		} else {
			_newKeyframe = null;
		}
	}

	// TODO ModifyLayerKeyframeOperation にも同じメソッドがあるので整理する。
	private void setLabel(boolean hasKeyframe, boolean oldKeyframe, boolean newKeyframe) {
		if (!oldKeyframe) {
			if (!newKeyframe) {
				throw new IllegalArgumentException();
			} else if (hasKeyframe) {
				setLabel("キーフレームの追加");
			} else {
				setLabel("ストップウォッチの使用");
			}
		} else if (!newKeyframe) {
			setLabel("キーフレームの消去");
		}
	}

	// TODO ModifyLayerKeyframeOperation にも同じメソッドがあるので整理する。
	private Interpolation defaultInterpolation(AnimatableValue<V> avalue, Keyframe<V> oldKeyframe, Time time) {
		if (oldKeyframe != null) {
			return oldKeyframe.interpolation;
		} else {
			SortedMap<Time, Keyframe<V>> head = avalue.headKeyframeMap(time);
			if (!head.isEmpty()) {
				return head.get(head.lastKey()).interpolation;
			} else {
				return avalue.getDefaultInterpolation();
			}
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		replaceKeyframe(_oldKeyframe, _newKeyframe, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		replaceKeyframe(_newKeyframe, _oldKeyframe, project, pm);
		return Status.OK_STATUS;
	}

	private void replaceKeyframe(
			Keyframe<V> target, Keyframe<V> replacement,
			Project project, ProjectManager pm) {

		LayerComposition comp = project.getComposition(_compId);
		EffectableLayer layer = (EffectableLayer) comp.getLayer(_layerId);
		Effect effect = layer.getEffects().get(_effectIndex);
		AnimatableValue<V> avalue = PropertyUtil.getProperty(effect, _property);
		Time time;

		if (replacement != null) {
			avalue.putKeyframe(replacement);
			time = replacement.time;
		} else {
			avalue.removeKeyframe(target.time);
			time = target.time;
		}

		if (pm != null) {
			pm.fireKeyframesChange(comp, new Object[][] { new Object[] { layer, _effectIndex, _property, time } });
		}
	}

	@Override
	protected boolean merge(ProjectOperation operation, IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) {

		if (pm == null) {
			// _newKeyframe の値は、シャドウプロジェクトに反映される前に
			// 後続のマージ可能オペレーションで上書きされているかもしれない。
			// しかし、シャドウオペレーションのキューが全て実行された時点での値は一致するので問題ない。
			replaceKeyframe(null, _newKeyframe, project, pm);
			return true;
		}

		if (!(operation instanceof ModifyEffectKeyframeOperation<?>)) {
			return false;
		}

		@SuppressWarnings("unchecked")
		ModifyEffectKeyframeOperation<V> newOp = (ModifyEffectKeyframeOperation<V>) operation;
		if (!newOp._compId.equals(_compId)
				|| !newOp._layerId.equals(_layerId)
				|| newOp._effectIndex != _effectIndex
				|| !newOp._property.equals(_property)) {
			return false;
		}

		Keyframe<V> newKeyframe = newOp._newKeyframe;

		if (_newKeyframe == null || newKeyframe == null) {
			return false;
		}

		if (!_newKeyframe.time.equals(newKeyframe.time)
				|| !_newKeyframe.interpolation.equals(newKeyframe.interpolation)) {
			return false;
		}

		if (!_newKeyframe.value.equals(newKeyframe.value)) {
			_newKeyframe = newKeyframe;
			replaceKeyframe(null, _newKeyframe, project, pm);

			// 結合の結果、このオペレーションの直前の値に戻ったかどうか。
			_noEffect = (_oldKeyframe != null && _oldKeyframe.value.equals(_newKeyframe.value));
		}

		return true;
	}

}
