/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.List;
import java.util.Map;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class LayerSlipOperation extends ProjectOperation {

	private class Entry {
		final String layerId;
		final Time oldStartTime;
		final Time newStartTime;

		public Entry(String layerId, Time oldStartTime, Time newStartTime) {
			super();
			this.layerId = layerId;
			this.oldStartTime = oldStartTime;
			this.newStartTime = newStartTime;
		}
	}


	private final String _compId;

	private List<Entry> _entries = Util.newList();

	private final Object[][] _keyframeData;

	private final String[] _savedAvalues;

	private String[] _newAvalues;

	private Time _deltaTime;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public LayerSlipOperation(
			ProjectManager projectManager, LayerComposition comp,
			Object[][] layersAndStartTimes, Object[][] keyframeData, String[] baseAnimatableValues,
			Time deltaTime, String relation) {

		super(projectManager, "レイヤー時間の変更", relation);
		InjectorHolder.getInjector().injectMembers(this);

		projectManager.checkComposition(comp);
		_compId = comp.getId();

		_keyframeData = keyframeData;
		_savedAvalues = new String[_keyframeData.length];
		_newAvalues = new String[_keyframeData.length];

		init(comp, layersAndStartTimes, baseAnimatableValues, deltaTime);
	}

	private void init(LayerComposition comp, Object[][] layersAndStartTimes, String[] baseAnimatableValues, Time deltaTime) {
		for (Object[] o : layersAndStartTimes) {
			Layer layer = (Layer) o[0];
			if (layer instanceof MediaItemLayer) {
				MediaInput input = ((MediaItemLayer) layer).getMediaInput();
				if (input != null) {
					Time duration = input.getDuration();
					if (duration != null) {
						Time inPoint = layer.getInPoint();
						Time outPoint = layer.getOutPoint();
						Time startTime = (Time) o[1];
						Time endTime = startTime.add(new Time((long)(duration.timeValue/Math.abs(layer.getRate())), duration.timeScale));

						Time t = inPoint.subtract(startTime);
						if (t.before(deltaTime)) {
							deltaTime = t;
						}

						t = outPoint.subtract(endTime);
						if (t.after(deltaTime)) {
							deltaTime = t;
						}
					}
				}
			}
		}

		for (Object[] o : layersAndStartTimes) {
			Layer layer = (Layer) o[0];
			Time newStartTime = ((Time) o[1]).add(deltaTime);
			Time oldStartTime = layer.getStartTime();
			_entries.add(new Entry(layer.getId(), oldStartTime, newStartTime));

			if (newStartTime.equals(oldStartTime)) {
				_noEffect = true;
			}
		}

		if (_noEffect) {
			return;
		}

		_deltaTime = deltaTime;


		AnimatableValue<?>[] newAvalues = new AnimatableValue[_keyframeData.length];
		Map<AnimatableValue<?>, AnimatableValue<?>> map = Util.newMap();

		for (int i = 0, n = _keyframeData.length; i < n; ++i) {
			Layer layer = comp.getLayer((String) _keyframeData[i][0]);
			Integer effectIndex = (Integer) _keyframeData[i][1];
			String property = (String) _keyframeData[i][2];
			Keyframe<?> kf = (Keyframe<?>) _keyframeData[i][3];

			AnimatableValue<?> avalue;
			if (effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(effectIndex);
				avalue = PropertyUtil.getProperty(effect, property);
			}

			AnimatableValue<?> newAvalue = map.get(avalue);
			if (newAvalue == null) {
				_savedAvalues[i] = _encoder.encodeElement(avalue);
				try {
					newAvalue = _decoder.decodeElement(baseAnimatableValues[i], avalue.getClass());
				} catch (ProjectDecodeException e) {
					throw new JavieRuntimeException(e);
				}
				map.put(avalue, newAvalue);
				newAvalues[i] = newAvalue;
			}

			@SuppressWarnings("unchecked")
			AnimatableValue<Object> tmp = (AnimatableValue<Object>) newAvalue;
			tmp.putKeyframe(kf.time.add(deltaTime), kf.value, kf.interpolation);
		}

		for (int i = 0, n = newAvalues.length; i < n; ++i) {
			if (newAvalues[i] != null) {
				_newAvalues[i] = _encoder.encodeElement(newAvalues[i]);
			}
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		slip(false, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		slip(true, project, pm);
		return Status.OK_STATUS;
	}

	private void slip(boolean undo, Project project, ProjectManager pm) throws ExecutionException {
		LayerComposition comp = project.getComposition(_compId);

		List<Layer> data1 = modifyStartTime(undo, pm != null, comp);
		Object[][] data2 = shiftKeyframes(undo, pm != null, comp);

		if (pm != null) {
			pm.fireLayerSlipEdit(comp, new Object[] { data1, data2 });
		}
	}

	private List<Layer> modifyStartTime(boolean undo, boolean createData, LayerComposition comp) {
		List<Layer> layers = null;
		if (createData) {
			layers = Util.newList();
		}

		for (Entry e : _entries) {
			Layer layer = comp.getLayer(e.layerId);
			layer.setStartTime(undo ? e.oldStartTime : e.newStartTime);

			if (createData) {
				layers.add(layer);
			}
		}

		return layers;
	}

	private Object[][] shiftKeyframes(boolean undo, boolean createData, LayerComposition comp) throws ExecutionException {
		Object[][] data = null;
		if (createData) {
			data = new Object[_keyframeData.length][];
		}

		for (int i = 0, n = _keyframeData.length; i < n; ++i) {
			Layer layer = comp.getLayer((String) _keyframeData[i][0]);
			Integer effectIndex = (Integer) _keyframeData[i][1];
			String property = (String) _keyframeData[i][2];
			Keyframe<?> kf = (Keyframe<?>) _keyframeData[i][3];

			AnimatableValue<?> avalue;
			if (effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(effectIndex);
				avalue = PropertyUtil.getProperty(effect, property);
			}

			if (undo) {
				if (_savedAvalues[i] != null) {
					restore(_savedAvalues[i], avalue);
				}
				if (createData) {
					data[i] = new Object[] { layer, effectIndex, property, kf.time };
				}
			} else {
				if (_newAvalues[i] != null) {
					restore(_newAvalues[i], avalue);
				}
				if (createData) {
					data[i] = new Object[] { layer, effectIndex, property, kf.time.add(_deltaTime) };
				}
			}
		}

		return data;
	}

	// TODO ThreeDSwitchOperation 等にほぼ同じメソッドがあるので整理する。
	private <A extends AnimatableValue<V>, V> void restore(String data, A dst) throws ExecutionException {
		try {
			@SuppressWarnings("unchecked")
			A src = _decoder.decodeElement(data, (Class<A>) dst.getClass());

			dst.reset(src.getStaticValue());
			dst.setExpression(src.getExpression());

			for (Keyframe<V> kf : src.getKeyframes()) {
				dst.putKeyframe(kf);
			}

		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding AnimatableValue data", e);
		}
	}

	@Override
	protected boolean merge(ProjectOperation operation, IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) {

		if (pm == null) {
			try {
				synchronized (this) {
					slip(false, project, null);
				}
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			}
			return true;
		}

		if (!(operation instanceof LayerSlipOperation)) {
			return false;
		}

		LayerSlipOperation newOp = (LayerSlipOperation) operation;
		if (!newOp._compId.equals(_compId)
				|| newOp._entries.size() != _entries.size()
				|| newOp._keyframeData != _keyframeData) {
			return false;
		}

		if (!newOp._deltaTime.equals(_deltaTime)) {
			List<Entry> mergedEntries = Util.newList();

			for (int i = 0, n = _entries.size(); i < n; ++i) {
				Entry oldEntry = _entries.get(i);
				Entry newEntry = newOp._entries.get(i);
				if (!newEntry.layerId.equals(oldEntry.layerId)) {
					return false;
				}
				mergedEntries.add(new Entry(oldEntry.layerId, oldEntry.oldStartTime, newEntry.newStartTime));
			}

			synchronized (this) {
				_entries = mergedEntries;
				_newAvalues = newOp._newAvalues;
				_deltaTime = newOp._deltaTime;
			}
			try {
				slip(false, project, pm);
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			}

			// 結合の結果、このオペレーションの直前の値に戻ったかどうか。
			_noEffect = (_deltaTime.timeValue == 0);
		}

		return true;
	}

}
