/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.List;
import java.util.Map;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectDescriptor;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.EffectRegistry;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class AddEffectsOperation extends ProjectOperation {

	private static class Entry {
		private String layerId;
		private String data;
		private int effectIndex;

		private Entry(String layerId, String data, int effectIndex) {
			super();
			this.layerId = layerId;
			this.data = data;
			this.effectIndex = effectIndex;
		}
	}


	private final String _compId;

	private final List<Entry> _entries = Util.newList();

	@Inject
	private EffectRegistry _effectRegistry;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public AddEffectsOperation(
			ProjectManager projectManager, Map<EffectableLayer, Integer> effectIndices, EffectDescriptor ed) {

		super(projectManager, String.format("%sの追加", ed.getLabel()));
		InjectorHolder.getInjector().injectMembers(this);

		if (effectIndices.isEmpty()) {
			_noEffect = true;
			_compId = null;
			return;
		}

		LayerComposition comp = projectManager.checkLayer(effectIndices.entrySet().iterator().next().getKey());
		projectManager.checkComposition(comp);
		_compId = comp.getId();

		for (Map.Entry<EffectableLayer, Integer> e : effectIndices.entrySet()) {
			EffectableLayer layer = e.getKey();
			Integer effectIndex = e.getValue();

			if (projectManager.checkLayer(layer) != comp) {
				throw new IllegalArgumentException();
			}

			effectIndex = Math.max(0, Math.min(effectIndex, layer.getEffects().size()));

			Effect effect = _effectRegistry.newEffect(ed.getType());
			String data = _encoder.encodeElement(effect);
			_entries.add(new Entry(layer.getId(), data, effectIndex));
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);

		for (Entry entry : _entries) {
			Effect effect;
			try {
				effect = _decoder.decodeElement(entry.data, Effect.class);
				effect.afterDecode(project);
			} catch (ProjectDecodeException e) {
				throw new JavieRuntimeException(e);
			}
			EffectableLayer layer = (EffectableLayer) comp.getLayer(entry.layerId);
			layer.getEffects().add(entry.effectIndex, effect);
		}

		if (pm != null) {
			fireEffectsAddRemove(true, comp, pm);
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);

		for (Entry entry : _entries) {
			EffectableLayer layer = (EffectableLayer) comp.getLayer(entry.layerId);
			layer.getEffects().remove(entry.effectIndex);
		}

		if (pm != null) {
			fireEffectsAddRemove(false, comp, pm);
		}

		return Status.OK_STATUS;
	}

	private Object[][] createData(LayerComposition comp) {
		List<Object[]> data = Util.newList();

		for (Entry entry : _entries) {
			EffectableLayer layer = (EffectableLayer) comp.getLayer(entry.layerId);
			data.add(new Object[] { layer, entry.effectIndex });	// TODO 削除のときはインデックスを -1 などにしておいた方がいいかも。
		}

		return data.toArray(new Object[data.size()][]);
	}

	private void fireEffectsAddRemove(boolean add, LayerComposition comp, ProjectManager pm) {
		Object[][] data = createData(comp);

		if (add) {
			pm.fireEffectsAdd(comp, data);
		} else {
			pm.fireEffectsRemove(comp, data);
		}
	}

}
