/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.actions;

import java.awt.Transparency;
import java.awt.color.ColorSpace;
import java.awt.image.BufferedImage;
import java.awt.image.ComponentColorModel;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.DataBufferUShort;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;

import javax.imageio.ImageIO;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.ui.IWorkbenchWindow;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.app.CommandIds;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;
import ch.kuramo.javie.core.services.VideoRenderContext;

import com.google.inject.Inject;

public class SequenceOutputAction extends Action {

	private static File defaultFolder;


	private final LayerCompositionView view;

	@Inject
	private ProjectEncoder encoder;

	@Inject
	private ProjectDecoder decoder;

	@Inject
	private VideoRenderContext vrContext;

	@Inject
	private IArrayPools _arrayPools;


	public SequenceOutputAction(LayerCompositionView view) {
		super("イメージシーケンス...");
		InjectorHolder.getInjector().injectMembers(this);

		this.view = view;

		setId(CommandIds.SEQUENCE_OUTPUT);
		setActionDefinitionId(CommandIds.SEQUENCE_OUTPUT);
		//setImageDescriptor(Activator.getImageDescriptor("/icons/sequence_output.png"));
	}

	public void run() {
		ProjectManager pm = ProjectManager.forWorkbenchWindow(getWindow());
		if (pm == null) {
			return;
		}

		if (defaultFolder == null) {
			File file = pm.getFile();
			if (file != null) {
				defaultFolder = file.getParentFile();
			}
		}

		CompositionItem compItem = view.getCompositionItem();
		File file = showSaveDialog(defaultFolder, compItem.getName());
		if (file != null) {
			defaultFolder = file.getParentFile();
			doOutput(pm.getProject(), compItem.getId(), file);
		}
	}

	private IWorkbenchWindow getWindow() {
		return view.getSite().getWorkbenchWindow();
	}

	private File showSaveDialog(File folder, String name) {
		String[] filterNames = new String[] { "All Files (*)" };
		String[] filterExtensions = new String[] { "*" };

		String platform = SWT.getPlatform();
		if (platform.equals("win32") || platform.equals("wpf")) {
			filterNames = new String[] { "All Files (*.*)" };
			filterExtensions = new String[] { "*.*" };
		}

		FileDialog dialog = new FileDialog(getWindow().getShell(), SWT.SAVE | SWT.SHEET);
		dialog.setFilterNames(filterNames);
		dialog.setFilterExtensions(filterExtensions);
		dialog.setFilterPath(folder != null ? folder.getAbsolutePath() : null);
		dialog.setFileName(name);
		dialog.setOverwrite(false);		// This is OK. ここで指定するのはファイル名のプレフィックスなので、上書きの警告を出しても無意味。

		String path = dialog.open();
		return (path != null) ? new File(path) : null;
	}

	private void doOutput(Project project, String compItemId, File file) {
		Project copy = null;
		try {
			copy = decoder.decodeElement(encoder.encodeElement(project), Project.class);
			copy.afterDecode();

			ProgressMonitorDialog dialog = new ProgressMonitorDialog(getWindow().getShell());
			dialog.create();
			dialog.getShell().setText("書き出し");
			dialog.run(true, true, new SequenceOutput((CompositionItem) copy.getItem(compItemId), file));

		} catch (ProjectDecodeException e) {
			throw new JavieRuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new JavieRuntimeException(e);
		} catch (InterruptedException e) {
			// ユーザーがキャンセルした場合
		} finally {
			if (copy != null) {
				// TODO Project#dispose メソッドを作る。
				for (Item i : copy.getItems()) {
					i.dispose();
				}
			}
		}
	}

	private class SequenceOutput implements  IRunnableWithProgress {

		private final CompositionItem compItem;

		private final File file;

		private final boolean png;


		private SequenceOutput(CompositionItem compItem, File file) {
			this.compItem = compItem;
			this.file = file;

			png = (compItem.getComposition().getColorMode() == ColorMode.RGBA16
							|| file.getName().toLowerCase().endsWith(".png"));
		}

		public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
			Composition comp = compItem.getComposition();
			Time frameDuration = comp.getFrameDuration();
			long numFrames = comp.getDuration().toFrameNumber(frameDuration);
			String filenameFormat = filenameFormat(file.getName(), numFrames);

			monitor.beginTask(String.format("書き出し: %s", compItem.getName()), (int) numFrames);

			vrContext.activate();
			try {
				comp.prepareExpression(vrContext.createInitialExpressionScope(comp));

				for (long i = 0; i < numFrames; ++i) {
					if (monitor.isCanceled()) {
						throw new InterruptedException();
					}
					monitor.subTask(String.format("フレーム: %d/%d", i, numFrames));

					Time time = Time.fromFrameNumber(i, frameDuration);
					vrContext.reset();
					vrContext.setRenderResolution(RenderResolution.FULL);
					vrContext.setTime(time);

					VideoBuffer vb = comp.renderVideoFrame();
					try {
						writeToFile(vb, String.format(filenameFormat, i));
					} catch (IOException e) {
						throw new InvocationTargetException(e);
					} finally {
						vb.dispose();
					}

					monitor.worked(1);
				}
			} finally {
				vrContext.deactivate();
			}

			monitor.done();
		}

		private String filenameFormat(String baseFilename, long numFrames) {
			int lastDot = baseFilename.lastIndexOf('.');
			if (lastDot != -1) {
				baseFilename = baseFilename.substring(0, lastDot);
			}
			return baseFilename.replaceAll("%", "%%")
					+ " %0" + String.valueOf(numFrames-1).length() + "d."
					+ (png ? "png" : "bmp");
		}

		private void writeToFile(VideoBuffer vb, String filename) throws IOException {
			File file = new File(this.file.getParentFile(), filename);

			Size2i size = vb.getImageSize();
			int width = size.width;
			int height = size.height;

			switch (vb.getColorMode()) {
				case RGBA8: {
					byte[] array = (byte[]) vb.getArray();

					if (png) {
						WritableRaster wr = Raster.createInterleavedRaster(
								new DataBufferByte(array, width * height * 4),
								width, height, width*4, 4, new int[] { 2, 1, 0, 3 }, null);
						ComponentColorModel cm = new ComponentColorModel(
								ColorSpace.getInstance(ColorSpace.CS_sRGB), true, false,
								Transparency.TRANSLUCENT, DataBuffer.TYPE_BYTE);
						BufferedImage image = new BufferedImage(cm, wr, false, null);
						ImageIO.write(image, "png", file);

					} else {
						IArray<byte[]> pa = _arrayPools.getByteArray(width*height*3);
						byte[] array2 = pa.getArray();
						try {
							for (int i = 0, n = width*height; i < n; ++i) {
								array2[i*3  ] = array[i*4  ];
								array2[i*3+1] = array[i*4+1];
								array2[i*3+2] = array[i*4+2];
							}
							WritableRaster wr = Raster.createInterleavedRaster(
									new DataBufferByte(array2, width * height * 3),
									width, height, width*3, 3, new int[] { 2, 1, 0 }, null);
							ComponentColorModel cm = new ComponentColorModel(
									ColorSpace.getInstance(ColorSpace.CS_sRGB), false, false,
									Transparency.OPAQUE, DataBuffer.TYPE_BYTE);
							BufferedImage image = new BufferedImage(cm, wr, false, null);
							ImageIO.write(image, "bmp", file);
						} finally {
							pa.release();
						}
					}
					break;
				}
				case RGBA16: {
					short[] array = (short[]) vb.getArray();
					WritableRaster wr = Raster.createInterleavedRaster(
							new DataBufferUShort(array, width * height * 4),
							width, height, width*4, 4, new int[] { 2, 1, 0, 3 }, null);
					ComponentColorModel cm = new ComponentColorModel(
							ColorSpace.getInstance(ColorSpace.CS_sRGB), true, false,
							Transparency.TRANSLUCENT, DataBuffer.TYPE_USHORT);
					BufferedImage image = new BufferedImage(cm, wr, false, null);
					ImageIO.write(image, "png", file);
					break;
				}
				default:
					throw new IllegalArgumentException("unsupported ColorMode: " + vb.getColorMode());
			}
		}

	}

}
