/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.actions;

import java.io.File;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.ui.IWorkbenchWindow;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.app.CommandIds;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.output.MacOSXQTMovieOutput;
import ch.kuramo.javie.core.output.ProgressMonitor;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class QTMovieOutputAction extends Action {

	private static byte[] defaultSettings;

	private static File defaultFolder;


	private final LayerCompositionView view;

	@Inject
	private ProjectEncoder encoder;

	@Inject
	private ProjectDecoder decoder;


	public QTMovieOutputAction(LayerCompositionView view) {
		super("QuickTime ムービー...");
		InjectorHolder.getInjector().injectMembers(this);

		this.view = view;

		setId(CommandIds.QTMOVIE_OUTPUT);
		setActionDefinitionId(CommandIds.QTMOVIE_OUTPUT);
		//setImageDescriptor(Activator.getImageDescriptor("/icons/qtmovie_output.png"));
	}

	public void run() {
		ProjectManager pm = ProjectManager.forWorkbenchWindow(getWindow());
		if (pm == null) {
			return;
		}

		byte[] settings = MacOSXQTMovieOutput.doUserDialogAndGetSettings(defaultSettings);
		if (settings == null) {
			return;
		}
		defaultSettings = settings;

		if (defaultFolder == null) {
			File file = pm.getFile();
			if (file != null) {
				defaultFolder = file.getParentFile();
			}
		}

		CompositionItem compItem = view.getCompositionItem();

		File file = showSaveDialog(defaultFolder, compItem.getName() + ".mov");
		if (file != null) {
			if (file.exists() && !file.delete()) {
				// TODO エラーメッセージを表示する。
				//      あるいは、ユニークな名前の一時ファイルを作ってそこへ書き出し、後でリネームする。
				return;
			}
			defaultFolder = file.getParentFile();
			doOutput(pm.getProject(), compItem.getId(), settings, file);
		}
	}

	private IWorkbenchWindow getWindow() {
		return view.getSite().getWorkbenchWindow();
	}

	private File showSaveDialog(File folder, String name) {
		String[] filterNames = new String[] { "QuickTime Movie Files", "All Files (*)" };
		String[] filterExtensions = new String[] { "*.mov", "*" };

		String platform = SWT.getPlatform();
		if (platform.equals("win32") || platform.equals("wpf")) {
			filterNames = new String[] { "QuickTime Movie Files", "All Files (*.*)" };
			filterExtensions = new String[] { "*.mov", "*.*" };
		}

		FileDialog dialog = new FileDialog(getWindow().getShell(), SWT.SAVE | SWT.SHEET);
		dialog.setFilterNames(filterNames);
		dialog.setFilterExtensions(filterExtensions);
		dialog.setFilterPath(folder != null ? folder.getAbsolutePath() : null);
		dialog.setFileName(name);
		dialog.setOverwrite(true);

		String path = dialog.open();
		return (path != null) ? new File(path) : null;
	}

	private void doOutput(Project project, String compItemId, byte[] settings, File file) {
		Project copy = null;
		try {
			copy = decoder.decodeElement(encoder.encodeElement(project), Project.class);
			copy.afterDecode();

			ProgressMonitorDialog dialog = new ProgressMonitorDialog(getWindow().getShell());
			dialog.create();
			dialog.getShell().setText("書き出し");
			dialog.run(true, true, new OutputWithProgress((CompositionItem) copy.getItem(compItemId), settings, file));

		} catch (ProjectDecodeException e) {
			throw new JavieRuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new JavieRuntimeException(e);
		} catch (InterruptedException e) {
			// ユーザーがキャンセルした場合
		} finally {
			if (copy != null) {
				// TODO Project#dispose メソッドを作る。
				for (Item i : copy.getItems()) {
					i.dispose();
				}
			}
		}
	}

	private class OutputWithProgress implements  IRunnableWithProgress {

		private final CompositionItem compItem;

		private final byte[] settings;

		private final File file;


		private OutputWithProgress(CompositionItem compItem, byte[] settings, File file) {
			this.compItem = compItem;
			this.settings = settings;
			this.file = file;
		}

		public void run(final IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
			MacOSXQTMovieOutput qtOut = InjectorHolder.getInjector().getInstance(MacOSXQTMovieOutput.class);
			qtOut.setSettings(settings);
			qtOut.doOutput(compItem.getComposition(), AudioMode.STEREO_48KHZ_INT16, file,
					new ProgressMonitor() {
						private int totalWork;
						private int currentWork;

						public void beginTask(int totalWork) {
							monitor.beginTask(String.format("書き出し: %s", compItem.getName()), totalWork);
							this.totalWork = totalWork;
						}

						public void currentWork(int currentWork) {
							if (currentWork > this.currentWork) {
								monitor.worked(currentWork - this.currentWork);
								monitor.subTask(String.format("%.1f%%", 100.0 * currentWork / totalWork));
								this.currentWork = currentWork;
							}
						}

						public void done() {
							monitor.done();
						}

						public boolean isCanceled() {
							return monitor.isCanceled();
						}
					});

			if (monitor.isCanceled()) {
				throw new InterruptedException();
			}
		}

	}

	public static boolean isAvailable() {
		try {
			Class.forName(MacOSXQTMovieOutput.class.getName());
			return true;
		} catch (Throwable t) {
			return false;
		}
	}

}
