/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import java.io.File;
import java.io.IOException;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;

import ch.kuramo.javie.app.project.ProjectManager;

public class SaveUtil {

	public enum SaveStatus { NOT_DIRTY, SAVED, UNSAVED, CANCELED, ERROR }


	public static SaveStatus save(ProjectManager pm) {
		try {
			if (pm.getFile() != null) {
				pm.save();
				Perspective.saveForProject(pm);
				return SaveStatus.SAVED;
			} else {
				File file = saveProjectDialog(shell(pm), null, ProjectManager.DEFAULT_NAME);
				if (file != null) {
					pm.saveAs(file);
					Perspective.saveForProject(pm);
					RecentProjects.add(pm);
					return SaveStatus.SAVED;
				} else {
					return SaveStatus.CANCELED;
				}
			}
		} catch (IOException e) {
			UIUtil.errorDialog(shell(pm), "Error", e);
			return SaveStatus.ERROR;
		}
	}

	public static SaveStatus saveAs(ProjectManager pm) {
		File file = pm.getFile();
		if (file == null) {
			file = saveProjectDialog(shell(pm), null, ProjectManager.DEFAULT_NAME);
		} else {
			file = saveProjectDialog(shell(pm), file.getParentFile().getAbsolutePath(), file.getName());
		}

		if (file == null) {
			return SaveStatus.CANCELED;
		}

		try {
			pm.saveAs(file);
			Perspective.saveForProject(pm);
			RecentProjects.add(pm);
			return SaveStatus.SAVED;
		} catch (IOException e) {
			UIUtil.errorDialog(shell(pm), "Error", e);
			return SaveStatus.ERROR;
		}
	}

	public static SaveStatus saveCopy(ProjectManager pm) {
		File file = pm.getFile();
		if (file == null) {
			file = saveProjectDialog(shell(pm), null, ProjectManager.DEFAULT_NAME);
		} else {
			String defaultName = file.getName().replaceFirst("^(.+?)(.[jJ][vV][pP])?$", "$1のコピー$2");
			file = saveProjectDialog(shell(pm), file.getParentFile().getAbsolutePath(), defaultName);
		}

		if (file == null) {
			return SaveStatus.CANCELED;
		}

		try {
			pm.saveCopy(file);

			// saveCopyの場合はPerspectiveを保存しない。
			//Perspective.saveForProject(pm);

			return SaveStatus.SAVED;
		} catch (IOException e) {
			UIUtil.errorDialog(shell(pm), "Error", e);
			return SaveStatus.ERROR;
		}
	}

	public static SaveStatus saveIfDirty(ProjectManager pm) {
		if (!pm.isDirty()) {
			if (pm.getFile() != null) {
				Perspective.saveForProject(pm);
			}
			return SaveStatus.NOT_DIRTY;
		}

		File file = pm.getFile();
		String name = (file != null) ? file.getName() : ProjectManager.DEFAULT_NAME;

		String dialogMessage = String.format("プロジェクト “%s” に加えた変更を保存しますか？", name);
		String[] buttonLabels = { "保存", "保存しない", "キャンセル" };

		MessageDialog dialog = new MessageDialog(shell(pm), null, null,
				dialogMessage, MessageDialog.QUESTION, buttonLabels, 0) {
			protected int getShellStyle() {
				return super.getShellStyle() | SWT.SHEET;
			}
		};

		switch (dialog.open()) {
			case 0:
				return save(pm);
			case 1:
				return SaveStatus.UNSAVED;
			default:
				return SaveStatus.CANCELED;
		}
	}

	private static Shell shell(ProjectManager pm) {
		return pm.getWorkbenchWindow().getShell();
	}

	private static File saveProjectDialog(Shell parent, String defaultFolder, String defaultName) {
		String[] filterNames = new String[] { "Javie Project Files", "All Files (*)" };
		String[] filterExtensions = new String[] { "*.jvp", "*" };

		String platform = SWT.getPlatform();
		if (platform.equals("win32") || platform.equals("wpf")) {
			filterNames = new String[] { "Javie Project Files", "All Files (*.*)" };
			filterExtensions = new String[] { "*.jvp", "*.*" };
		}

		FileDialog dialog = new FileDialog(parent, SWT.SAVE | SWT.SHEET);
		dialog.setFilterNames(filterNames);
		dialog.setFilterExtensions(filterExtensions);
		dialog.setFilterPath(defaultFolder);
		dialog.setFileName(defaultName);
		dialog.setOverwrite(true);

		String path = dialog.open();
		return (path != null) ? new File(path) : null;
	}

}
