/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import java.io.IOException;
import java.net.URL;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.scannotation.AnnotationDB;
import org.scannotation.ClasspathUrlFinder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.plugin.JaviePlugin;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.services.EffectRegistry;
import ch.kuramo.javie.core.services.ProjectElementFactory;
import ch.kuramo.javie.core.services.ShaderBuildException;
import ch.kuramo.javie.core.services.ShaderBuilder;

import com.google.inject.Inject;

public class Initializer {

	public static void initialize() {
		Initializer initializer = InjectorHolder.getInjector().getInstance(Initializer.class);
		initializer.scanCoreClasses();
		initializer.scanExtensions();
	}


	private static final Logger _logger = LoggerFactory.getLogger(Initializer.class);

	@Inject
	private EffectRegistry _effectRegistry;

	@Inject
	private ProjectElementFactory _elementFactory;

	@Inject
	private ShaderBuilder _shaderBuilder;


	Initializer() {
		super();
	}

	private void scanCoreClasses() {
		try {
			URL url = FileLocator.resolve(ClasspathUrlFinder.findClassBase(Project.class));
			AnnotationDB db = new AnnotationDB();
			db.scanArchives(url);

			ClassLoader cl = Project.class.getClassLoader();
			_effectRegistry.searchClasses(db, cl);
			_elementFactory.searchClasses(db, cl);
			_shaderBuilder.buildShaders(db, cl);

		} catch (IOException e) {
			throw new JavieRuntimeException(e);
		} catch (ShaderBuildException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private void scanExtensions() {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry.getExtensionPoint("ch.kuramo.javie.api.plugin");
		if (point == null) {
			return;
		}

		for (IExtension ext : point.getExtensions()) {
			for (IConfigurationElement cfgElem : ext.getConfigurationElements()) {
				if (!"javie-plugin".equals(cfgElem.getName())) {
					continue;
				}

				String id = ext.getUniqueIdentifier();
				String name = ext.getLabel();
				_logger.info(String.format("loading JaviePlugin: %s [%s]", name, id));

				JaviePlugin plugin;
				try {
					plugin = (JaviePlugin) cfgElem.createExecutableExtension("class");
				} catch (CoreException e) {
					_logger.warn("can't load JaviePlugin: " + id, e);
					continue;
				} catch (ClassCastException e) {
					_logger.warn("can't load JaviePlugin: " + id, e);
					continue;
				}

				try {
					Class<?> clazz = plugin.getClass();
					URL url = FileLocator.resolve(ClasspathUrlFinder.findClassBase(clazz));
					AnnotationDB db = new AnnotationDB();
					db.scanArchives(url);

					ClassLoader cl = clazz.getClassLoader();
					_effectRegistry.searchClasses(db, cl);
					_elementFactory.searchClasses(db, cl);
					_shaderBuilder.buildShaders(db, cl);

				} catch (Exception e) {
					_logger.warn("can't load JaviePlugin: " + id, e);
				}
			}
		}
	}

}
