package org.dyndns.nuda.tools.regex.plugins.cache;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.dyndns.nuda.logger.LoggerAdaptor;
import org.dyndns.nuda.logger.LoggerAdaptorInjection;
import org.dyndns.nuda.tools.regex.annotation.Regex;
import org.dyndns.nuda.tools.regex.reflection.cache.CacheManager;
import org.dyndns.nuda.tools.regex.reflection.cache.DefaultRegexCache;
import org.dyndns.nuda.tools.regex.reflection.cache.RegexCache;
import org.dyndns.nuda.tools.regex.settings.JAutoRegexerSettingRepository;
import org.dyndns.nuda.tools.regex.util.exception.RegexPatternException;

import com.danga.MemCached.MemCachedClient;
import com.danga.MemCached.SockIOPool;

/**
 * Memcachedを用いた正規表現キャッシュです
 * @author koseki
 *
 */
public class MemcachedRegexCache implements RegexCache {

	@LoggerAdaptorInjection(name = "MemcachedCacheManager")
	private static LoggerAdaptor logger;

	private static MemCachedClient client = null;

	private static final String MEMCACHED_REGEX_CACHE_PREFIX = "org.dyndns.nuda.tools.regex.plugins.cache.MemcachedRegexCache";

	private static final String SEPARATOR = "___";
	
	public static final String SERVER_INFO_PROPERTY_KEY = "org.dyndns.nuda.tools.regex.plugins.cache.MemcachedRegexCache.property_key";
	
	private static final Pattern PTN = Pattern.compile("(.+?):(.+)");
	
	private static final Object LOCK = new Object();
	private static final Object LOCK02 = new Object();
	
	@Override
	public void init() {
		logger.debug("プラグイン初期化[{}]", "Memcached-Regex-Cache-Pluginは有効です");
		CacheManager.setRegexCache(this);
		
		SockIOPool pool = SockIOPool.getInstance();
		Object setting = JAutoRegexerSettingRepository.getSetting(SERVER_INFO_PROPERTY_KEY);
		
		String[] servers = null;
		
		if(setting == null) {
			servers = new String[]{"localhost:11211"};
		} else {
			try {
				servers = (String[]) setting;
				logger.info("{}", "設定情報を利用します");
				
				for(String server : servers) {
					Matcher m0 = PTN.matcher(server);
					if(m0.matches()) {
						logger.info("Memcachedサーバ情報[{}:{}]", m0.group(1),
								m0.group(2));				
					}

				}
				
			} catch(Exception e) {
				servers = new String[]{"localhost:11211"};
			}
		}
		pool.setServers(servers);
		
		pool.initialize();
		if(!pool.getAliveCheck()) {
			logger.error("{}", "不活性なサーバが存在するため、デフォルトのサーバ設定を利用します");
			
			servers = new String[]{"localhost:11211"};
			pool.setServers(servers);
			
			pool.initialize();
		}

		for(String server : servers) {
			Matcher m0 = PTN.matcher(server);
			if(m0.matches()) {
				logger.info("次のアドレスのmemcachedクライアントに接続しました[{}:{}]", m0.group(1),
						m0.group(2));				
			}

		}


		client = new MemCachedClient();
	}

	// 正規表現フラグはデータ量が小さいため
	// インメモリで持つ
	private static Map<Class<?>, Integer> regexFlgMap = new HashMap<Class<?>, Integer>();
	
	@Override
	public <T> int getRegexFlg(Class<T> cls) throws RegexPatternException {
		if (cls == null) {
			throw new RegexPatternException("対象クラスがnullのため正規表現フラグを生成できません");
		}
		if (!cls.isAnnotationPresent(Regex.class)) {
			throw new RegexPatternException("対象クラスにRegexアノテーションが付与されておりません");
		}

		if(regexFlgMap.containsKey(cls)) {
			return regexFlgMap.get(cls);
		} else {
			Regex regex = cls.getAnnotation(Regex.class);

			int flg = 0;
			if (regex.isMultiLine()) {
				flg = flg | Pattern.MULTILINE;
			}
			if (regex.isDoTall()) {
				flg = flg | Pattern.DOTALL;
			}
			if (regex.isCaseInsentive()) {
				flg = flg | Pattern.CASE_INSENSITIVE;
			}
			if (regex.isComments()) {
				flg = flg | Pattern.COMMENTS;
			}
			if (regex.isLiteral()) {
				flg = flg | Pattern.LITERAL;
			}
			if (regex.isUnixLines()) {
				flg = flg | Pattern.UNIX_LINES;
			}
			if (regex.isUnicodeCase()) {
				flg = flg | Pattern.UNICODE_CASE;
			}
			if (regex.isCanonEq()) {
				flg = flg | Pattern.CANON_EQ;
			}
			
			synchronized(LOCK02) {
				if(!regexFlgMap.containsKey(cls)) {
					regexFlgMap.put(cls, flg);
				}
			}
			
			return flg;
		}
		

	}

	@Override
	public <T> Pattern createPattern(Class<T> cls) throws RegexPatternException {
		if (cls == null) {
			throw new RegexPatternException("対象クラスがnullのため、正規表現オブジェクトを生成できません");
		}
		if (!cls.isAnnotationPresent(Regex.class)) {
			throw new RegexPatternException("対象クラスにRegexアノテーションが付与されておりません:"
					+ cls.getCanonicalName());
		}

		if (client != null) {
			String key = new StringBuilder().append(MEMCACHED_REGEX_CACHE_PREFIX).append(SEPARATOR).append(cls.getCanonicalName()).toString();
			Object obj = client.get(key);
			if (obj == null) {
				Regex regex = cls.getAnnotation(Regex.class);

				int flg = getRegexFlg(cls);

				Pattern pattern = null;
				if (flg != 0) {
					pattern = Pattern.compile(regex.pattern(), flg);
				} else {
					pattern = Pattern.compile(regex.pattern());
				}

				// ここで同期化を行う(マルチスレッド対応)
				// キャッシュの実行はキャッシュのヒットに比べて
				// 少ないはずなので、処理速度には大して影響は出ない想定
				synchronized(LOCK) {
					if(client.get(key) == null) {
						client.set(key, pattern);
					}
				}

				return pattern;
			} else {
				if (obj instanceof Pattern) {
					return (Pattern) obj;
				} else {
					logger.error("{}", 
							"memcachedサーバには妥当なパターンオブジェクトが登録されておりません");
					
					return DefaultRegexCache.getInstance().createPattern(cls);
				}
			}
		} else {
			logger.error("{}", "Memcachedクライアントが存在しません");
			return DefaultRegexCache.getInstance().createPattern(cls);
		}

	}

}
