/******************************************************************************/
/*! @file apolloron.h
    @brief Header file of libapolloron.
    @author Masashi Astro Tachibana, Apolloron Project.
    @date 2018-04-11
 ******************************************************************************/

#ifndef _APOLLORON_H_
#define _APOLLORON_H_

#include <time.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/file.h>
#include <netinet/in.h>
#include <string>

namespace apolloron {

class String;
class List;
class Keys;

/*! @brief Version of libapolloron.
 */
#define LIBAPOLLORON_VERSION "0.46.6"

extern const char *BUILD_DATE;
extern const char *BUILD_HOST;
extern const char *BUILD_OS;
extern const char *BUILD_OS_DISTRO;
extern const char *BUILD_OS_KERNEL;


/*----------------------------------------------------------------------------*/
/* String class                                                               */
/*----------------------------------------------------------------------------*/
/*! @brief Class of variable length character sequence.
 */
class String {
protected:
    // nBinaryLength <= nFixedLength <= nCapacity
    char *pText; // String
    long nLength; // Size of string
    long nCapacity; // Size of all allocated memory
    long nFixedLength; // Memory size which carries out minimum allocation
    long nBinaryLength; // Size in the case of treating as binary data
    bool nLengthRenewRecommended; // this is true if the length of string is uncertain
    String ***pTmp; // Temporary String
    long *pTmpLen; // Number of array of temporary String
    virtual String *tmpStr() const; // Reference of temporary String
    virtual bool clearTmpStr() const; // Deletion of all temporary String
    virtual bool pTextReplace(char *str, long text_len, long binary_len, long capacity); // Replacement pText
    virtual bool resize(long); // Change the size of allocated memory
    virtual bool nLengthRenew(); // renew nLength
    virtual bool nLengthNeedRenew(); // It calls, when length may become unfixed
    virtual bool mresize(long = 0); // Change size of minimum memory allocation
public:
    String();
    String(const String &value);
    String(const char *value);
    String(long value);
    String(int value);
    String(char value);
    String(double value);
    virtual ~String();

    // Deletion of String instance
    virtual bool clear();

    // Check if String is empty
    virtual bool empty() const;

    // Deletion of all temporary String
    virtual bool gc() const;

    // Reference of length of string
    virtual long len() const;

    // Reference of width of string
    virtual long width(const char * src_charset = "UTF-8") const;

    // Get capacity of variable
    virtual long capacity () const;

    // Memory size which carries out minimum allocation
    virtual long fixedLength() const;
    virtual bool setFixedLength(long size);

    // Reference of size as binary String
    virtual long binaryLength() const;

    // Change to binary mode (or change binary size)
    virtual bool useAsBinary(long size);

    // Change to text mode
    virtual bool useAsText();

    // Reference of current mode
    virtual bool isText() const;
    virtual bool isBinary() const;

    // Cast
    virtual const char* c_str() const;
    virtual long toLong() const;
    virtual int toInt() const;
    virtual char toChar() const;
    virtual double toDouble() const;

    virtual operator const char*() const;
    virtual operator long() const;
    virtual operator int() const;
    virtual operator const char() const;
    virtual operator double() const;

    // Reference of array
    virtual char& operator [] (long index);
    virtual char& operator [] (int index);

    // Setup
    virtual bool set(const String &value);
    virtual bool set(const char *value, long length=-1L);
    virtual bool set(long value);
    virtual bool set(int value);
    virtual bool set(char value);
    virtual bool set(double value);
    virtual bool setBinary(char value);
    virtual bool setBinary(const char *value, long length);

    virtual const String& operator = (const String &value);
    virtual const String& operator = (const char *value);
    virtual const String& operator = (long value);
    virtual const String& operator = (int value);
    virtual const String& operator = (char value);
    virtual const String& operator = (double value);

    // Comparison
    virtual int compare(const String &value) const;
    virtual int compare(const char *value) const;
    virtual int compare(long value) const;
    virtual int compare(int value) const;
    virtual int compare(char value) const;
    virtual int compare(double value) const;

    virtual int operator == (const String &value) const;
    virtual int operator == (const char *value) const;
    virtual int operator == (long value) const;
    virtual int operator == (int value) const;
    virtual int operator == (char value) const;
    virtual int operator == (double value) const;

    virtual int operator != (const String &value) const;
    virtual int operator != (const char *value) const;
    virtual int operator != (long value) const;
    virtual int operator != (int value) const;
    virtual int operator != (char value) const;
    virtual int operator != (double value) const;

    virtual int operator <= (const String &value) const;
    virtual int operator <= (const char *value) const;
    virtual int operator <= (long value) const;
    virtual int operator <= (int value) const;
    virtual int operator <= (char value) const;
    virtual int operator <= (double value) const;

    virtual int operator >= (const String &value) const;
    virtual int operator >= (const char *value) const;
    virtual int operator >= (long value) const;
    virtual int operator >= (int value) const;
    virtual int operator >= (char value) const;
    virtual int operator >= (double value) const;

    virtual int operator < (const String &value) const;
    virtual int operator < (const char *value) const;
    virtual int operator < (long value) const;
    virtual int operator < (int value) const;
    virtual int operator < (char value) const;
    virtual int operator < (double value) const;

    virtual int operator > (const String & value) const;
    virtual int operator > (const char *value) const;
    virtual int operator > (long value) const;
    virtual int operator > (int value) const;
    virtual int operator > (char value) const;
    virtual int operator > (double value) const;

    // Addition
    virtual bool add(const String &value);
    virtual bool add(const char *value);
    virtual bool add(const String &value, long length);
    virtual bool add(const char *value, long length);
    virtual bool add(long value);
    virtual bool add(int value);
    virtual bool add(char value);
    virtual bool add(double value);
    virtual bool addBinary(char value);
    virtual bool addBinary(const char *value, long index);

    virtual const String& operator += (const String &value);
    virtual const String& operator += (const char *value);
    virtual const String& operator += (long value);
    virtual const String& operator += (int value);
    virtual const String& operator += (char value);
    virtual const String& operator += (double value);

    virtual String& operator + (const String &value) const;
    virtual String& operator + (const char *value) const;
    virtual String& operator + (long value) const;
    virtual String& operator + (int value) const;
    virtual String& operator + (char value) const;
    virtual String& operator + (double value) const;

    // Insertion
    virtual String& insert(long pos, const String &value) const;
    virtual String& insert(long pos, const char *value) const;
    virtual String& insert(long pos, long value) const;
    virtual String& insert(long pos, int value) const;
    virtual String& insert(long pos, char value) const;
    virtual String& insert(long pos, double value) const;

    // Trim
    virtual String& left(long length) const;
    virtual String& right(long length) const;
    virtual String& mid(long pos, long length = -1L) const;
    virtual String& trimR() const;
    virtual String& trimL() const;
    virtual String& trim() const;

    // Upper / lower
    virtual String& upper() const;
    virtual String& lower() const;

    // Search
    virtual long search(const String &key, long pos = 0L) const;
    virtual long searchCase(const String &key, long pos = 0L) const;
    virtual long searchChar(const char c, long pos = 0L) const;
    virtual String& replace(const String &key, const String &replace_str, long pos = 0L) const;
    virtual String& replaceCase(const String &key, const String &replace_str, long pos = 0L) const;

    // Format
    virtual int sprintf(const char *format, ...);

    // URL encoding
    virtual String& encodeURL() const;
    virtual String& decodeURL(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8") const;

    // Escape HTML tag
    virtual String& escapeHTML(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8", const char *options="") const;
    virtual String& unescapeHTML(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8") const;

    // Escape XML tag
    virtual String& escapeXML(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8", const char *options="") const;
    virtual String& unescapeXML(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8") const;

    // Escape for JSON
    virtual String& escapeJSON(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8") const;
    virtual String& unescapeJSON(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8") const;

    // Escape Back-slash, Double-quote, Single-quote
    virtual String& escapeQuote(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8") const;
    virtual String& unescapeQuote(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8") const;

    // Escape for CSV (" -> "")
    virtual String& escapeCSV(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8", const char * return_str = "\n") const;
    virtual String& unescapeCSV(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8", const char * return_str = "\n") const;

    // HEX encoding
    virtual String& encodeHEX() const;
    virtual String& decodeHEX() const;

    // UUencoding
    virtual String& decodeUU(String * filename = (String *)0) const;

    // MIME encoding
    virtual String& encodeMIME(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8", long max_width = 72, const char * return_str = "\n", const char enctype = 'B') const;
    virtual String& decodeMIME(const char * src_charset = "AUTODETECT", const char * dest_charset = "UTF-8") const;
    virtual String& encodeBASE64(long max_width = 72, const char * return_str = "\n", const char * prefix_str = (const char *)0, const char * suffix_str = (const char *)0) const;
    virtual String& decodeBASE64() const;
    virtual String& encodeQuotedPrintable(long max_width = 72, const char * return_str = "\n", const char * prefix_str = (const char *)0, const char * suffix_str = (const char *)0) const;
    virtual String& decodeQuotedPrintable() const;
    virtual bool parseEmail(String& name, String& user, String& domain, const char * dest_charset = "") const;

    // Crypt / Decrypt
    virtual String& crypt() const;
    virtual String& decrypt() const;

    // MD5 digest key
    virtual String& md5() const;
    virtual String& hmacMd5(const String & key) const;
    virtual String& cramMd5(const String & user, const String & challenge) const;

    // SHA1 hash
    virtual String& sha1() const;
    virtual String& hmacSha1(const String & key) const;

    // Plain text or HTML template convertion
    virtual String& evalText(Keys replace_keys) const;

    // HTML to plain text
    virtual String& convertHTMLToPlain(const char *src_charset = "UTF-8", const char *dest_charset = "UTF-8") const;

    // Character code conversion
    virtual String& strconv(const char *src_charset, const char *dest_charset) const;

    // Character code detection
    virtual const char* detectCharSet() const;
    virtual const char* detectCharSetJP() const;

    // return code (CR/LF/CRLF)
    virtual String& changeReturnCode(const char * return_str = "\n") const;

    // Character width converting
    virtual String& changeWidth(const char *option, const char *src_charset = "UTF-8", const char *dest_charset = "UTF-8") const;

    // Regular expression
    virtual String& reMatch(const char *regex) const;
    virtual String& reSubst(const char *regex1, const char *regex2) const;
    virtual String& reTrans(const char *regex1, const char *regex2) const;

    // Wild Card
    virtual String& wcMatch(const char *wildcard) const;
    virtual bool isWcMatchAll(const char *wildcard) const;

    // Multi-line string processing
    virtual String& uniq() const;

    // Files
    virtual long loadFile(const String& filename);
    virtual long loadFile(const char *filename);
    virtual long loadFilePos(const String& filename, long start_pos, long size);
    virtual long loadFilePos(const char *filename, long start_pos, long size);
    virtual bool saveFile(const String& filename) const;
    virtual bool saveFile(const char *filename) const;
    virtual bool saveFilePos(const String& filename, long start_pos, long size) const;
    virtual bool saveFilePos(const char *filename, long start_pos, long size) const;

    // MIDI data conversion
    virtual String& toMIDI() const;
};


/*----------------------------------------------------------------------------*/
/* OutStream class (an emulation of std::ostream)                             */
/*----------------------------------------------------------------------------*/
class OutStream : public String {
public :
    // OutStream << operator.
    OutStream & operator << (const String & in) {
        *this += in;
        return *this;
    }

    // OutStream << operator.
    OutStream & operator << (const char * in) {
        *this += in;
        return *this;
    }
};


/*----------------------------------------------------------------------------*/
/* List class                                                                 */
/*----------------------------------------------------------------------------*/
/*! @brief Class of array.
 */
class List {
protected:
    String **pList; // content of array
    int pListSize; // Size of array
    virtual bool addString(const String &); // Addition of string
    virtual bool delString(long index); // Deletion of string
    virtual bool insertString(long index, const String & value); // Insertion of string
public:
    List();
    List(const List & list);
    virtual ~List();

    // Deletion of object instance
    virtual bool clear();

    // Reference of element of array
    virtual String& operator [] (long index);
    virtual String& operator [] (int index);
    virtual const char* read(long index) const; // for multi-threaded use
    virtual const char* read(int index) const; // for multi-threaded use

    // Number of elements of array
    virtual long max() const;

    // Setup of array
    virtual bool set(const List & list);
    virtual const List& operator = (const List &list);
    virtual bool setDirFileList(const String & dirname, const char *wildcard="*", bool fullpath=false);
    virtual bool setFileList(const String & dirname, const char *wildcard="*", bool fullpath=false);
    virtual bool setDirList(const String & dirname, const char *wildcard="*", bool fullpath=false);
    virtual bool setEmails(const String & emails_str); /* comma separated */

    // Addition of element of array
    virtual const List& operator += (const String &value);
    virtual bool add(const String &value);

    // Insertion of element of array
    virtual bool insert(long index, const String &value);

    // Deletion of array
    virtual bool remove(long index);
};


// Binary tree element structure
typedef struct {
    char *key;
    String *value;
    long small;
    long big;
} Element;

/*----------------------------------------------------------------------------*/
/* Keys class                                                                 */
/*----------------------------------------------------------------------------*/
/*! @brief Class of associative array.
 */
class Keys {
protected:
    long nElementStart[64]; // start point (BASE64 key)
    long nElementTotal[64];
    Element ***pElement; // (pointers of array) x 64
    String *tmpString; // temporary string
    String *emptyString; // empty string to retuen if not found
public:
    Keys();
    Keys(const Keys & keys);
    virtual ~Keys();

    // Deletion of object instance
    virtual bool clear();

    // Reference of value of element of key
    virtual String& operator [] (const String &key);
    virtual const String& operator [] (const String &key) const;
    virtual String& operator [] (const char *key);
    virtual const String& operator [] (const char *key) const;

    virtual String& getValue(const String &key) const;
    virtual String& getValue(const char *key) const;
    virtual const char* read(const String &key) const; // for multi-threaded use
    virtual const char* read(const char *key) const; // for multi-threaded use

    // Key's existence check
    virtual bool isKeyExist(const String &key) const;
    virtual bool isKeyExist(const char *key) const;

    // Get number of key
    virtual long max() const;

    // Setup of keys
    virtual bool set(const Keys & keys);
    virtual const Keys& operator = (const Keys &keys);

    // Looking up key
    virtual const char *key(long index) const;
    virtual String& value(long index) const;
    virtual Element* array(long index) const;

    // Addition of key
    virtual bool addKey(const char *key, const String &value);
    virtual bool addKey(const char *key, const char *value = "");

    // Deletion of key
    virtual bool delKey(const String &key);
    virtual bool delKey(const char *key);

    // Serialize
    virtual String& toString();
    virtual bool setString(const String &str);

    // Files
    virtual bool loadFile(const String &filename);
    virtual bool loadFile(const char *filename);
    virtual bool saveFile(const String &filename) const;
    virtual bool saveFile(const char *filename) const;
};


/*----------------------------------------------------------------------------*/
/* Sheet class                                                                */
/*----------------------------------------------------------------------------*/
/*! @brief Class of sheet.
 */
class Sheet {
protected:
    List columns; // Column names (with order)
    Keys **pRows; // Rows
    long rowsMax;
    String *tmpString; // temporary string
    String *tmpCSVString; // temporary string for getCSV()
public:
    Sheet();
    Sheet(Sheet & sheet);
    virtual ~Sheet();

    // Deletion of object instance
    virtual bool clear();

    // Get number of cells
    virtual long maxCol() const;
    virtual long maxRow() const;

    // Setup of Sheet
    virtual bool set(Sheet &sheet);
    virtual const Sheet& operator = (Sheet &sheet);

    // compare of sheet
    virtual int operator == (Sheet &sheet);
    virtual int operator != (Sheet &sheet);

    // Columns
    virtual bool setCols(const List &list);
    virtual String& col(long col);
    virtual long colIndex(const String &colname);
    virtual bool addCol(const String &colname);
    virtual bool delCol(const String &colname);
    virtual bool renameCol(const String &old_colname, const String &new_colname);

    // Rows
    virtual bool addRow();
    virtual bool insertRow(long row);
    virtual bool delRow(long row);

    // Get or set value
    virtual String& getValue(const String &colname, long row);
    virtual String& getValue(long col, long row);
    virtual bool setValue(const String &colname, long row, const String &value);
    virtual bool setValue(long col, long row, const String &value);

    // Search
    virtual long searchRow(const String &colname, const String &value);

    // Sort
    virtual bool sort(const String &colname);
    virtual bool sortr(const String &colname);
    virtual bool sortnum(const String &colname);
    virtual bool sortnumr(const String &colname);

    // CSV
    virtual String& getCSV(const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8", const char * return_str = "\n");
    virtual bool setCSV(const String &csv, const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8");
    virtual bool loadCSV(const String &filename, const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8");
    virtual bool saveCSV(const String &filename, const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8", const char * return_str = "\n");
};


typedef struct {
    int mday;    // Day.     [1-31]
    int mon;     // Month.   [1-12]
    int year;    // Year.    [1970-2037]
    int wday;    // Day of week. [0-6]
    int yday;    // Days in year.[0-365]
} TDate;

typedef struct {
    int sec;     // Seconds. [0-61] (2 leap second) */
    int min;     // Minutes. [0-59] */
    int hour;    // Hours.   [0-23] */
} TTime;

typedef struct {
    TDate date;  // Date
    TTime time;  // Time
    long gmtoff; // Registrant timezone (GMT offset)
} TDateTime;

/*----------------------------------------------------------------------------*/
/* DateTime class                                                             */
/*----------------------------------------------------------------------------*/
/*! @brief Class of date and time.
 */
class DateTime {
protected:
    TDateTime tmpDateTime; // Date and time
    String ***pTmp; // Temporary String
    long *pTmpLen; // Number of array of temporary String
    virtual String *tmpStr() const; // Reference of temporary String
    virtual bool clearTmpStr() const; // Deletion of all temporary String
public:
    DateTime();
    DateTime(DateTime &datetime);
    DateTime(TDateTime &datetime);
    DateTime(time_t time_t_time, long gmtoff = 0L);
    DateTime(String &datetimestr);
    virtual ~DateTime();

    // Deletion of object instance
    virtual bool clear();

    // Setup of date and time
    virtual DateTime &operator = (const DateTime &datetime);
    virtual bool setNow();
    virtual bool set(const DateTime &datetime);
    virtual bool set(const String &datetimestr);
    virtual bool set(const TDateTime &datetime);
    virtual bool set(const time_t time_t_time, long gmtoff = 0L);
    virtual bool setDate(const TDate &date);
    virtual bool setTime(const TTime &time);
    virtual bool setGMTOffset(long gmtoff);

    // Getting date and time
    virtual String &toString(const String &format) const;
#define DATEFORMAT_RFC822  "%a, %02d %b %04Y %02H:%02M:%02S %z"
#define DATEFORMAT_INTERNALDATE "%02d-%b-%04Y %02H:%02M:%02S %z"
#define DATEFORMAT_ISO8601 "%04Y-%02m-%02dT%02H:%02M:%02S%Z"
    virtual String &getMessage(const char * msgid, const char * lang = "en", const char * charset = "UTF-8") const;
    virtual const TDateTime &getDateTime() const;
    virtual const TDate &getDate() const;
    virtual const TTime &getTime() const;
    virtual long getGmtOff() const;

    // Calculating date and time
    virtual bool adjustYear(int year);
    virtual bool adjustMonth(int month);
    virtual bool adjustDate(int day);
    virtual bool adjustHour(int hour);
    virtual bool adjustSeconds(int sec);
    virtual bool changeGMTOffset(long gmtoff);

    // Comparison
    virtual int compare(const DateTime &value) const;

    virtual int operator == (const DateTime &value) const;
    virtual int operator != (const DateTime &value) const;
    virtual int operator <= (const DateTime &value) const;
    virtual int operator >= (const DateTime &value) const;
    virtual int operator < (const DateTime &value) const;
    virtual int operator > (const DateTime &value) const;
};


/*----------------------------------------------------------------------------*/
/* MIMEHeader class                                                           */
/*----------------------------------------------------------------------------*/
/*! @brief Class of MIMEHeader.
 */
class MIMEHeader {
protected:
    List names; // Header names
    List values; // Header values
    String *tmpString; // temporary string
public:
    MIMEHeader();
    MIMEHeader(const String & mime_header);
    MIMEHeader(MIMEHeader & mime_header);
    virtual ~MIMEHeader();

    // Deletion of object instance
    virtual bool clear();

    // Get number of line
    virtual long maxRow() const;

    // Setup of MIMEHeader
    virtual bool set(const String & mime_header);
    virtual bool set(MIMEHeader &mime_header);
    virtual const MIMEHeader& operator = (const String & mime_header);
    virtual const MIMEHeader& operator = (MIMEHeader &mime_header);

    // Access
    virtual String& getName(long row);
    virtual long searchRow(const String &name);
    virtual String& getValue(long row);
    virtual String& getValue(const String &name); // ignore case, first match
    virtual bool addRow(const String &name, const String &value); // overwrite if same name exist
    virtual bool insertRow(long row, const String &name, const String &value);
    virtual bool delRow(long row);
    virtual bool setName(long row, const String &name);
    virtual bool setValue(long row, const String &value);

    // Get header as text
    virtual String& toString();
};


/*----------------------------------------------------------------------------*/
/* Socket class                                                               */
/*----------------------------------------------------------------------------*/
/*! @brief Class of basic networking of TCP socket.
 */
class Socket {
protected:
    int dstSocket; // Destination socket number
    int nTimeout; // Timeout value (second)
    int nErrno; // Error number
    bool bConnected; // false: Disconnected  true: Connected
    String *tmpString; // Cash for receiving data
    bool bSSLStream; // false: non SSL  true: SSL
    void *pSSL; // SSL*
    void *pSSLCTX; // SSL_CTX*
public:
    Socket(int recycle_dstSocket = -1);
    virtual ~Socket();

    // Deletion of object instance
    virtual bool clear();

    // Connection
    virtual bool connect(const char *host, const char *port, bool ssl = false);

    // Disconnection
    virtual bool disconnect();

    // Timeout setup
    virtual bool setTimeout(int sec);

    // Refer to error number
    virtual int error() const;

    // Connection status check
    virtual bool connected() const;

    // STARTTLS
    virtual bool startTls();

    // Data transmission
    virtual bool send(const String &str);
    virtual bool send(const char *str);

    // Data reception
    virtual String &receive(long size);
    virtual String &receiveLine();
    virtual String &receivedData() const;
};


/*----------------------------------------------------------------------------*/
/* CGI class                                                                  */
/*----------------------------------------------------------------------------*/
/*! @brief CGI Class.
 */
class CGI {
protected:
    Keys requestData; // CGI request values
    Keys requestFileName; // CGI request file names
    Keys cookie; // CGI request file names
    bool putHead; // put header done?
    List header; // header strings
public:
    CGI();
    virtual ~CGI();

    // Deletion of object instance
    virtual bool clear();

    virtual String& getValue(const String &key);
    virtual String& getValue(const char *key);
    virtual const char *getEnv(const String &key) const;
    virtual const char *getEnv(const char *key) const;
    virtual String& getFileName(const String &key, const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8");
    virtual String& getFileName(const char *key, const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8");
    virtual String& getCookie(const char *key, const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8");

    virtual bool setHeader(const String &key, const String &value);
    virtual bool clearHeader();
    virtual bool putHeader();
    virtual bool putContent(const String &content, long size = -1);
    virtual bool putContent(const char *content, long size = -1);
    virtual bool putError(const String &err_str, long size = -1);
    virtual bool putError(const char *err_str, long size = -1);
};


#ifndef _FCGIAPP_H
typedef struct FCGX_Stream {
    unsigned char *rdNext;    /* reader: first valid byte
                               * writer: equals stop */
    unsigned char *wrNext;    /* writer: first free byte
                               * reader: equals stop */
    unsigned char *stop;      /* reader: last valid byte + 1
                               * writer: last free byte + 1 */
    unsigned char *stopUnget; /* reader: first byte of current buffer
                               * fragment, for ungetc
                               *                                * writer: undefined */
    int isReader;
    int isClosed;
    int wasFCloseCalled;
    int FCGI_errno;                /* error status */
    void (*fillBuffProc) (struct FCGX_Stream *stream);
    void (*emptyBuffProc) (struct FCGX_Stream *stream, int doClose);
    void *data;
} FCGX_Stream;

typedef struct FCGX_Request {
    int requestId;            /* valid if isBeginProcessed */
    int role;
    FCGX_Stream *in;
    FCGX_Stream *out;
    FCGX_Stream *err;
    char **envp;

    /* Don't use anything below here */

    struct Params *paramsPtr;
    int ipcFd;               /* < 0 means no connection */
    int isBeginProcessed;     /* FCGI_BEGIN_REQUEST seen */
    int keepConnection;       /* don't close ipcFd at end of request */
    int appStatus;
    int nWriters;             /* number of open writers (0..2) */
    int flags;
    int listen_sock;
} FCGX_Request;
#endif

/*----------------------------------------------------------------------------*/
/* FCGI class                                                                 */
/*----------------------------------------------------------------------------*/
/*! @brief FCGI Class.
 */

int FCGX_Init(void);
int FCGX_InitRequest(FCGX_Request *request, int sock, int flags);
int FCGX_Accept_r(FCGX_Request *request);
void FCGX_Finish_r(FCGX_Request *request);

class FCGI : public CGI {
protected:
    FCGX_Request *request;
public:
    FCGI(FCGX_Request *req);
    virtual ~FCGI();

    // Deletion of object instance
    virtual bool clear();

    virtual String& getValue(const String &key);
    virtual String& getValue(const char *key);
    virtual const char *getEnv(const String &key) const;
    virtual const char *getEnv(const char *key) const;
    virtual String& getFileName(const String &key, const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8");
    virtual String& getFileName(const char *key, const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8");
    virtual String& getCookie(const char *key, const char * src_charset = "UTF-8", const char * dest_charset = "UTF-8");

    virtual bool setHeader(const String &key, const String &value);
    virtual bool clearHeader();
    virtual bool putHeader();
    virtual bool putContent(const String &content, long size = -1);
    virtual bool putContent(const char *content, long size = -1);
    virtual bool putError(const String &err_str, long size = -1);
    virtual bool putError(const char *err_str, long size = -1);
};


/*----------------------------------------------------------------------------*/
/* HTTPClient class                                                           */
/*----------------------------------------------------------------------------*/
/*! @brief Class of HTTP client
 */
class HTTPClient {
protected:
    String content;
    String origCharset;
public:
    HTTPClient();
    virtual ~HTTPClient();

    // Deletion of String instance
    virtual bool clear();

    // get content specified by URL
    virtual String& getURL(const String &url, long timeout=5);

    // get content original charset
    virtual String& getOrigCharset();
};


/*----------------------------------------------------------------------------*/
/* FTPStream class                                                           */
/*----------------------------------------------------------------------------*/
/*! @brief Class of FTP Stream
 */

typedef enum {UNIX = 1, Windows_NT = 2} TFTPSystemType;
typedef enum {ASCII = 1, IMAGE = 3} TFTPDataType;

typedef struct {
    int socket;
    struct sockaddr_in saddr;
    TFTPSystemType system_type;
    TFTPDataType data_type;
    int flag_passive;
    int timeout_sec;

    int reply_code;
    char error_message[256];
} LIBOFTP;

class FTPStream {
protected:
    String host;
    String port;
    String user; // FTP auth ID
    String passwd; // FTP auth password
    bool loggedIn;
    LIBOFTP ftp_obj; // LIBOFTP object
    bool pasvMode;
    int timeout;
    String tmpString;
    Sheet tmpSheet;
public:
    FTPStream();
    virtual ~FTPStream();
    bool clear();

    // Timeout setup
    virtual bool setTimeout(int sec);

    virtual bool login(const String &user, const String &passwd,
                       const String &host, const String &port="21");
    virtual bool logout(); // QUIT and kill socket stream
    virtual bool isLoggedIn();
    virtual bool passive(bool pasv_mode=true);
    virtual bool isPassive();
    virtual bool reset();
    virtual bool noop();

    virtual bool site(const String &cmdline);
    virtual bool dele(const String &filename);
    virtual bool rename(const String &from, const String &to);

    virtual String &receiveBuffer(const String &filename);
    virtual bool sendBuffer(const String &buf, const String &filename);
    virtual bool appendBuffer(const String &buf, const String &filename);
    virtual bool receiveFile(const String &filename, const String &local_filename);
    virtual bool sendFile(const String &local_filename, const String &filename);
    virtual bool appendFile(const String &local_filename, const String &filename);

    virtual Sheet &list();
    virtual Sheet &nlist();
    virtual bool mkdir(const String &dirname);
    virtual bool mkdirp(const String &dirname);
    virtual bool rmdir(const String &dirname);
    virtual String &pwd();
    virtual bool cd(const String &dirname);
};


/*----------------------------------------------------------------------------*/
/* POP3Stream class                                                           */
/*----------------------------------------------------------------------------*/
/*! @brief Class of POP3 Stream
 */

class POP3Stream {
protected:
    String host;
    String port;
    String user; // POP3 auth ID
    String passwd; // POP3 auth password
    bool loggedIn;
    Socket socket; // POP3 stream
    String tmpString;
public:
    POP3Stream();
    virtual ~POP3Stream();
    bool clear();

    // Timeout setup
    virtual bool setTimeout(int sec);

    virtual bool login(const String &user, const String &passwd,
                       const String &host, const String &port="110", bool apop = false);
    virtual bool logout(); // QUIT and kill socket stream
    virtual bool isLoggedIn(); // NOOP

    virtual bool stat(long &total_num, long &total_size); // STAT
    virtual bool size(List &size); // LIST
    virtual String& fetchHeader(long n); // TOP n 0
    virtual String& fetch(long n); // RETR n
    virtual String& uidl(long n); // UIDL n
    virtual bool dele(long n); // DELE n
};


/*----------------------------------------------------------------------------*/
/* IMAPStream class                                                           */
/*----------------------------------------------------------------------------*/
/*! @brief Class of IMAP Stream
 */

class IMAPStream {
protected:
    String host;
    String port;
    String user; // IMAP auth ID
    String passwd; // IMAP auth password
    bool loggedIn;
    String mailbox; // selected modified UTF-7 mailbox name
    bool readOnly; // select(false) or examine(true)
    String capability; // list of IMAP extentions
    Socket socket; // IMAP stream
    long nextTag;
    String tmpString;
public:
    IMAPStream();
    virtual ~IMAPStream();
    bool clear();

    // Timeout setup
    virtual bool setTimeout(int sec);

    virtual bool login(const String &user, const String &passwd,
                       const String &host, const String &port="143");
    virtual bool logout(); // logout and kill socket stream
    virtual bool isLoggedIn() const;

    virtual bool select(const String &mailbox);
    virtual bool examine(const String &mailbox);
    virtual bool unselect();
    virtual const String& getCapability();
    virtual const String& getSelectedMailBox();
    virtual bool isReadOnly();

    virtual bool append(const String &eml, const String &internal_date="");
    virtual bool createMailBox(const String &mailbox);
    virtual bool deleteMailBox(const String &mailbox);

    virtual long getNextTag();

    virtual bool send(const String& line);
    virtual String& receiveLine();
    virtual String& receive(long size);
};


/*----------------------------------------------------------------------------*/
/* IMAPMailBoxList class                                                      */
/*----------------------------------------------------------------------------*/
/*! @brief Class of IMAP mail boxes
 */

class IMAPMailBoxList {
protected:
    IMAPStream *imapStream;
    List mailboxListUtf7; // modified UTF-7 original mailbox list
    List mailboxList; // UTF-8 mailbox list with '/' delimiter
    List namespaceDelimList; // namespace delimiter ('.', '/' or '\\')
    virtual bool setMailBoxList();
public:
    IMAPMailBoxList(IMAPStream *imap_stream);
    virtual ~IMAPMailBoxList();

    // Deletion of object instance
    virtual bool clear();

    // sync with IMAP (usually not needed, it's automatic)
    virtual bool reload();

    virtual const List &getMailBoxListUtf7();
    virtual const List &getMailBoxList();
    virtual const List &getNamespaceDelimList();

    virtual bool loadFile(const String &filename); // load from mailbox cache
    virtual bool saveFile(const String &filename); // save to mailbox cache
};


/*----------------------------------------------------------------------------*/
/* IMAPSearch class                                                         */
/*----------------------------------------------------------------------------*/
/*! @brief Class of IMAP mail list
 */

class IMAPSearch {
protected:
    IMAPStream *imapStream;
    String mailbox; // modified UTF-7 mailbox name
    String sortCondition;
    String searchCondition;
    String searchCharSet;
    long *uids; // NULL terminated list
public:
    IMAPSearch(IMAPStream *imap_stream, const String &mailbox,
               const String &sort_cond="REVERSE ARRIVAL",
               const String &search_cond="ALL",
               const String &search_charset="UTF-8");
    virtual ~IMAPSearch();

    // Deletion of object instance
    virtual bool clear();

    virtual bool setUIDs();
    virtual const long *getUIDs();

    virtual bool loadFile(const String &filename); // load from mail list cache
    virtual bool saveFile(const String &filename); // save to mail list cache
};


/*----------------------------------------------------------------------------*/
/* IMAPMail class                                                             */
/*----------------------------------------------------------------------------*/
/*! @brief Class of IMAP mail
 */

typedef struct {
    char* id; // ex. "1.2.1"
    char* filename; // UTF-8 file name
    char* contentType; // Content-Type header
    char* contentTransferEncoding; // Content-Transfer-Encoding header
    char* contentID; // Content-ID header
} TMailPart;

typedef struct {
    char* internalDate;
    long size;
    char* headerDate;
    char* messageID;
    char* subject;
    List fromList;
    List senderList;
    List toList;
    List ccList;
    List bccList;
    List replyToList;
    List referenceList;
    List flagList;
    long partsLength;
    TMailPart *parts;
} TMail;

class IMAPMail {
protected:
    IMAPStream *imapStream;
    String mailbox; // modified UTF-7 mailbox name
    long uid;
    TMail mailData;
    List AttachFilePartIDList;
    String tmpString;
    virtual bool setMailStructure();
public:
    IMAPMail(IMAPStream *imap_stream, const String &mailbox, long uid);
    virtual ~IMAPMail();

    // Deletion of object instance
    virtual bool clear();

    virtual String& getEML(); // raw mail source
    virtual String& getHeader(); // raw header text
    virtual String& getInteralDate(); // IMAP INTERNALDATE
    virtual long getSize(); // IMAP RFC822.SIZE
    virtual String& getHeaderDate(); // MIME-decoded date header (UTF-8)
    virtual String& getMessageID(); // Message-ID
    virtual String& getSubject(); // UTF-8 Subject
    virtual List& getFromList(); // UTF-8 From list
    virtual List& getSenderList(); // UTF-8 Sender list
    virtual List& getToList(); // UTF-8 To list
    virtual List& getCcList(); // UTF-8 Cc list
    virtual List& getBccList(); // UTF-8 Bcc list
    virtual List& getReplyToList(); // UTF-8 Reply-To list
    virtual List& getReferenceList(); // References
    virtual List& getFlagList(); // UTF-8 Bcc list
    virtual String& getPlainTextBody(); // UTF-8 plain text body
    virtual String& getHTMLBody(); // UTF-8 HTML body
    virtual List& getAttachFilePartIDList();
    virtual String& getAttachFileNane(const String &part_id); // UTF-8 file name
    virtual String& getAttachFile(const String &part_id); // file content(binary)
    virtual const TMailPart& getMailPart(const String &part_id);
};


/*----------------------------------------------------------------------------*/
/* SMTPStream class                                                           */
/*----------------------------------------------------------------------------*/
/*! @brief Class of SMTP Stream
 */
class SMTPStream {
protected:
    String host;
    String port;
    Socket socket; // SMTP stream
    List responseList;
public:
    SMTPStream();
    virtual ~SMTPStream();

    // Deletion of String instance
    virtual bool clear();

    virtual bool setTimeout(int sec);

    virtual bool connect(const String &host, const String &port="25");
    virtual bool disconnect(); // send QUIT and kill socket stream

    virtual bool sendEHLO(const String &fqdn);
    virtual bool sendMailFrom(const String &mail_from);
    virtual bool sendRcptTo(const String &rcpt_to);
    virtual bool sendData(const String &eml);

    virtual const List& getResponseList() const;
};


/*----------------------------------------------------------------------------*/
/* Utility functions                                                          */
/*----------------------------------------------------------------------------*/
/*! @brief Utility functions.
 */
bool isFileExist(const String &filename);
bool isDirExist(const String &dirname);
bool isEmailAddress(const String &email);
bool isEmailLocalPart(const String &local_part);
bool isEmailDomain(const String &domain);


/*----------------------------------------------------------------------------*/
/* ANSI compatible functions                                                  */
/*----------------------------------------------------------------------------*/
/*! @brief ANSI clone functions.
 */
int itoa(char *buf, int value, int s=1, int base=10);
int ltoa(char *buf, long value, int s=1, int base=10);
int dtoa(char *buf, double value);
int dtoa_lang(char *buf, double value, int figure, const char *lang);
int mkdirp(const char *pathname, mode_t mode);
int rmrf(const char *pathname);

#if defined(__GNUC__)
#ifdef strcasestr
#undef strcasestr
#endif
#define strcasestr(x, y) ({ \
  const char *_strcasestr_pstr; \
  int _strcasestr_i; \
  char *_strcasestr_ret; \
  const char *_strcasestr_y = (y); \
  if (!*(x) || !(_strcasestr_y)) { \
    _strcasestr_ret = NULL; \
  } else if (!*(y)) { \
    _strcasestr_ret = (char *)(x); \
  } else { \
    _strcasestr_ret = NULL; \
    for (_strcasestr_pstr = (x); *_strcasestr_pstr; _strcasestr_pstr++) { \
      for (_strcasestr_i = 0; (y)[_strcasestr_i]; _strcasestr_i++) { \
        if (toupper(_strcasestr_pstr[_strcasestr_i]) != toupper((y)[_strcasestr_i])) break; \
      } \
      if (!((y)[_strcasestr_i])) { \
        _strcasestr_ret = (char *)_strcasestr_pstr; \
        break; \
      } \
    } \
  } \
  (_strcasestr_ret); \
})
#endif

#if defined(__GNUC__)
#ifdef strlen
#undef strlen
#endif
#define strlen(x) ({ \
  const char *_strlen_str; \
  int _strlen_len; \
  _strlen_str = (const char *)(x); \
  _strlen_len = 0; \
  while (_strlen_str[_strlen_len] != '\0') _strlen_len++; \
  (_strlen_len); \
})
#endif

#if defined(__GNUC__)
#ifdef strcpy
#undef strcpy
#endif
#define strcpy(x, y) ({ \
  char *_strcpy_dest, *_strcpy_save; \
  const char *_strcpy_src; \
  _strcpy_dest = (x); \
  _strcpy_src = (y); \
  _strcpy_save = _strcpy_dest; \
  for (; (*_strcpy_dest = *_strcpy_src) != '\0'; ++_strcpy_src, ++_strcpy_dest); \
  (_strcpy_save); \
})
#endif

#if defined(__GNUC__)
#ifdef strcmp
#undef strcmp
#endif
#define strcmp(x, y) ({ \
  const char *_strcmp_string1, *_strcmp_string2; \
  int _strcmp_first, _strcmp_second; \
  _strcmp_string1 = (x); \
  _strcmp_string2 = (y); \
  do { \
    _strcmp_first  = (int)((unsigned char)*_strcmp_string1); \
    _strcmp_second = (int)((unsigned char)*_strcmp_string2); \
    _strcmp_string1++; \
    _strcmp_string2++; \
  } while (_strcmp_first && _strcmp_first == _strcmp_second); \
  (_strcmp_first - _strcmp_second); \
})
#endif

#if defined(__GNUC__)
#ifdef strncmp
#undef strncmp
#endif
#define strncmp(x, y, z) ({ \
  const char *_strncmp_string1, *_strncmp_string2; \
  int _strncmp_first = 0, _strncmp_second = 0, _strncmp_n; \
  _strncmp_string1 = (x); \
  _strncmp_string2 = (y); \
  _strncmp_n = (z); \
  while (0 < _strncmp_n) { \
    _strncmp_first  = (int)((unsigned char)*_strncmp_string1); \
    _strncmp_second = (int)((unsigned char)*_strncmp_string2); \
    _strncmp_string1++; \
    _strncmp_string2++; \
    _strncmp_n--; \
    if (!(_strncmp_first && _strncmp_first == _strncmp_second)) break; \
  } \
  (_strncmp_first - _strncmp_second); \
})
#endif

#if defined(__GNUC__)
#ifdef strcasecmp
#undef strcasecmp
#endif
#define strcasecmp(x, y) ({ \
  const char *_strcasecmp_string1, *_strcasecmp_string2; \
  int _strcasecmp_first, _strcasecmp_second; \
  _strcasecmp_string1 = (x); \
  _strcasecmp_string2 = (y); \
  do { \
    _strcasecmp_first  = tolower(*_strcasecmp_string1); \
    _strcasecmp_second = tolower(*_strcasecmp_string2); \
    _strcasecmp_string1++; \
    _strcasecmp_string2++; \
  } while (_strcasecmp_first && _strcasecmp_first == _strcasecmp_second); \
  (_strcasecmp_first - _strcasecmp_second); \
})
#endif

#if defined(__GNUC__)
#ifdef strncasecmp
#undef strncasecmp
#endif
#define strncasecmp(x, y, z) ({ \
  const char *_strncasecmp_string1, *_strncasecmp_string2; \
  int _strncasecmp_n = (z); \
  int _strncasecmp_first = 0, _strncasecmp_second = 0; \
  _strncasecmp_string1 = (x); \
  _strncasecmp_string2 = (y); \
  while (0 < _strncasecmp_n) { \
    _strncasecmp_first  = tolower(*_strncasecmp_string1); \
    _strncasecmp_second = tolower(*_strncasecmp_string2); \
    _strncasecmp_string1++; \
    _strncasecmp_string2++; \
    _strncasecmp_n--; \
    if (!(_strncasecmp_first && _strncasecmp_first == _strncasecmp_second)) break; \
  } \
  (_strncasecmp_first - _strncasecmp_second); \
})
#endif

#if defined(__GNUC__)
#ifdef strcat
#undef strcat
#endif
#define strcat(x, y) ({ \
  char *_strcat_dest, *_strcat_save; \
  const char *_strcat_src; \
  _strcat_dest = (x); \
  _strcat_src = (y); \
  _strcat_save = _strcat_dest; \
  for (; *_strcat_dest != '\0'; ++_strcat_dest); \
  for (; (*_strcat_dest = *_strcat_src) != '\0'; ++_strcat_src, ++_strcat_dest); \
  (_strcat_save); \
})
#endif

/* ctype.h macros */
#ifndef _CTYPE_H  /* for Solaris */
#define _CTYPE_H
#endif
#ifndef _CTYPE_H_ /* for MacOSX */
#define _CTYPE_H_
#endif
#ifndef __H_CTYPE /* for AIX */
#define __H_CTYPE
#endif
#ifndef _CTYPE_INCLUDED /* for HP-UX */
#define _CTYPE_INCLUDED
#endif

#ifdef isalnum
#undef isalnum
#endif
#define isalnum(c)   (isalpha(c) || isdigit(c))

#ifdef isalpha
#undef isalpha
#endif
#define isalpha(c)   (isupper(c) || islower(c))

#ifdef isascii
#undef isascii
#endif
#define isascii(c)   (0 < ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)0x7F)

#ifdef iscntrl
#undef iscntrl
#endif
#define iscntrl(c)   (((unsigned char)(c)) <= (unsigned char)0x1F || ((unsigned char)(c)) == (unsigned char)0x7F)

#ifdef isdigit
#undef isdigit
#endif
#define isdigit(c)   ((unsigned char)'0' <= ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'9')

#ifdef isgraph
#undef isgraph
#endif
#define isgraph(c)   (((unsigned char)(c)) != (unsigned char)' ' && isprint(c))

#ifdef islower
#undef islower
#endif
#define islower(c)   ((unsigned char)'a' <= ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'z')

#ifdef isprint
#undef isprint
#endif
#define isprint(c)   ((unsigned char)' ' <= ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'~')

#ifdef ispunct
#undef ispunct
#endif
#define ispunct(c)   (((unsigned char)' ' < ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'~') && !isalnum(c))

#ifdef isspace
#undef isspace
#endif
#define isspace(c)   (((unsigned char)(c)) == (unsigned char)' ' || ((unsigned char)(c)) == (unsigned char)'\f' || ((unsigned char)(c)) == (unsigned char)'\n' || ((unsigned char)(c)) == (unsigned char)'\r' || ((unsigned char)(c)) == (unsigned char)'\t' || ((unsigned char)(c)) == (unsigned char)'\v')

#ifdef isblank
#undef isblank
#endif
#define isblank(c)   ((unsigned char)(c)) == (unsigned char)' ')

#ifdef isupper
#undef isupper
#endif
#define isupper(c)   ((unsigned char)'A' <= ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'Z')

#ifdef isxdigit
#undef isxdigit
#endif
#define isxdigit(c)  (isxupper(c) || isxlower(c))

#ifdef isxlower
#undef isxlower
#endif
#define isxlower(c)  (isdigit(c) || ((unsigned char)'a' <= ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'f'))

#ifdef isxupper
#undef isxupper
#endif
#define isxupper(c)  (isdigit(c) || ((unsigned char)'A' <= ((unsigned char)(c)) && ((unsigned char)(c)) <= (unsigned char)'F'))

#ifdef tolower
#undef tolower
#endif
#define tolower(c)   (isupper(c) ? (((unsigned char)(c)) - (unsigned char)'A' + (unsigned char)'a') : (c))

#ifdef toupper
#undef toupper
#endif
#define toupper(c)   (islower(c) ? (((unsigned char)(c)) - (unsigned char)'a' + (unsigned char)'A') : (c))

#ifdef _AIX
#undef LOCK_SH
#undef LOCK_EX
#undef LOCK_NB
#undef LOCK_UN
#endif
#if !defined(LOCK_SH) || defined(_AIX)
#define LOCK_SH F_LOCK   /* shared lock (for reading) */
#define LOCK_EX F_LOCK   /* exclusive lock (for writing) */
#define LOCK_NB 0        /* don't block when locking */
#define LOCK_UN F_ULOCK  /* unlock */
#define flock(fd, operation) lockf((int)(fd), (int)(operation), 0)
#endif

} // namespace apolloron

#endif
