/******************************************************************************/
/*! @file ansi.cc
    @brief ANSI compatible functions.
    @author Tachibanamasashi, Apolloron Project.
 ******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "apolloron.h"


namespace apolloron {

/*! clone of strcasestr()
   Case insensitive string search
   @param src string
   @param needle sub-string
   @return pointer of sub-string in target string
 */
char *apl_strcasestr(const char *src, const char *needle) {
  const char *p;
  int i;
  char *ret;

  if (!src || !*src || !needle) {
    ret = NULL;
  } else if (!*needle) {
    ret = (char *)src;
  } else {
    ret = NULL;
    for (p = src; *p; p++) {
      for (i = 0; needle[i]; i++) {
        if (toupper(p[i]) != toupper(needle[i])) break;
      }
      if (!(needle[i])) {
        ret = (char *)p;
        break;
      }
    }
  }

  return ret;
}


/*! clone of strcasecmp()
   Case insensitive string compare
   @param  str1 Compare text 1.
   @param  str2 Compare text 2.
   @retval 0
   @return <0      str1 < str2
   @return 0<      str2 < str1
 */
int apl_strcasecmp(const char *str1, const char *str2) {
  int first, second;

  do {
    first  = tolower(*str1);
    second = tolower(*str2);
    str1++;
    str2++;
  } while (first && first == second);

  return (first - second);
}


/*! clone of strcasencmp()
    @param str1 Compare text 1.
    @param str2 Compare text 2.
    @param n Length of text.
    @retval Result of compare: Result (-1: str1 < str2  0: str1 == str2  1: str1 > str2).
 */
int apl_strncasecmp(const char *str1, const char *str2, long n) {
  int ret;

  ret = 0;
  if ((str1 == NULL && str2 == NULL) || n <= 0L) {
    ret = 0;
  } else if (str1 == NULL) {
    ret = -1;
  } else if (str2 == NULL) {
    ret = 1;
  } else {
    long col;
    col = 0L;
    while (0L <= col && col < n) {
      unsigned char up_str1, up_str2;

      if (str1[col] == '\0' && str2[col] == '\0') {
        ret = 0;
        break;
      } else if (str1[col] == '\0') {
        ret = -1;
        break;
      } else if (str2[col] == '\0') {
        ret = 1;
        break;
      }

      if ('a' <= (unsigned char)(str1[col]) && (unsigned char)(str1[col]) <= 'z') {
        up_str1 = (unsigned char)('A' + ((unsigned char)(str1[col]) - 'a'));
      } else {
        up_str1 = (unsigned char)str1[col];
      }
      if ('a' <= (unsigned char)(str2[col]) && (unsigned char)(str2[col]) <= 'z') {
        up_str2 = (unsigned char)('A' + ((unsigned char)(str2[col]) - 'a'));
      } else {
        up_str2 = (unsigned char)str2[col];
      }
      if (up_str1 != up_str2) {
        ret = (int)up_str1 - (int)up_str2;
        break;
      }
      col++;
    }
  }

  return ret;
}


/*! convert a integer to string
    @param buf Buffer to store converted value.
    @param value int value to convert
    @param s Signed(1) or unsigned(0)
    @param base Base above 36, or below 1 (default=10)
    @retval length of result string
 */
int itoa(char *buf, int value, int s, int base) {
  int col1, col2, len;
  unsigned int i;

  col1 = 0;

  if (s != 0 && value < 0) {
    buf[col1] = '-';
    col1++;
    i = (unsigned int)(0-value);
  } else {
    i = (unsigned int)value;
  }

  col2 = col1;
  do {
    buf[col1] = i%base > 9 ? 'A' + i%base - 10 : '0' + i%base;
    i = i / base;
    col1++;
  } while(i != 0);

  buf[col1] = '\0';
  len = col1;
  col1--;

  while (col2 < col1) {
    buf[col1] ^= buf[col2];
    buf[col2] ^= buf[col1];
    buf[col1] ^= buf[col2];
    col1--;
    col2++;
  }

  return len;
}


/*! convert a long int to string
    @param buf Buffer to store converted value.
    @param value long value to convert
    @param s Signed(1) or unsigned(0)
    @param base Base above 36, or below 1 (default=10)
    @retval length of result string
 */
int ltoa(char *buf, long value, int s, int base) {
  int col1, col2, len;
  unsigned long u;

  col1 = 0;

  if (s != 0 && value < 0) {
    buf[col1] = '-';
    col1++;
    u = (unsigned long)(0-value);
  } else {
    u = (unsigned long)value;
  }

  col2 = col1;
  do {
    buf[col1] = u%base > 9 ? 'A' + u%base - 10 : '0' + u%base;
    u = u / base;
    col1++;
  } while(u != 0);

  buf[col1] = '\0';
  len = col1;
  col1--;

  while (col2 < col1) {
    buf[col1] ^= buf[col2];
    buf[col2] ^= buf[col1];
    buf[col1] ^= buf[col2];
    col1--;
    col2++;
  }

  return len;
}


/*! convert a double to string
    @param buf Buffer to store converted value.
    @param value double value to convert
    @return length of result string
 */
int dtoa(char *buf, double value) {
  const int FRACT_DIGIT_MAX = 9; // max of fractional digits (limited to 9)
  int i, pre, zero_len, remainder, len;
  double flg, zeros, round;
  long  fract_part;

  if (value == 0.0) {
    buf[0] = '0';
    buf[1] = '.';
    buf[2] = '0';
    buf[3] = '\0';
    return 3;
  }

  len = 0;
  if (value < 0.0) {
    value = -value;
    buf[len++] = '-';
  }

  // This could also be done with an array 1.0, 10.0, 100.0 etc.
  zeros = 1.0;
  for (i = 0; i < FRACT_DIGIT_MAX; i++) {
    zeros *= 10;
  }

  round = 0.5/zeros;
  value += round;
  flg = (long)value;
  value = value - flg;

  len += ltoa(&buf[len], (long)flg);
  buf[len++] = '.';

  buf[len+FRACT_DIGIT_MAX] = '\0';
  fract_part = (long)(value * zeros);
  pre = 0;
  zero_len = 0;
  for (i = len+FRACT_DIGIT_MAX-1; len <= i; i--) {
    remainder = fract_part % 10;
    if (pre == 0 && remainder == 0) {
      buf[i] = '\0';
      zero_len++;
    } else {
      buf[i] = remainder + 0x30;
      pre = remainder;
    }
    fract_part /= 10;
  }

  len += FRACT_DIGIT_MAX - zero_len;
  return len;
}


/*! make directory hierarchically (like "mkdir -p")
    @param pathname path name to create
    @param mode mode like 0755 (affect: mode & ~umask & 0777)
    @retval 0  success
    @retval -1 failure
 */
int mkdirp(const char *pathname, mode_t mode) {
    char *pstr;
    int i;

    if (pathname == NULL || pathname[0] == '\0') {
        return -1;
    }

    if (isDirExist(pathname) == true) {
        // already exist
        return 0;
    }

    pstr = new char [strlen(pathname) + 1];
    i = 0;
    while (pstr[i] != '\0') {
        pstr[i] = pathname[i];
        i++;
        if (pathname[i] == '/' || pathname[i] == '\0') {
            pstr[i] = '\0';
            if (isDirExist(pathname) == false) {
                if (mkdir(pstr, mode) != 0) {
                    delete [] pstr;
                    return -1;
                }
            }
        }
    }

    delete [] pstr;
    return 0;
}


} // namespace apolloron
