/*
 * $Id: Converter.java,v 1.5 2002/05/19 13:56:04 ymakise Exp $
 */

/*
 * Copyright (c) 2002, MAKISE Yoshitaro
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. Neither the name of the iModoki nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package jp.sourceforge.imodoki.converter;

import java.io.*;
import java.net.URL;
import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;

/**
 * Ѵʬᥤ
 * ȥǥ쥯ȥiץ jam, jar ե
 * Configurator Ƥӡɬפʥե롣
 */
public class Converter {
    private PrintStream m_errorStream;
    private PrintStream m_logStream;
    private URL m_jamURL;
    private ConverterOption m_converterOption;
    private JAMFile m_jamContent;
    private JarFileInspector m_jarInspector;
    private RuntimeCompileOption m_compileOption;
    private String m_sourceURL;

    /**
     * 󥹥ȥ饯
     */
    public Converter(URL jamURL, ConverterOption converterOption,
                     RuntimeCompileOption compileOoption) {
        m_jamURL = jamURL;
        m_converterOption = converterOption;
        m_compileOption = compileOoption;

        m_errorStream = System.err;
        m_logStream = System.out;
    }

    public void setErrorStream(PrintStream stream) {
        m_errorStream = stream;
    }

    public void setLogStream(PrintStream stream) {
        m_logStream = stream;
    }

    /**
     * ѴԤʤ
     *
     * @return  Ѵ true
     *          ant ƤӽФˤäƤʤ
     */
    public boolean convert() {
        File iappDir = new File(m_converterOption.getWorkDir(), "iappli");
        File binDir = new File(m_converterOption.getWorkDir(), "bin");

        /* ȥǥ쥯ȥ */
        if (mkdirs(m_converterOption.getWorkDir()) == false)
            return false;
        if (mkdirs(iappDir) == false)
            return false;
        if (mkdirs(binDir) == false)
            return false;

        /* jam ե iappDir ¸ */
        File jamFile = new File(iappDir, getURLBaseName(m_jamURL));
        log("Getting iappli jam file: " + m_jamURL);
        try {
            saveURL(m_jamURL, jamFile);
        } catch (IOException ioe) {
            m_errorStream.println("Failed to get jam file '" + m_jamURL +
                                  "' (" + ioe.toString() + ")");
            return false;
        }

        /* jam եȤĴ٤ */
        try {
            m_jamContent = new JAMFile(jamFile);
        } catch (IOException ioe) {
            m_errorStream.println("Failed to parse jam file '" + jamFile +
                                  "' (" + ioe.toString() + ")");
            return false;
        }

        /* jar ե iappDir ¸ */

        String packageURL = m_jamContent.getPackageURL();
        URL jarURL;
        try {
            jarURL = new URL(m_jamURL, packageURL);
        } catch (MalformedURLException urle) {
            m_errorStream.println("Failed to parse PackageURL '" + packageURL +
                                  "' (" + urle.toString() + ")");
            return false;
        }

        log("Getting iappli jar file: " + jarURL);

        String baseName = getURLBaseName(jarURL);
        int idx = baseName.lastIndexOf('.');
        if (idx != -1)
            baseName = baseName.substring(0, idx);

        File jarFile = new File(iappDir, baseName + ".jar");
        try {
            saveURL(jarURL, jarFile);
        } catch (IOException ioe) {
            m_errorStream.println("Failed to save jar file '" + jarURL +
                                  "' (" + ioe.toString() + ")");
            return false;
        }

        m_sourceURL = getParentURL(jarURL).toString();

        /* jam, jar Ĵɬפʾ */
        try {
            m_jarInspector = new JarFileInspector(jarFile);
        } catch (IOException ioe) {
            m_errorStream.println("Failed to inspect jar file '" + jarFile +
                                  "' (" + ioe.toString() + ")");
            return false;
        }
        m_compileOption.configure(m_jamContent, m_jarInspector);

        /* MIDlet Manifest ե */
        log("Generating manifest file");
        File manifestFile = new File(binDir, "Manifest.mf");
        try {
            String manifestEncoding = "UTF-8";
            createManifest(manifestFile, manifestEncoding);
        } catch (IOException ioe) {
            m_errorStream.println("Failed to create manifest file '" +
                                  manifestFile + "' (" + ioe.toString() + ")");
            return false;
        }

        /* build.properties ե */
        log("Generating build.properties");
        File propFile = new File(m_converterOption.getWorkDir(),
                                 "build.properties");
        try {
            createBuildProperties(propFile, baseName);
        } catch (IOException ioe) {
            m_errorStream.println("Failed to create property file '" +
                                  propFile + "' (" + ioe.toString() + ")");
            return false;
        }

        /* build.xml ե֤ */
        log("Copying build.xml");
        try {
            FileUtils.copyFile(new File(
                                   new File(System.getProperty("imodoki.home"),
                                            "resource"), "build.xml"),
                               new File(m_converterOption.getWorkDir(),
                                        "build.xml"));
        } catch (IOException ioe) {
            m_errorStream.println("Failed to copy build.xml " +
                                  " (" + ioe.toString() + ")");
            return false;
        }

        try {
            m_jarInspector.close();
        } catch (IOException ioe) {
            m_errorStream.println("Warning: failed to close jar file '" +
                                  jarFile + "' (" + ioe.toString() + ")");
        }

        if (!m_converterOption.isKeepWorkDir()) {
            log("Launching ant");
            execAnt(new File(m_converterOption.getWorkDir(), "build.xml"),
                    new String[] { "output", "remove-workdir" });
            throw new RuntimeException("Unreachable; exec ant failed?");
        } else {
            return true;
        }
    }

    /** MIDlet Manifest ե롣 */
    private void createManifest(File file, String encoding)
        throws IOException {
        Map content = new TreeMap();

        content.put("MIDlet-Name", m_jamContent.getAppName());
        content.put("MIDlet-Version", "1.0");
        content.put("MIDlet-Vendor", System.getProperty("user.name"));
        content.put("MIDlet-1", m_jamContent.getAppName() + ",," +
                                "imodoki.MainMIDlet");
        content.put("MicroEdition-Configuration", "CLDC-1.0");
        content.put("MicroEdition-Profile", "MIDP-1.0");
        if (m_jamContent.getSPsize() > 0) {
            content.put("MIDlet-Data-Size",
                        String.valueOf(m_jamContent.getSPsize()));
        }
        if (m_jarInspector.callsMethod("com.nttdocomo.ui.IApplication",
                                       "getSourceURL",
                                       "()Ljava/lang/String;")) {
            content.put("IMDK-SourceURL", m_sourceURL);
        }
        String appParam = m_jamContent.getAppParam();
        if (appParam != null && appParam.length() > 0) {
            content.put("IMDK-AppParam", appParam);
        }

        writeManifest(file, encoding, content);
    }

    /** MIDlet Manifest ե񤭽Ф */
    private static void writeManifest(File file, String encoding, Map content)
        throws IOException {
        String eol = System.getProperty("line.separator");

        Writer out =
            new OutputStreamWriter(new FileOutputStream(file), encoding);
        try {
            Iterator itr = content.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry entry = (Map.Entry)itr.next();
                out.write((String)entry.getKey());
                out.write(": ");
                out.write((String)entry.getValue());
                out.write(eol);
            }
        } finally {
            out.close();
        }
    }

    /** build.properties ե롣 */
    private void createBuildProperties(File file, String baseName)
        throws IOException {
        Properties props = new Properties();

        props.setProperty("app.basename", baseName);
        props.setProperty("iappli.app.class", m_jamContent.getAppClass());
        props.setProperty("output.dir",
                          m_converterOption.getOutputDir().toString());
        props.setProperty("enabled.options",
                          m_compileOption.getBooleanList(true));
        props.setProperty("disabled.options",
                          m_compileOption.getBooleanList(false));

        OutputStream out = new FileOutputStream(file);
        try {
            props.store(out, null);
        } finally {
            out.close();
        }
    }

    /**
     * ant ƤӽФƤӽФȡäƤʤ
     */
    private static void execAnt(File buildfile, String[] targets) {
        String[] options = new String[] { "-emacs",
                                          "-quiet",
                                          "-buildfile",
                                          buildfile.toString() };
        List args = new ArrayList(options.length + targets.length);
        args.addAll(Arrays.asList(options));
        args.addAll(Arrays.asList(targets));
        org.apache.tools.ant.Main.main((String[])args.toArray(new String[0]));
    }

    /** Ϥ롣 */
    private void log(String mesg) {
        if (m_logStream != null) {
            m_logStream.println(mesg);
        }
    }

    /** 顼դ File#mkdirs() */
    private boolean mkdirs(File dir) {
        if (dir.isDirectory())
            return true;

        if (dir.mkdirs() == false) {
            m_errorStream.println("Failed to create directory '" + dir + "'.");
            return false;
        }

        return true;
    }

    /** URL ǻꤵ줿ե̾ʬ֤ */
    private static String getURLBaseName(URL url) {
        String path = url.getPath();
        int idx = path.lastIndexOf('/');
        if (idx == -1) {
            return path;
        } else {
            return path.substring(idx + 1);
        }
    }

    /**
     * ꤵ줿 URL οƤ URL ֤
     * : http://www.hoge.com/abc/def -&gt; http://www.hoge.com/abc/
     */
    private static URL getParentURL(URL url) {
        try {
            return new URL(url, ".");
        } catch (MalformedURLException urle) {
            throw new RuntimeException("Cannot happen: " + urle.toString());
        }
    }

    /** URL ǻꤵ줿꥽ե¸롣 */
    private static void saveURL(URL url, File file) throws IOException {
        InputStream in = null;
        OutputStream out = null;

        try {
            in = url.openStream();
            out = new FileOutputStream(file);
            FileUtils.copyStream(in, out);
        } finally {
            if (in != null)
                in.close();
            if (out != null)
                out.close();
        }
    }
}
