#/*
# *  Copyright 2007-2008 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: BBS.pm 1763 2008-07-21 13:34:23Z hikarin $
#

package Img0ch::BBS;

use strict;

BEGIN {
    my $self    = __PACKAGE__;
    my @classes = qw(
        Archive Cap Log Metadata Plugin Setting
        Subject Template Thread Upload
    );
    for my $class_name (@classes) {
        my $file   = 'Img0ch/' . $class_name . '.pm';
        my $class  = 'Img0ch::' . $class_name;
        my $method = 'get_' . lcfirst($class_name) . '_instance';
        no strict 'refs';
        *{"${self}::${method}"} = sub {
            defined &{"${class}::new"} or require $file;
            my $object  = $class->new(@_);
            my $coderef = $object->can('load');
            $coderef and $coderef->($object);
            return $object;
            }
    }
}

sub new {
    my ( $iClass, $iKernel, $hash ) = @_;
    my $iConfig = $iKernel->get_config();
    my $iRepos  = $iKernel->get_repos( $iKernel->get_repos_path('common') );
    my ( $dir, $id ) = ( ( $hash->{bbs} || '' ), ( $hash->{id} || '0' ) );

    if ( $iConfig->get('compatible.bbs') ) {
        require Img0ch::Compat::BBS;
        Img0ch::Compat::BBS::update( $iRepos, $dir );
    }
    if ($dir) {
        $id = $iRepos->get_int("I:B.d.${dir}");
        $dir = $id ? $dir : '';
    }
    else {
        $dir = $iRepos->get("I:B.i.${id}");
        $id = $dir ? $id : 0;
    }

    bless {
        _dir  => $dir,
        _id   => $id,
        _path => join( '/', $iConfig->get('BBSPath'), $dir ),
        _url  => join( '/', $iConfig->get('Server'), $dir ),
        _utf8 => (
                   $iRepos->get("I:B.i.${id}.utf8")
                || $iRepos->get("I.B.d.${dir}.utf8")
                || 0
        ),
        _kernel => $iKernel,
        _repos  => $iRepos,
        _system =>
            $iKernel->get_repos( $iKernel->get_repos_path("${dir}/system") ),
    }, $iClass;
}

sub get_kernel { $_[0]->{_kernel} }

sub get_common_repos { $_[0]->{_repos} }

sub get_system_repos { $_[0]->{_system} }

sub get_id { $_[0]->{_id} }

sub is_utf8 { $_[0]->{_utf8} ? 1 : 0 }

sub get_encoding { $_[0]->{_utf8} ? 'utf8' : 'sjis' }

*id = \&get_id;

sub get_name { $_[0]->{_dir} }

*bbs = \&get_name;

sub get_repos_path {
    my ( $iBBS, @dirs ) = @_;
    unshift @dirs, $iBBS->{_dir};
    join '/', $iBBS->{_kernel}->get_repos_path( join( '/', @dirs ) );
}

sub path {
    my ( $iBBS, @dirs ) = @_;
    join '/', $iBBS->{_path}, @dirs;
}

sub url {
    my ( $iBBS, @dirs ) = @_;
    join '/', $iBBS->{_url}, @dirs;
}

1;
__END__

=head1 NAME

Img0ch::BBS - 掲示板を管理するクラス

=head1 SYNOPSYS

  use Img0ch::BBS

  my $iBBS = Img0ch::BBS->new($iKernel, $bbs);
  my $id = $iBBS->get_id();
  my $bbs = $iBBS->get_name();

=head1 DESCRIPTION

掲示板単体を1つのオブジェクトとするクラスです。

=head2 new

=over 4

=item Arguments

$iKernel (Img0ch::Kernel), $bbs_dir

=item Return Value

$iBBS (Img0ch::BBS itself)

=back

I<Img0ch::BBS>のオブジェクトを作成します。

=head2 get_kernel

=over 4

=item Arguments

none

=item Return Value

$iKernel

=back

I<Img0ch::Kernel>のオブジェクトを返します。

=head2 get_common_repos

=over 4

=item Arguments

none

=item Return Value

$iReposCommon

=back

共通レポジトリのオブジェクトを返します。
この関数はバージョンが2.1.x-3.xでのみ使用可能です。

=head2 get_system_repos

=over 4

=item Arguments

none

=item Return Value

$iReposSystem

=back

掲示板内のみ有効なシステムレポジトリのオブジェクトを返します。
この関数はバージョンが2.1.x-3.xでのみ使用可能です。

=head2 get_id

=over 4

=item Arguments

none

=item Return Value

$bbs_id

=back

現在の掲示板のIDを返します。存在しない場合は0を返します。

=head2 id

=over 4

=item Arguments

none

=item Return Value

$bbs_id

=back

get_id()と同様ですが、今後削除される予定の非推奨の関数です。

=head2 get_name

=over 4

=item Arguments

none

=item Return Value

$bbs_dir

=back

現在の掲示板のディレクトリ名を返します。存在しない場合はナルストリングを返します。

=head2 get_name

=over 4

=item Arguments

none

=item Return Value

$bbs_dir

=back

get_name()と同様ですが、今後削除される予定の非推奨の関数です。

=head2 path

=over 4

=item Arguments

@dirs

=item Return Value

$joined_path

=back

引数から掲示板のディレクトリの含めたパスを作成します。

=head2 url

=over 4

=item Arguments

@dirs

=item Return Value

$joined_url

=back

引数から掲示板のディレクトリの含めたURLを作成します。

=head2 get_archive_instance

=over 4

=item Arguments

@argument_to_construct

=item Return Value

$iArchive

=back

I<Img0ch::Archive>クラスを生成し、I<Img0ch::Archive>のインスタンスを返します。

=head2 get_cap_instance

=over 4

=item Arguments

@argument_to_construct

=item Return Value

$iCap

=back

I<Img0ch::Cap>クラスを生成し、I<Img0ch::Cap>のインスタンスを返します。

=head2 get_log_instance

=over 4

=item Arguments

@argument_to_construct

=item Return Value

$iLog

=back

I<Img0ch::Log>クラスを生成し、I<Img0ch::Log>のインスタンスを返します。

=head2 get_metadata_instance

=over 4

=item Arguments

@argument_to_construct

=item Return Value

$iMeta

=back

I<Img0ch::Metadata>クラスを生成し、I<Img0ch::Metadata>のインスタンスを返します。

=head2 get_plugin_instance

=over 4

=item Arguments

@argument_to_construct

=item Return Value

$iPlugin

=back

I<Img0ch::Plugin>クラスを生成し、I<Img0ch::Plugin>のインスタンスを返します。

=head2 get_setting_instance

=over 4

=item Arguments

@argument_to_construct

=item Return Value

$iSetting

=back

I<Img0ch::Setting>クラスを生成し、I<Img0ch::Setting>のインスタンスを返します。

=head2 get_subject_instance

=over 4

=item Arguments

@argument_to_construct

=item Return Value

$iSubject

=back

I<Img0ch::Subject>クラスを生成し、I<Img0ch::Subject>のインスタンスを返します。

=head2 get_template_instance

=over 4

=item Arguments

@argument_to_construct

=item Return Value

$iTemplate

=back

I<Img0ch::Template>クラスを生成し、I<Img0ch::Template>のインスタンスを返します。

=head2 get_thread_instance

=over 4

=item Arguments

@argument_to_construct

=item Return Value

$iThread

=back

I<Img0ch::Thread>クラスを生成し、I<Img0ch::Thread>のインスタンスを返します。

=head2 get_upload_instance

=over 4

=item Arguments

@argument_to_construct

=item Return Value

$iUpload

=back

I<Img0ch::Upload>クラスを生成し、I<Img0ch::Upload>のインスタンスを返します。

=head1 AUTHOR

hkrn E<lt>hikarin@users.sourceforge.jpE<gt>

=cut
