#/*
# *  Copyright 2007 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: Interface.pm 1555 2007-11-13 15:55:09Z hikarin $
#

package Zeromin::Log::Interface;

use strict;
use File::Basename qw();
use File::Find qw();

sub retrive_by_day {
    my ( $zError, $iObject, $re, $year, $month, $day ) = @_;
    return $zError->all(
        $re,
        File::Basename::dirname(
            $iObject->get_repos_path("${year}/${month}/${day}/")
        )
    );
}

sub retrive_by_day_with_page {
    my ( $zLog, $year, $month, $day, $item_per_page, $offset ) = @_;
    return _page( $zLog->retrive_by_day( $year, $month, $day ),
        $item_per_page, $offset );
}

sub retrive_by_day_in {
    my ( $zLog, $year, $month, $day, $bbs, $key ) = @_;
    return $zLog->_grep( $zLog->retrive_by_day( $year, $month, $day ),
        $bbs, $key );
}

sub retrive_by_day_in_with_page {
    my ( $zLog, $year, $month, $day, $bbs, $key, $item_per_page, $offset )
        = @_;
    return _page( $zLog->retrive_by_day_in( $year, $month, $day, $bbs, $key ),
        $item_per_page, $offset );
}

sub retrive_by_month {
    my ( $zError, $iObject, $re, $year, $month ) = @_;
    return $zError->all(
        $re,
        File::Basename::dirname(
            $iObject->get_repos_path("${year}/${month}/")
        )
    );
}

sub retrive_by_month_with_page {
    my ( $zLog, $year, $month, $item_per_page, $offset ) = @_;
    return _page( $zLog->retrive_by_month( $year, $month ),
        $item_per_page, $offset );
}

sub retrive_by_month_in {
    my ( $zLog, $year, $month, $bbs, $key ) = @_;
    return $zLog->_grep( $zLog->retrive_by_month( $year, $month ), $bbs,
        $key );
}

sub retrive_by_month_in_with_page {
    my ( $zLog, $year, $month, $bbs, $key, $item_per_page, $offset ) = @_;
    return _page( $zLog->retrive_by_month_in( $year, $month, $bbs, $key ),
        $item_per_page, $offset );
}

sub all {
    my ( $zLog, $re, $start_path ) = @_;
    my $iKernel = $zLog->{_kernel};

    my $path = $start_path
        || File::Basename::dirname( $iKernel->get_repos_path(0) );
    my $repositories = [];
    File::Find::find(
        {   bydepth  => 1,
            no_chdir => 1,
            untaint  => 1,
            wanted   => sub {
                $File::Find::name =~ $re
                    and push @$repositories, $File::Find::name;
                }
        },
        $path,
    );
    return $zLog->_retrive( $repositories, $iKernel );
}

sub all_with_page {
    my ( $zLog, $item_per_page, $offset ) = @_;
    return _page( $zLog->all(), $item_per_page, $offset );
}

sub all_in {
    my ( $zLog, $bbs, $key ) = @_;
    return $zLog->_grep( $zLog->all(), $bbs, $key );
}

sub all_in_with_page {
    my ( $zLog, $bbs, $key, $item_per_page, $offset ) = @_;
    return _page( $zLog->all_in( $bbs, $key ), $item_per_page, $offset );
}

sub to_array {
    my ( $zLog, $re ) = @_;
    my $date_table = {};

    $zLog->{_rs}->iterate(
        sub {
            my ( $key, $value, $date_table, $re ) = @_;
            if ( $key =~ $re ) {
                $date_table->{$1} = 1;
            }
            return 0;
        },
        $date_table,
        $re,
    );
    return [ keys %$date_table ];
}

sub to_array { require Carp; Carp::croak('This is a virtual method') }

sub to_array_with_page {
    my ( $zLog, $item_per_page, $offset ) = @_;
    return _page( $zLog->to_array(), $item_per_page, $offset );
}

sub _page {
    my ( $entries, $item_per_page, $offset ) = @_;
    defined $Data::Page::VERSION or require Data::Page;
    my $page = Data::Page->new( scalar @$entries, $item_per_page, $offset );
    return ( [ $page->splice($entries) ], $page );
}

sub _retrive { require Carp; Carp::croak('This is a virtual method') }

sub _grep {
    my ( $zError, $result, $bbs, $key ) = @_;
    my $new_result = [];

    $bbs ||= $zError->{_bbs};
    if ($bbs) {
        @$new_result = grep { $_->{bbs} eq $bbs } @$result;
    }
    if ($key) {
        @$new_result = grep { $_->{key} eq $key } @$result;
    }

    return $new_result;
}

1;
__END__
