#/*
# *  Copyright 2007 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: Metadata.pm 164 2007-02-02 14:52:06Z hikarin $
#

package Img0ch::Metadata;

use strict;

sub new {
    my ( $iClass, $iBBS ) = @_;
    my $iReposCommon = $iBBS->get_common_repos();
    my $iReposSystem = $iBBS->get_system_repos();

    if ( $iBBS->get_kernel()->get_config()->get('compatible.metadata') ) {
        require Img0ch::Compat::Metadata;
        Img0ch::Compat::Metadata::load( $iReposSystem, $iBBS );
    }

    bless {
        _bbs => $iBBS->get_name(),
        _common => $iReposCommon,
        _system => $iReposSystem,
    }, $iClass;
}

sub head { shift->_load( 'head', 0, @_ ) }

sub meta { shift->_load( 'meta', 0, @_ ) }

sub foot { shift->_load( 'foot', 0, @_ ) }

sub main_banner { shift->_load( 'banner.main', 1, @_ ) }

sub sub_banner { shift->_load( 'banner.sub', 1, @_ ) }

sub mobile_banner { shift->_load( 'banner.mobile', 1, @_ ) }

sub thread_end { shift->_load( 'thread', 0, @_ ) }

sub _load {
    my ( $iMeta, $type, $is_global ) = @_;
    if ($is_global) {
        return $iMeta->{_common}->get_binary("I:M.${type}")
            || $iMeta->{_common}->get_binary("I:M.test.${type}");
    }
    else {
        my $bbs = $iMeta->{_bbs};
        return $iMeta->{_system}->get_binary("I:M.${bbs}.${type}")
            || $iMeta->{_common}->get_binary("I:M.${bbs}.${type}");
    }
}

1;
__END__

=head1 NAME

Img0ch::Config - メタデータを格納するクラス

=head1 SYNOPSYS

  use Img0ch::Metadata

  my $iMeta = Img0ch::Metadata->new($iBBS);

  # head.txtの中身を取り出す
  my $head = $iMeta->head();

  # meta.txtの中身を取り出す
  my $meta = $iMeta->meta();

  # foot.txtの中身を取り出す
  my $foot = $iMeta->foot();

  # 告知欄の中身を取り出す
  my $main_banner = $iMeta->main_banner();

  # 副告知欄の中身を取り出す
  my $sub_banner = $iMeta->sub_banner();

  # 携帯用告知欄の中身を取り出す
  my $mobile_banner = $iMeta->mobile_banner();

  # 1000.txtの中身を表示する
  my $thread_end = $iMeta->thread_end();

=head1 DESCRIPTION

1つのメタデータを1つのオブジェクトとするクラスです。

=head2 new

=over 4

=item Arguments

$iBBS (Img0ch::BBS)

=item Return Value

$iMeta (Img0ch::Metadata itself)

=back

I<Img0ch::Metadata>のオブジェクトを作成します。

=head2 head

=over 4

=item Arguments

none

=item Return Value

$content

=back

head.txtの内容を返します。

=head2 meta

=over 4

=item Arguments

none

=item Return Value

$content

=back

meta.txtの内容を返します。

=head2 foot

=over 4

=item Arguments

none

=item Return Value

$content

=back

foot.txtの内容を返します。

=head2 main_banner

=over 4

=item Arguments

none

=item Return Value

$content

=back

告知欄の内容を返します。

=head2 sub_banner

=over 4

=item Arguments

none

=item Return Value

$content

=back

副告知欄の内容を返します。

=head2 mobile_banner

=over 4

=item Arguments

none

=item Return Value

$content

=back

携帯用告知欄の内容を返します。

=head2 thread_end

=over 4

=item Arguments

none

=item Return Value

$content

=back

1000.txtの内容を返します。

=head1 AUTHOR

hkrn E<lt>hikarin@users.sourceforge.jpE<gt>

=cut
