#/*
# *  Copyright 2007 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: Setting.pm 60 2006-12-31 00:29:40Z hikarin $
#

package Zeromin::Setting;

use strict;
use base qw(Img0ch::Setting);
use Config::Tiny qw();

my $subs = {
    'integer' => sub {
        my ( $value, $boolean ) = @_;
        Img0ch::Kernel::intval($value);
    },
    'boolean' => sub {
        my ( $value, $boolean ) = @_;
        $boolean eq $value ? 1 : 0;
    },
    'string' => sub { return $_[0] },
};

my $stringify = {
    'integer' => sub {
        my ( $value, $boolean ) = @_;
        Img0ch::Kernel::intval($value);
    },
    'boolean' => sub {
        my ( $value, $boolean ) = @_;
        $value ? $boolean : '';
    },
    'string' => sub { return $_[0] }
};

sub new {
    my ( $zClass, $iObject ) = @_;
    my $iBBS     = ( ref $iObject || '' ) eq 'Zeromin::BBS'
                 ? $iObject->parent() : $iObject;
    my $iKernel  = $iBBS->get_kernel();
    my $zSetting = $zClass->SUPER::new($iBBS);
    $zSetting->{__bbs}      = $iBBS;
    $zSetting->{__setting}  = {};
    $zSetting->{__kernel}   = $iKernel;
    $zSetting->{__encoding} = $iKernel->get_encoding(1);

    my $base = join '/', $iKernel->get_config()->get('SystemPath'),
        'setting.txt';
    my $config = Config::Tiny->new();
    my $info = $config->read($base) or $iKernel->throw_io_exception($base);
    $zSetting->{__info} = $info;

    while ( my ( $key, undef ) = each %{$info} ) {
        my $value = $zSetting->get($key);
        $zSetting->set( $key, $value );
    }

    $zSetting;
}

sub get { $_[0]->{__setting}->{ $_[1] } || $_[0]->SUPER::get( $_[1] ) }

sub gets {
    my ( $zSetting, $wanted ) = @_;
    my $info = $zSetting->{__info};
    my $ret  = {};

    foreach my $one ( @{$wanted} ) {
        $one or next;
        my $hash = $info->{$one};
        $hash or next;
        my $do = $subs->{ $hash->{type} };
        my $value = $zSetting->get($one) || $hash->{default};
        $ret->{$one} = $do->( $value, $hash->{boolean} );
    }

    $ret;
}

sub get_normalized {
    my ( $zSetting, $key, $unijp ) = @_;
    my $info = $zSetting->{__info};

    my $hash  = $info->{$key} or return '';
    my $do    = $subs->{ $hash->{type} };
    my $value = $zSetting->get($key);

    if ( $unijp and $hash->{type} eq 'string' ) {
        $value = $unijp->set( $value, $zSetting->{__encoding} )->get();
    }
    $value;
}

sub get_default { shift->_get( shift, 'default' ) }

sub get_order { shift->_get( shift, 'order' ) }

sub get_require_privilege { shift->_get( shift, 'require' ) }

sub get_type { shift->_get( shift, 'type' ) }

sub is_open { shift->_get( shift, 'opened' ) }

sub _get {
    my ( $zSetting, $key, $name ) = @_;
    my $info = $zSetting->{__info};
    exists $info->{$key} ? $info->{$key}->{$name} : '';
}

sub set {
    my ( $zSetting, $key, $value ) = @_;
    my $info = $zSetting->{__info};
    my $hash = $info->{$key} || return '';
    my $do   = $subs->{ $hash->{type} };

    $value ||= '';
    $value =~ tr/\r\t//d;
    $value =~ s/\n+/,/g;
    $value =~ s/,+/,/g;
    $zSetting->{__setting}->{$key} = $do->( $value, $hash->{boolean} );
    return;
}

sub merge {
    my ( $zSetting, $iObject ) = @_;
    my $info  = $zSetting->{__info};
    my $merge = {};

    if ( UNIVERSAL::isa( $iObject, __PACKAGE__ ) ) {
        for my $key ( keys %{$info} ) {
            $merge->{$key} = $iObject->get($key);
        }
    }
    elsif ( UNIVERSAL::isa( $iObject, 'HASH' ) ) {
        %{$merge} = %{$iObject};
    }
    else {
        Carp::Clan::croak('Zeromin::Setting or HASH reference not given');
    }

    while ( my ( $key, $value ) = each %{$merge} ) {
        next if $key eq 'BBS_TITLE' or $key eq 'BBS_MODE';
        my $hash = $info->{$key};
        $hash or next;
        my $do = $subs->{ $hash->{type} };
        my $value = $value || $hash->{default};
        $zSetting->set( $key, $do->( $value, $hash->{boolean} ) );
    }

    1;
}

sub keyset {
    my ( $zSetting, $all ) = @_;
    my $info   = $zSetting->{__info};
    my $ret    = [];
    my $keyset = [
        map      { $_->[1] }
            sort { $a->[0] <=> $b->[0] }
            map  { [ $info->{$_}->{order}, $_ ] }
            keys %$info
    ];

    $all ||= 0;
    foreach my $key ( @{$keyset} ) {
        my $hash = $info->{$key};
        ( !$all and !$hash->{opened} ) and next;
        push @$ret, $key;
    }
    $ret;
}

sub type {
    my ( $zSetting, $key ) = @_;
    my $info = $zSetting->{__info};

    exists $info->{$key} ? $info->{$key} : 'string';
}

sub type_int {'integer'}

sub type_boolean {'boolean'}

sub type_string {'string'}

sub save {
    my ( $zSetting, $no_disk, $path ) = @_;
    my $iRepos = $zSetting->{_rs};
    my $bbs    = $zSetting->{_bbs};
    my $cache  = {};
    my $i      = 0;
    my $info   = $zSetting->{__info};
    my $keyset = $zSetting->keyset(1);

    for my $key ( @{$keyset} ) {
        my $hash = $info->{$key};
        my $value = $zSetting->{__setting}->{$key} || '';
        $value = $stringify->{ $hash->{type} }->( $value, $hash->{boolean} );
        $cache->{$key} = $value;
        $iRepos->set( "I:S.${bbs}.${key}", $value );
        $iRepos->set( "I:S.${i}",          $key );
        $i++;
    }
    $iRepos->set( 'I:S._', $i );
    $iRepos->save();

    $no_disk ||= 0;
    my $iBBS = $zSetting->{__bbs};
    if ( !$no_disk ) {
        my $iKernel = $zSetting->{__kernel};
        my $dir     = $iBBS->bbs();
        $path ||= $zSetting->path();
        local ( $!, *FH );
        open *FH, ">${path}"    ## no critic
            or $iKernel->throw_io_exception($path);
        print {*FH} $dir, '@', crypt( $dir, $dir ), "\n";
        for my $key ( @{$keyset} ) {
            my $value = $cache->{$key};
            $info->{$key}->{opened} or next;
            ( $key eq 'BBS_MODE' and $value eq 'none' ) and next;
            print {*FH} $key, '=', $value, "\n"
                or $iKernel->throw_io_exception($path);
        }
        close *FH or $iKernel->throw_io_exception($path);
    }

    require Zeromin::Plugin;
    my $zPlugin = Zeromin::Plugin->new($iBBS);
    $zPlugin->do( 'zeromin.update.setting', $zSetting, [$iBBS] );
    1;
}

sub remove {
    my ( $zSetting, $no_disk ) = @_;
    my $iRepos = $zSetting->{_rs};
    my $path   = $zSetting->path();

    %{ $zSetting->{__setting} } = ();
    $zSetting->flush();
    $iRepos->iterate(
        sub {
            my ( $key, $value, $bbs ) = @_;
            $key =~ /\AI:S\.$bbs\./xms and return -1;
            return 0;
        },
        $zSetting->{_bbs}
    );
    $iRepos->save();
    unlink $path or $zSetting->{__kernel}->throw_io_exception($path);

    1;
}

1;
__END__
