/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkpoboximpl.c,v 1.1.2.11 2003/03/08 12:36:48 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include "skkpoboximpl.h"
#include "skkdictitem.h"
#include "skkcomm.h"
#include "skktypes.h"

static SkkQueryFunctionImpl *get_new (SkkQuery *query);
static GList *do_query (SkkQueryFunctionImpl *impl, const gchar *buf, const gchar *okuri, gint *found_num);
static void destroy (SkkQueryFunctionImpl *impl);
static gchar *get_info (void);
static gint get_type (void);

static SkkQueryFunctionImpl dictimpl = {
	get_new,
	NULL,
	NULL,
	NULL,
	do_query,
	NULL,
	NULL,
	destroy,
	get_info,
	get_type,
	NULL,
	NULL,
};

static SkkQueryFunctionImpl *get_new (SkkQuery *query)
{
	SkkPoboxImpl *impl;
	impl = g_new0 (SkkPoboxImpl, 1);
	impl->impl = dictimpl;
	impl->sock = -1;
	if (!query) {
		impl->hostname = NULL;
		impl->port = NULL;
	} else {
		impl->hostname = skk_query_get_conf_string (query, "skk-pobox-server-host");
		impl->port = skk_query_get_conf_string (query, "skk-pobox-server-port");
	}
	return (SkkQueryFunctionImpl*) impl;
}

static GList *
do_query (SkkQueryFunctionImpl *impl, const gchar *buf, const gchar *okuri, gint *found_num)
{
	GList *ret = NULL;
	gchar *query_result;
	gchar *query_str;
	gchar **list;
	gchar *list_count;
	gchar **item;
	gchar *item_count;
	gint query_status;
	gint found;
	gint i,j;
	SkkPoboxImpl *d = (SkkPoboxImpl *)impl;
	if (!buf)
		return NULL;
retry:
	if (d->sock < 0) {
		d->sock = skk_comm_makesock (d->hostname ? d->hostname : PBSERV_HOST,
				d->port ? d->port : PBSERV_PORT);
	}
	if (d->sock < 0)
		return NULL;
	if (okuri) {
		query_str = g_strdup_printf ("%s%c", buf, okuri[0]);
	} else {
		query_str = g_strdup (buf);
	}
	if (!skk_comm_send (d->sock, SKKSERV_SEARCH, query_str)) {
		close (d->sock);
		goto retry;
	}
	skk_comm_recv (d->sock, SKKSERV_SEARCH, &query_result, &query_status);
	if (query_status == 4 || !query_result) {
		if (query_result) {
			g_free (query_result);
		}
		if (found_num) {
			*found_num = 0;
		}
		g_free (query_str);
		return NULL;
	}
	list = g_strsplit (query_result, "/", strlen (query_result));
	for (i = 0, list_count = list[0], found = 0;
			list_count;
			i++, list_count = list[i]) {
		if (list_count && *list_count == '\0') {
			continue;
		}
		DEBUG_DO(g_message ("%s",list_count));
		item = g_strsplit (list_count, "\t", strlen (list_count));
		for (j = 0, item_count = item[0]; item_count; j++, item_count = item[j]) {
			if (item_count && *item_count == '\0') {
				continue;
			}
			ret = g_list_append (ret, (gpointer) skk_dict_item_new_with_value (item_count, NULL));
			found++;
		}
		g_strfreev (item);
	}
	if (found_num)
		*found_num = found;
	g_free (query_result);
	g_free (query_str);
	g_strfreev (list);
	return ret;
}

static void
destroy (SkkQueryFunctionImpl *impl)
{
	SkkPoboxImpl *d = (SkkPoboxImpl *)impl;
	if (!(d->sock < 0)) {
		skk_comm_bye (d->sock);
		shutdown (d->sock, 2);
	}
	if (d->hostname) {
		g_free (d->hostname);
	}
	if (d->port) {
		g_free (d->port);
	}
	g_free (d);
	return;
}

static gint
get_type (void)
{
	return SKKQUERY_POBOX_SERVER;
}

static gchar *
get_info (void)
{
	return NULL;
}

SkkQueryFunctionImpl *
skk_dict_new (SkkQuery *query)
{
	return get_new (query);
}

SkkQueryFunctionImpl 
*impl_init (SkkQuery *query)
{
	return get_new (query);
}

#ifdef SKKPOBOXIMPL_DEBUG
int
main (void)
{
	SkkQueryFunctionImpl *dict;
	GList *result;
	dict = get_new (NULL);
	result = dict->do_query (dict, "礦", NULL, NULL);
	g_message ("do_query %s", skk_dict_item_to_string_all (result, "/", ";"));
	result = dict->do_query (dict, "k", NULL, NULL);
	g_message ("do_query %s", skk_dict_item_to_string_all (result, "/", ";"));
	dict->destroy (dict);
	return 0;
}
#endif
