/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkmode.c,v 1.4.2.8 2003/03/08 12:35:31 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <stdio.h>
#include <string.h>
#include "skkmode.h"
#include "skkutils.h"
#include "skkdebug.h"

static void clear (SkkMode *mode);

/* Get Status String */
static gchar* get_mode_string (SkkMode *mode);

/* Set Status */
static void set_mode (SkkMode *mode, SkkMajorMode status);
static void set_query_mode (SkkMode *mode, SkkQueryMode status);

/* Get Status */
static SkkMajorMode get_mode (SkkMode *mode);
static SkkQueryMode get_query_mode (SkkMode *mode);

/* Prepare prefix, postfix */
static void prepare_mark (SkkMode *mode);

/* Event */
static void status_emit (SkkMode *mode);
static void query_mode_emit (SkkMode *mode);


typedef struct _SkkModeFunc SkkModeFunc;

struct _SkkModeFunc {
	SkkModeListener listener;
	gpointer user_data;
};

static void
status_emit (SkkMode *mode)
{
	GSList *tmp_list;
	SkkModeFunc *func;
	if (!mode)
		return;
	for (tmp_list = mode->mode_listener; tmp_list; tmp_list = g_slist_next (tmp_list)) {
		func = (SkkModeFunc *)tmp_list->data;
		if (func) {
			func->listener (mode, skk_mode_get_mode (mode), func->user_data);
		}
	}
	return;
}

static void
query_mode_emit (SkkMode *mode)
{
	GSList *tmp_list;
	SkkModeFunc *func;
	if (!mode)
		return;
	for (tmp_list = mode->query_mode_listener; tmp_list; tmp_list = g_slist_next (tmp_list)) {
		func = (SkkModeFunc *)tmp_list->data;
		if (func)
			func->listener (mode, skk_mode_get_query_mode (mode), func->user_data);
	}
	return;
}

void
prepare_mark (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering prepare_mark\n"));
	if (!mode) return;
	switch (mode->query_mode) {
		case SKK_QUERY_NONE:
			if (mode->query_prefix) {
				g_free (mode->query_prefix);
				mode->query_prefix = NULL;
			}
			if (mode->query_suffix) {
				g_free (mode->query_suffix);
				mode->query_suffix = NULL;
			}
			break;
		case SKK_QUERY_BEGIN:
			if (mode->query_prefix) {
				g_free (mode->query_prefix);
			}
			mode->query_prefix = g_strdup ("");
			if (mode->query_suffix) {
				g_free (mode->query_suffix);
				mode->query_suffix = NULL;
			}
			break;
		case SKK_QUERY_OKURI:
			if (mode->query_prefix) {
				g_free (mode->query_prefix);
			}
			mode->query_prefix = g_strdup ("");
			if (mode->query_suffix) {
				g_free (mode->query_suffix);
			}
			mode->query_suffix = g_strdup ("*");
			break;
		case SKK_QUERY_DONE:
			if (mode->query_prefix) {
				g_free (mode->query_prefix);
			}
			mode->query_prefix = g_strdup ("");
			if (mode->query_suffix) {
				g_free (mode->query_suffix);
				mode->query_suffix = NULL;
			}
			break;
		default:
			break;
	}
	return;
}

static gchar *
get_mode_string (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering get_mode_string\n"));
	if (!mode) return NULL;
	if (mode->status_buf) g_free (mode->status_buf);
	switch (mode->mode)
	{
		case SKK_J_MODE:
			if (mode->jisx0201_kana) {
				mode->status_buf = g_strdup (mode->jisx0201_kana_str);
				break;
			}
			if (mode->katakana)
				mode->status_buf = g_strdup (mode->katakana_str);
			else
				mode->status_buf = g_strdup (mode->hirakana_str);
			break;
		case SKK_LATIN_MODE:
			mode->status_buf = g_strdup (mode->latin_str);
			break;
		case SKK_JISX0208_LATIN_MODE:
			mode->status_buf = g_strdup (mode->jisx0208_latin_str);
			break;
		case SKK_ABBREV_MODE:
			/* FIXME need to be customizable ? */
			mode->status_buf = g_strdup (mode->abbrev_str);
			break;
		default:
			mode->status_buf = g_strdup ("Unknown");
			break;
	}
	return mode->status_buf;
}

static void
clear (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering clear\n"));
	if (!mode) return;
	if (mode->query_prefix) {
		g_free (mode->query_prefix);
		mode->query_prefix = NULL;
	}
	if (mode->query_suffix) {
		g_free (mode->query_suffix);
		mode->query_suffix = NULL;
	}
	if (mode->status_buf) {
		g_free (mode->status_buf);
		mode->status_buf = NULL;
	}
	// mode->katakana = katakana;
	if (mode->mode != SKK_J_MODE)
		skk_mode_set_mode (mode, SKK_J_MODE);
	if (mode->query_mode != SKK_QUERY_NONE)
		skk_mode_set_query_mode (mode, SKK_QUERY_NONE);
	return;
}

static void
set_mode (SkkMode *mode, SkkMajorMode status)
{
	DEBUG_DO (printf ("skkmode.c : Entering set_mode\n"));
	if (!mode) return;
	mode->mode = status;
	skk_mode_prepare_mark (mode);
	if (status == SKK_ABBREV_MODE) {
		skk_mode_set_query_mode (mode, SKK_QUERY_BEGIN);
	}
	status_emit (mode);
	return;
}

static void
set_query_mode (SkkMode *mode, SkkQueryMode status)
{
	DEBUG_DO (printf ("skkmode.c : Entering set_query_mode\n"));
	if (!mode)
		return;
	mode->query_mode = status;
	skk_mode_prepare_mark (mode);
	query_mode_emit (mode);
	return;
}	

static SkkQueryMode
get_query_mode (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering get_query_mode\n"));
	if (!mode)
		return SKK_QUERY_NONE;
	return mode->query_mode;
}

static SkkMajorMode
get_mode (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering get_mode\n"));
	if (!mode) return SKK_J_MODE;
	return mode->mode;
}

SkkMode *
skk_mode_new (void)
{
	SkkMode *ret;
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_new\n"));
	ret = g_new (SkkMode,1);
	memset (ret,0,sizeof (SkkMode));
	ret->mode = SKK_J_MODE;
	ret->query_mode = SKK_QUERY_NONE;
	ret->latin_str = g_strdup ("SKK");
	ret->hirakana_str = g_strdup ("");
	ret->katakana_str = g_strdup ("");
	ret->jisx0208_latin_str = g_strdup ("");
	ret->jisx0201_kana_str = g_strdup ("Ⱦ");
	ret->abbrev_str = g_strdup ("a");
	ret->clear = clear;
	ret->set_mode = set_mode;
	ret->set_query_mode = set_query_mode;
	ret->get_mode = get_mode;
	ret->get_mode_string = get_mode_string;
	ret->get_query_mode = get_query_mode;
	ret->prepare_mark = prepare_mark;
	return ret;
}

void
skk_mode_destroy (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_destroy\n"));
	if (!mode) return;
	skk_mode_clear (mode);
	if (mode->mode_listener) {
		skk_utils_slist_free (mode->mode_listener, TRUE, NULL, NULL);
		mode->mode_listener = NULL;
	}
	if (mode->query_mode_listener) {
		skk_utils_slist_free (mode->query_mode_listener, TRUE, NULL, NULL);
		mode->query_mode_listener = NULL;
	}
	if (mode->latin_str)
		g_free (mode->latin_str);
	if (mode->hirakana_str)
		g_free (mode->hirakana_str);
	if (mode->katakana_str)
		g_free (mode->katakana_str);
	if (mode->jisx0208_latin_str)
		g_free (mode->jisx0208_latin_str);
	if (mode->jisx0201_kana_str)
		g_free (mode->jisx0201_kana_str);
	if (mode->abbrev_str)
		g_free (mode->abbrev_str);
	g_free (mode);
	return;
}

void
skk_mode_set_mode (SkkMode *mode,SkkMajorMode status)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_set_mode\n"));
	if (!mode) return;
	if (mode->set_mode) {
		mode->set_mode (mode,status);
	}
	return;
}

void
skk_mode_set_jisx0201_katakana (SkkMode *mode, gboolean val)
{
	if (!mode)
		return;
	if (mode->jisx0201_kana != val) {
		mode->jisx0201_kana = val;
		status_emit (mode);
	}
}

gboolean
skk_mode_get_jisx0201_katakana (SkkMode *mode)
{
	if (!mode)
		return FALSE;
	return mode->jisx0201_kana;
}

void
skk_mode_set_katakana (SkkMode* mode,gboolean val)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_set_katakana\n"));
	if (!mode) return;
	if (mode->katakana != val) {
		mode->katakana = val;
		status_emit (mode);
	}
}

gboolean
skk_mode_get_katakana (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_katakana\n"));
	if (!mode) return FALSE;
	return mode->katakana;
}


void
skk_mode_set_query_mode (SkkMode *mode, SkkQueryMode status)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_set_query_mode\n"));
	if (!mode)
		return;
	if (mode->set_query_mode) {
		mode->set_query_mode (mode, status);
	}
	return;
}

void
skk_mode_get_query_mode_strings (SkkMode *mode, const gchar **prefix, const gchar **append)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_query_mode_strings\n"));
	if (!mode) return;
	if (prefix)
		*prefix = mode->query_prefix;
	if (append)
		*append = mode->query_suffix;
	return;
}

SkkQueryMode
skk_mode_get_query_mode (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_query_mode\n"));
	if (!mode) return SKK_QUERY_NONE;
	if (mode->get_query_mode) {
		return mode->get_query_mode (mode);
	} else {
		return SKK_QUERY_NONE;
	}
}

const gchar*
skk_mode_get_query_mode_prefix (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_query_mode_prefix\n"));
	if (!mode) return NULL;
	return mode->query_prefix;
}

const gchar *
skk_mode_get_query_mode_append (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_query_mode_append\n"));
	if (!mode) return NULL;
	return mode->query_suffix;
}
		

SkkMajorMode
skk_mode_get_mode (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_mode\n"));
	if (!mode) return SKK_J_MODE;
	if (mode->get_mode) {
		return mode->get_mode (mode);
	} else {
		return SKK_J_MODE;
	}
}

void
skk_mode_clear (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_clear\n"));
	if (!mode) return;
	if (mode->clear) {
		mode->clear (mode);
	}
}

gchar*
skk_mode_get_mode_string (SkkMode *mode)
{
	DEBUG_DO (printf ("skkmode.c : Entering skk_mode_get_mode_string\n"));
	if (!mode) 
		return g_strdup ("Unknown");
	if (mode->get_mode_string) {
		return mode->get_mode_string (mode);
	} else {
		return g_strdup ("Unknown");
	}
}

void
skk_mode_prepare_mark (SkkMode *mode)
{
	if (!mode)
		return;
	if (mode->prepare_mark) {
		mode->prepare_mark (mode);
	}
	return;
}

void
skk_mode_add_mode_listener (SkkMode *mode, SkkModeListener listener, gpointer user_data)
{
	SkkModeFunc *func;
	if (!mode)
		return;
	if (!listener)
		return;
	func = g_new (SkkModeFunc, 1);
	func->listener =listener;
	func->user_data = user_data;
	mode->mode_listener = g_slist_append (mode->mode_listener, func);
}

void
skk_mode_add_query_mode_listener (SkkMode *mode, SkkModeListener listener, gpointer user_data)
{
	SkkModeFunc *func;
	if (!mode)
		return;
	if (!listener)
		return;
	func = g_new (SkkModeFunc, 1);
	func->listener = listener;
	func->user_data = user_data;
	mode->query_mode_listener = g_slist_append (mode->query_mode_listener, func);
	return;
}

void
skk_mode_set_latin_str (SkkMode *mode, const gchar *str)
{
	if (!mode || !str)
		return;
	if (mode->latin_str) {
		if (!strcmp (mode->latin_str, str)) {
			return;
		}
		g_free (mode->latin_str);
	}
	mode->latin_str = g_strdup (str);
	status_emit (mode);
	return;
}

void
skk_mode_set_hirakana_str (SkkMode *mode, const gchar *str)
{
	if (!mode || !str)
		return;
	if (mode->hirakana_str) {
		if (!strcmp (mode->hirakana_str, str)) {
			return;
		}
		g_free (mode->hirakana_str);
	}
	mode->hirakana_str = g_strdup (str);
	status_emit (mode);
	return;
}

void
skk_mode_set_katakana_str (SkkMode *mode, const gchar *str)
{
	if (!mode || !str)
		return;
	if (mode->katakana_str) {
		if (!strcmp (mode->katakana_str, str)) {
			return;
		}
		g_free (mode->katakana_str);
	}
	mode->katakana_str = g_strdup (str);
	status_emit (mode);
	return;
}

void
skk_mode_set_jisx0208_latin_str (SkkMode *mode, const gchar *str)
{
	if (!mode || !str)
		return;
	if (mode->jisx0208_latin_str) {
		if (!strcmp (mode->jisx0208_latin_str, str)) {
			return;
		}
		g_free (mode->jisx0208_latin_str);
	}
	mode->jisx0208_latin_str = g_strdup (str);
	status_emit (mode);
	return;
}

void
skk_mode_set_jisx0201_kana_str (SkkMode *mode, const gchar *str)
{
	if (!mode || !str)
		return;
	if (mode->jisx0201_kana_str) {
		if (!strcmp (mode->jisx0201_kana_str, str)) {
			return;
		}
		g_free (mode->jisx0201_kana_str);
	}
	mode->jisx0201_kana_str = g_strdup (str);
	status_emit (mode);
	return;
}

void
skk_mode_set_abbrev_str (SkkMode *mode, const gchar *str)
{
	if (!mode || !str)
		return;
	if (mode->abbrev_str) {
		if (!strcmp (mode->abbrev_str, str)) { /* do nothing */
			return;
		}
		g_free (mode->abbrev_str);
	}
	mode->abbrev_str = g_strdup (str);
	status_emit (mode);
	return;
}

gboolean
skk_mode_remove_mode_listener (SkkMode *mode, SkkModeListener listener)
{
	return FALSE;
}

gboolean
skk_mode_remove_query_mode_listener (SkkMode *mode, SkkModeListener listener)
{
	return FALSE;
}	

#ifdef SKKMODE_DEBUG
static void
mode_listener (SkkMode *mode, gint status, gpointer user_data)
{
	g_message ("status changed to %d", status);
	g_message ("mode string is %s", skk_mode_get_mode_string (mode));
}

static void
query_mode_listener (SkkMode *mode, gint status, gpointer user_data)
{
	g_message ("query_mode changed to %d", status);
}

int
main (void)
{
	SkkMode *mode;
	mode = skk_mode_new ();
	/* add listener */
	skk_mode_add_mode_listener (mode, mode_listener, NULL);
	skk_mode_add_query_mode_listener (mode, query_mode_listener, NULL);
	skk_mode_set_mode (mode, SKK_J_MODE);
	skk_mode_set_query_mode (mode, SKK_QUERY_DONE);
	return 0;
}
#endif
