/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkfunctionmap.c,v 1.1.2.9 2003/04/17 06:19:22 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <glib.h>
#include <string.h>
#include "skkfunctionmap.h"
#include "skkfunctionmap_private.h"
#include "skkutils.h"

static SkkFunctionMapItem* find_item (SkkFunctionMap *func, GSList *list, gint mode, gint submode, gint key, gint mask);


SkkFunctionMap *
skk_function_map_new (void)
{
	SkkFunctionMap *ret;
	int i;
	ret = g_new (SkkFunctionMap, 1);
	memset (ret, 0, sizeof (SkkFunctionMap));
	for (i = 0; i < DefaultItemSize; i++) {
		skk_function_map_add_item (ret, &(skk_default_functionmaps[i]));
	}
	ret->ref_count++;
	return ret;
}

static void
item_free_func (gpointer data, gpointer user_data)
{
	if (data) {
		skk_function_map_item_destroy ((SkkFunctionMapItem *)data);
		data = NULL;
	}
}

void
skk_function_map_clear (SkkFunctionMap *func)
{
	gint i;
	if (!func)
		return;
	if (func->functions) {
		skk_utils_slist_free (func->functions, TRUE, item_free_func, NULL);
	}
	func->functions = NULL;
	for (i = 0; i < DefaultItemSize; i++) {
		skk_function_map_add_item (func, &(skk_default_functionmaps[i]));
	}
	return;
}

void
skk_function_map_destroy (SkkFunctionMap *func)
{
	if (!func)
		return;
	func->ref_count--;
	if (func->ref_count > 0) {
		return;
	}
	if (func->functions) {
		skk_utils_slist_free (func->functions, TRUE, item_free_func, NULL);
	}
	g_free (func);
	return;
}

void
skk_function_map_ref (SkkFunctionMap *func)
{
	if (!func)
		return;
	func->ref_count++;
	return;
}

void
skk_function_map_unref (SkkFunctionMap *func)
{
	if (!func)
		return;
	func->ref_count--;
	if (!func->ref_count)
		skk_function_map_destroy (func);
	return;
}

void
skk_function_map_add_item (SkkFunctionMap *func, SkkFunctionMapItem *item)
{
	SkkFunctionMapItem *rm;
	GSList *ri;
	if (!func)
		return;
	if (!item)
		return;
	rm = find_item (func, func->functions, item->mode, item->submode, item->key, item->mask);
	if (rm) {
		ri = g_slist_find (func->functions, rm);
		func->functions = g_slist_remove_link (func->functions, ri);
		skk_function_map_item_destroy (rm);
		g_slist_free_1 (ri);
	}
	func->functions = g_slist_append (func->functions, item);
	return;
}

static gboolean
find_item_custom (gconstpointer p1, gconstpointer p2) 
{
	SkkFunctionMapItem *item1;
	SkkFunctionMapItem *item2;
	if (!p1)
		return TRUE;
	if (!p2)
		return TRUE;
	item1 = (SkkFunctionMapItem *)p1;
	item2 = (SkkFunctionMapItem *)p2;

	if ( (item1->key == item2->key) &&
			(item1->mode & item2->mode) &&
			(item1->submode & item2->submode) &&
			(item1->mask & item2->mask)) {
		return FALSE;
	}
	return TRUE;
}

static SkkFunctionMapItem*
find_item (SkkFunctionMap *func, GSList *list, gint mode, gint submode, gint key, gint mask)
{
	SkkFunctionMapItem query;
	GSList *ret;
	if (!func)
		return NULL;
	if (!list)
		return NULL;
	memset (&query, 0, sizeof (SkkFunctionMapItem));
	query.mode = mode;
	query.submode = submode;
	query.key = key;
	query.mask = mask;
	ret = g_slist_find_custom (list, (gpointer)&query, find_item_custom);
	if (!ret)
		return NULL;
	return (SkkFunctionMapItem *) ret->data;
}

void
skk_function_map_remove_item (SkkFunctionMap *funcmap, guint mode, guint submode, guint key, guint mask)
{
	SkkFunctionMapItem *item;
	GSList *ri;
	if (!funcmap)
		return;
	item = find_item (funcmap, funcmap->functions, mode, submode, key, mask);
	if (!item)
		return;
	g_message ("%x %x %x %x", mode, submode, key, mask);
	g_message ("%s %x %x %x %x", item->name, item->mode, item->submode, item->key, item->mask);
	item->mode &= ~(mode);
	item->submode &= ~(submode);
	g_message ("%x %x %x %x", item->mode, item->submode, item->key, item->mask);
	if (!item->mode) {
		ri = g_slist_find (funcmap->functions, item);
		funcmap->functions = g_slist_remove_link (funcmap->functions, ri);
		skk_function_map_item_destroy (item);
		g_slist_free_1 (ri);
	}
	return;
}

SkkFunctionMapItem*
skk_function_map_find_function (SkkFunctionMap *funcmap, guint mode, guint submode, guint key, guint mask)
{
	if (!funcmap)
		return NULL;
	return find_item (funcmap, funcmap->functions, mode, submode, key, mask);
}

SkkFunctionMapItem *
skk_function_map_item_new (const gchar *name, guint mode, guint submode, guint key, guint mask, guint actual_key)
{
	SkkFunctionMapItem *ret;
	if (!name)
		return NULL;
	ret = g_new0 (SkkFunctionMapItem, 1);
	ret->name = g_strdup (name);
	ret->mode = mode;
	ret->submode = submode;
	ret->key = key;
	ret->mask = mask;
	ret->actual_key = actual_key;
	ret->user_defined = TRUE;
	return ret;
}

void
skk_function_map_item_destroy (SkkFunctionMapItem *item)
{
	if (!item)
		return;
	g_message ("skk_function_map_item %s destroyed", item->name);
	if (!item->user_defined)  {
		return;
	}
	g_free (item->name);
	g_free (item);
	return;
}

#ifdef SKKFUNC_DEBUG

#include <skkconf.h>
#include <skkkeysymutil.h>

void
callback (SkkConf *conf, GSList *v, const gchar *type, gpointer user_data)
{
	SkkFunctionMap *map = (SkkFunctionMap *)user_data;
	SkkConfValue *value;
	GSList *l;
	gchar *name = NULL;
	guint mode = 0;
	guint submode = 0;
	guint key = 0;
	guint mask = 0;
	guint actual_key = 0;
	g_message ("fuga");
	if (!type)
		return;
	for (l = v; l; l = g_slist_next (l)) {
		value = (SkkConfValue *)l->data;
		if (!strcmp (value->name, "name")) {
			name = value->value;
		} else if (!strcmp (value->name, "mode")) {
			mode = skk_keysym_keyname_to_mode (value->value);
		} else if (!strcmp (value->name, "querymode")) {
			submode = skk_keysym_keyname_to_query_mode (value->value);
		} else if (!strcmp (value->name, "key")) {
			if (!*(value->value))
				key = 0;
			else 
				key = skk_keysym_keyname_to_keycode (value->value);
		} else if (!strcmp (value->name, "mask")) {
			mask = skk_keysym_keyname_to_mask (value->value);
		} else if (!strcmp (value->name, "actualkey")) {
			actual_key = skk_keysym_keyname_to_keycode (value->value);
		}
	}
	if (!strcmp (type, "remove")) {
		skk_function_map_remove_item (map,
				mode, submode, key, mask);
	} else if (!strcmp (type, "add")) {
		skk_function_map_add_item (map, 
				skk_function_map_item_new (name,
					mode, submode, key, mask, actual_key));
	}
	return;
}

int
main (void)
{
	SkkFunctionMap *func;
	SkkFunctionMapItem *item;
	SkkFunctionMapItem *result;
	SkkConf *conf;
	func = skk_function_map_new ();
	conf = skk_conf_new ();
	skk_conf_add_callback (conf, "/iiimf-skk/functionmap", callback, func);
	result = skk_function_map_find_function (func, SKK_FUNC_JMODE, SKK_QUERY_ALL, SKK_VK_j, SKK_CONTROL_MASK);
	if (result)
		g_message ("find result");
	else
		g_message ("result not found");
	skk_conf_load_rc (conf, NULL);
	item = skk_function_map_item_new ("tmp", SKK_FUNC_JMODE, SKK_QUERY_NONE, 'a', SKK_CONTROL_MASK, 'b');
	skk_function_map_add_item (func, item);
	result = skk_function_map_find_function (func, SKK_FUNC_JMODE, SKK_QUERY_ALL, SKK_VK_j, SKK_CONTROL_MASK);
	if (result)
		g_message ("find result");
	else
		g_message ("result not found");
	result = skk_function_map_find_function (func, SKK_FUNC_JMODE, SKK_QUERY_NONE, 'a', SKK_CONTROL_MASK);
	if (result) {
		g_message ("result actual 1 %c ", result->actual_key);
	}
	result = skk_function_map_find_function (func, SKK_FUNC_ABBREV, SKK_QUERY_BEGIN, SKK_VK_SPACE, SKK_CONTROL_MASK);
	if (result) {
		g_message ("result actual 2 %d ", result->actual_key);
	}
	result = skk_function_map_find_function (func, SKK_FUNC_ABBREV, SKK_QUERY_ALL, SKK_VK_h, SKK_CONTROL_MASK);
	if (result) {
		g_message ("result actual 3 %d ", result->actual_key);
	}
	result = skk_function_map_find_function (func, SKK_FUNC_JMODE, SKK_QUERY_ALL , SKK_VK_q, SKK_CONTROL_MASK);
	if (result) {
		g_message ("result actual 4 %d ", result->actual_key);
	}
	result = skk_function_map_find_function (func, SKK_FUNC_JMODE, SKK_QUERY_BEGIN , SKK_VK_q, SKK_CONTROL_MASK);
	if (result) {
		g_message ("result actual 4 %d ", result->actual_key);
	}
	return 0;
}
#endif
