/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2003 Motonobu Ichimura <famao@momonga-linux.org>
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkfunction_private.c,v 1.1.2.10 2003/04/16 02:53:58 famao Exp $ */

/* vi:set ts=4 sw=4: */

#include "skkfunction.h"
#include "skkgadget.h"

#define SKK_FUNCTION_PROTO(x)  static gint x(SkkBuffer *buf, SkkFunctionMapItem *item, gpointer user_data)

typedef struct _SkkFunctionList SkkFunctionList;

struct _SkkFunctionList {
	gchar *name;
	SkkFunctionCallback callback;
};

SKK_FUNCTION_PROTO(skk_backward_delete);
SKK_FUNCTION_PROTO(skk_cancel);
SKK_FUNCTION_PROTO(skk_category_change);
SKK_FUNCTION_PROTO(skk_do_completion);
SKK_FUNCTION_PROTO(skk_completion_next);
SKK_FUNCTION_PROTO(skk_completion_prev);
SKK_FUNCTION_PROTO(skk_complete_translate);
SKK_FUNCTION_PROTO(skk_codetable);
SKK_FUNCTION_PROTO(skk_cursor_left);
SKK_FUNCTION_PROTO(skk_cursor_right);
SKK_FUNCTION_PROTO(skk_kakutei);
SKK_FUNCTION_PROTO(skk_toggle_kana);
SKK_FUNCTION_PROTO(skk_halfwidth_kana_switch);
SKK_FUNCTION_PROTO(skk_mode_change);
SKK_FUNCTION_PROTO(skk_postfix);
SKK_FUNCTION_PROTO(skk_prefix);
SKK_FUNCTION_PROTO(skk_today);
SKK_FUNCTION_PROTO(skk_begin_translate);
SKK_FUNCTION_PROTO(skk_undo);

static SkkFunctionList skk_default_function_lists[] = {
	{"skk-backward-delete", skk_backward_delete},
	{"skk-cancel", skk_cancel},
	{"skk-category-change", skk_category_change},
	{"skk-do-completion", skk_do_completion},
	{"skk-completion-next", skk_completion_next},
	{"skk-completion-prev", skk_completion_prev},
	{"skk-complete-translate", skk_complete_translate},
	{"skk-codetable", skk_codetable},
	{"skk-cursor-left", skk_cursor_left},
	{"skk-cursor-right", skk_cursor_right},
	{"skk-kakutei", skk_kakutei},
	{"skk-toggle-kana", skk_toggle_kana},
	{"skk-halfwidth-kana-switch", skk_halfwidth_kana_switch},
	{"skk-mode-change", skk_mode_change},
	{"skk-postfix", skk_postfix},
	{"skk-prefix", skk_prefix},
	{"skk-today", skk_today},
	{"skk-begin-translate", skk_begin_translate},
	{"skk-undo", skk_undo},
};

#define SkkDefaultFunctionSize ((sizeof (skk_default_function_lists)/sizeof (skk_default_function_lists[0])))


SKK_FUNCTION_PROTO(skk_backward_delete)
{
	SkkQueryMode qmode = skk_buffer_get_query_mode (buf);
	switch (skk_buffer_get_mode (buf)) {
		case SKK_J_MODE:
			if (qmode == SKK_QUERY_DONE &&
					!skk_conf_get_bool (skkbuf_conf (buf), "skk-delete-implies-kakutei")) {
				skk_buffer_undo (buf);
				return SKK_NOTHING;
			}
			if (skk_buffer_has_preedit (buf)) {
				skk_buffer_delete_backward (buf);
				if (skk_buffer_can_clear (buf)) {
					skk_buffer_clear (buf);
				}
				return SKK_NOTHING;
			} else {
				return SKK_NOTRANS;
			}
			break;
		case SKK_ABBREV_MODE:
			skk_buffer_delete_backward (buf);
			if (!skkbuf_p (buf)) {
				skk_buffer_set_mode (buf, SKK_J_MODE);
			}
			return SKK_NOTHING;
			break;
		default:
			return SKK_NOTHING;
			break;
	}
	return -1;
}

SKK_FUNCTION_PROTO(skk_cancel)
{
	SkkQueryMode qmode = skk_buffer_get_query_mode (buf);
	switch (skk_buffer_get_mode (buf)) {
		case SKK_J_MODE:
			if (qmode == SKK_QUERY_DONE) {
				skkbuf_candidate_num (buf) = 0;
				if (skkbuf_okuri (buf)) {
					if (skk_conf_get_bool (skkbuf_conf (buf), "skk-delete-okuri-when-quit")) {
						/* TODO dirty hack */
						g_free (skkbuf_okuri (buf));
						g_free (skkbuf_okurip (buf));
						skkbuf_okuri (buf) = NULL;
						skkbuf_okurip (buf) = NULL;
					}
					skk_buffer_undo (buf);
				} else {
					skk_buffer_undo (buf);
				}
				return SKK_NOTHING;
			}
			skk_buffer_clear (buf);
			return SKK_NOTHING;
			break;
		case SKK_ABBREV_MODE:
			if (qmode == SKK_QUERY_DONE) {
				skkbuf_candidate_num (buf) = 0;
				skk_buffer_undo (buf);
			} else {
				skk_buffer_clear (buf);
			}
			return SKK_NOTHING;
			break;
		default:
			return SKK_NOTHING;
			break;
	}
}

SKK_FUNCTION_PROTO(skk_category_change)
{
	skk_buffer_change_category (buf);
	return SKK_NOTHING;
}

SKK_FUNCTION_PROTO(skk_do_completion)
{
	if (skk_buffer_get_query_mode (buf) == SKK_QUERY_BEGIN) {
		skk_buffer_get_completion (buf);
		return SKK_NOTHING;
	}
	return SKK_NOTRANS;
}

SKK_FUNCTION_PROTO(skk_completion_next)
{
	if (skkbuf_in_completion (buf)) {
		if (skk_conf_get_bool (skkbuf_conf (buf), "skk-dabbrev-like-completion")) {
			skk_buffer_get_completion (buf);
		} else {
			skk_buffer_set_next_completion (buf);
		}
		return SKK_NOTHING;
	} else {
		return SKK_INSERT_CHAR;
	}
}

SKK_FUNCTION_PROTO(skk_completion_prev)
{
	if (skkbuf_in_completion (buf)) {
		skk_buffer_set_prev_completion (buf);
		return SKK_NOTHING;
	} else {
		return SKK_INSERT_CHAR;
	}
	return SKK_NOTHING;
}

SKK_FUNCTION_PROTO(skk_complete_translate)
{
	/* reset search plugin */
	skk_query_set_first (skkbuf_query (buf));
	skk_buffer_get_completion (buf);
	skk_buffer_set_query_mode (buf, SKK_QUERY_DONE);
	skk_buffer_query_dict (buf);
	/* finish */
	skk_buffer_end_completion (buf);
	return SKK_NOTHING;
}

SKK_FUNCTION_PROTO(skk_codetable)
{
	skk_buffer_codetable_emit (buf);
	return SKK_NOTHING;
}

SKK_FUNCTION_PROTO(skk_cursor_left)
{
	skk_buffer_move_left (buf);
	return SKK_NOTHING;
}

SKK_FUNCTION_PROTO(skk_cursor_right)
{
	skk_buffer_move_right (buf);
	return SKK_NOTHING;
}

SKK_FUNCTION_PROTO(skk_kakutei)
{
	SkkQueryMode qmode = skk_buffer_get_query_mode (buf);
	switch (qmode) {
		case SKK_QUERY_NONE:
			if (skk_buffer_has_preedit (buf)) {
#if 0
				skk_buffer_clear (buf);
#endif
				skk_buffer_commit (buf, FALSE);
			} else {
				return SKK_NOTRANS;
			}
			break;
		case SKK_QUERY_BEGIN:
		case SKK_QUERY_OKURI:
			skk_buffer_commit (buf, FALSE);
			skk_buffer_clear (buf);
			break;
		case SKK_QUERY_DONE:
			skk_buffer_commit (buf, TRUE);
			skk_buffer_clear (buf);
			break;
		default:
			break;
	}
	if (skk_conf_get_bool (skkbuf_conf (buf), "skk-egg-like-newline")) {
		return SKK_NOTHING;
	} else {
		return SKK_NOTRANS;
	}
}

SKK_FUNCTION_PROTO(skk_toggle_kana)
{
	gboolean state = skk_buffer_get_katakana (buf);
	skk_buffer_commit (buf, FALSE);
	skk_buffer_clear (buf);
	skk_buffer_set_katakana (buf, !state);
	/* anyway, do */
	skk_buffer_set_jisx0201_katakana (buf, FALSE);
	return SKK_NOTHING;
}

SKK_FUNCTION_PROTO(skk_halfwidth_kana_switch)
{
	gboolean state = skk_buffer_get_jisx0201_katakana (buf);
	skk_buffer_clear (buf);
	skk_buffer_set_jisx0201_katakana (buf, !state);
	return SKK_NOTHING;
}

SKK_FUNCTION_PROTO(skk_mode_change)
{
	if (skk_buffer_has_preedit (buf)) {
		skk_buffer_commit (buf, FALSE);
	}
	skk_buffer_set_jisx0201_katakana (buf, FALSE);
	skk_buffer_set_mode (buf, item->actual_key);
	skk_buffer_preedit_emit (buf);
	return SKK_NOTHING;
}

SKK_FUNCTION_PROTO(skk_postfix)
{
	SkkQueryMode qmode = skk_buffer_get_query_mode (buf);
	if (qmode == SKK_QUERY_DONE) {
		skk_buffer_commit (buf, TRUE);
		skk_buffer_clear (buf);
		skk_buffer_set_query_mode (buf, SKK_QUERY_BEGIN);
		skk_buffer_add_char (buf, item->actual_key);
	}
	return SKK_NOTHING;
}

SKK_FUNCTION_PROTO(skk_prefix)
{
	SkkQueryMode qmode = skk_buffer_get_query_mode (buf);
	if (qmode == SKK_QUERY_BEGIN) {
		skk_buffer_add_char (buf, item->actual_key);
		skk_buffer_set_query_mode (buf, SKK_QUERY_DONE);
		skk_buffer_query_dict (buf);
		return SKK_NOTHING;
	}
	return SKK_NOTHING;
}

SKK_FUNCTION_PROTO(skk_today)
{
	SkkQueryMode qmode = skk_buffer_get_query_mode (buf);
	if (skk_buffer_get_mode (buf) == SKK_J_MODE) {
		gchar *today = skk_gadget_get_jdate ();
		switch (qmode) {
			case SKK_QUERY_NONE:
				skk_buffer_clear (buf);
				skk_buffer_set_result (buf, today);
				break;
			case SKK_QUERY_BEGIN:
			case SKK_QUERY_OKURI:
				skk_buffer_add_preedit (buf, today);
				break;
			case SKK_QUERY_DONE:
				skk_buffer_commit (buf, TRUE);
				skk_buffer_clear (buf);
				skk_buffer_set_result (buf, today);
				break;
			default:
				break;
		}
		if (today)
			g_free (today);
		return SKK_NOTHING;
	}
	return SKK_NOTHING;
}

SKK_FUNCTION_PROTO(skk_begin_translate)
{
	SkkQueryMode qmode = skk_buffer_get_query_mode (buf);
	if (qmode == SKK_QUERY_BEGIN) {
		skk_buffer_set_query_mode (buf, SKK_QUERY_DONE);
		skk_query_set_first (skkbuf_query (buf));
		skk_buffer_query_dict (buf);
		return SKK_NOTHING;
	} else if (qmode == SKK_QUERY_DONE) {
		if (skk_buffer_has_next_candidate (buf)) {
			gint candidate_count = skk_conf_get_num (skkbuf_conf (buf),
					"skk-candidate-count");
			/* TODO */
			if (buf->candidate_count >= candidate_count) {
				buf->candidate_count++;
				skk_buffer_lookup_emit (buf);
			} else {
				skk_buffer_set_next_candidate (buf);
			}
		} else {
			skk_buffer_adddict_emit (buf);
		}
		return SKK_NOTHING;
	}
	return SKK_NOTRANS;
}

SKK_FUNCTION_PROTO(skk_undo)
{
	SkkQueryMode qmode = skk_buffer_get_query_mode (buf);
	if (qmode == SKK_QUERY_DONE) {
		skk_buffer_undo (buf);
		return SKK_NOTHING;
	}
	return SKK_NOTRANS;
}


void
skk_function_add_default_function (SkkFunction *function)
{
	gint i;
	if (!function)
		return;
	for (i = 0; i < SkkDefaultFunctionSize; i++) {
		skk_function_add_function (function, skk_default_function_lists[i].name,
				skk_default_function_lists[i].callback, NULL);
	}
	return;
}
