/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkconv_kana.c,v 1.1.1.1.2.24 2003/04/14 22:26:13 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <fcntl.h>
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef HAVE_DB4_DB_H
#include <db4/db.h>
#elif HAVE_DB3_DB_H
#include <db3/db.h>
#else
#include <db.h>
#endif
#include "skkconv_kana_private.h"
#include "skkutils.h"

/* Initialize */
static gboolean rule_init (SkkConvRule *rule);
static void rule_item_init (SkkConvRule *rule);
static void rule_item_clear (SkkConvRule *rule);

/* Search */
static SkkConvRuleItem *do_search (SkkConvRule *rule, const gchar *key);

/* Finalize */
static void rule_destroy (SkkConvRule *rule);


static SkkConvRuleItem *
do_search (SkkConvRule *rule, const gchar *key)
{
	DBT db_key,db_data;
	int err;
	memset (&db_key,0,sizeof (DBT));
	memset (&db_data,0,sizeof (DBT));
	db_key.size = strlen (key) + 1;
	db_key.data = (void*)key;
	if ((err = rule->rule_db->get (rule->rule_db,NULL,&db_key,&db_data,0)) != 0) {
		if (err != DB_NOTFOUND) 
			fprintf (stderr,"iiimf-skk: do_search %s\n",db_strerror (err));
		return NULL;
	}
	/* FOUND */
	return (SkkConvRuleItem*)(db_data.data);
}

gint
skk_conv_is_exist (SkkConvRule *rule, const gchar *key)
{
	DBT db_key,db_data;
	DBC *db_cursor;
	int err;
	int exist_count = 0;
	int key_len = 0;
	if (!key)
		return 0;
	if (!rule)
		return 0;
	key_len = strlen (key);
	if (!rule->initialized)
		rule_init (rule);
	memset (&db_key,0,sizeof (DBT));
	memset (&db_data,0,sizeof (DBT));
	rule->rule_db->cursor (rule->rule_db,NULL,&db_cursor,0);
	db_cursor->c_get (db_cursor,&db_key,&db_data,DB_FIRST);
	if (!strncmp (key,(const char*)db_key.data,key_len)) {
		exist_count++;
	}
	while (TRUE) {
		err = db_cursor->c_get (db_cursor,&db_key,&db_data,DB_NEXT);
		if (err == DB_NOTFOUND) {
			break;
		}
		if (!strncmp (key,(const char*)db_key.data,key_len)) {
#if 0
			db_cursor->c_close (db_cursor);
			return TRUE;
#else
			exist_count++;
#endif
		}
	}
	db_cursor->c_close (db_cursor);
#if 1
	if (exist_count > 0)
		return exist_count;
	else
		return 0;
#else
	return FALSE;
#endif
}

static void
rule_destroy (SkkConvRule *rule)
{
	if (!rule)
		return;
	if (!rule->initialized)
		return;
	rule_item_clear (rule);
	rule->rule_db->close(rule->rule_db, 0);
	rule->initialized = FALSE;
}

static void
rule_item_clear (SkkConvRule *rule)
{
	DBT db_key, db_data;
	DBC *db_cursor;
	int err;
	if (!rule)
		return;
	if (!rule->initialized)
		return;
	memset (&db_key, 0, sizeof (DBT));
	memset (&db_data, 0, sizeof (DBT));
	rule->rule_db->cursor (rule->rule_db, NULL, &db_cursor, 0);
	db_cursor->c_get (db_cursor, &db_key, &db_data, DB_FIRST);
	skk_conv_rule_item_destroy ((SkkConvRuleItem *)db_data.data);
	db_cursor->c_del (db_cursor, 0);
	while (TRUE) {
		err = db_cursor->c_get (db_cursor, &db_key, &db_data, DB_NEXT);
		if (err == DB_NOTFOUND) {
			break;
		}
		skk_conv_rule_item_destroy ((SkkConvRuleItem *)db_data.data);
		db_cursor->c_del (db_cursor, 0);
	}
	return;
}

gboolean
skk_conv_delete_rule (SkkConvRule *rule, const gchar *key)
{
	DBT db_key, db_data;
	int err;
	if (!rule)
		return FALSE;
	if (!rule->initialized)
		rule_init (rule);
	if (!key)
		return FALSE;
	memset (&db_key, 0, sizeof (DBT));
	memset (&db_data, 0, sizeof (DBT));
	db_key.data = (void *)key;
	db_key.size = strlen (key) + 1;

	if ((err = rule->rule_db->get (rule->rule_db, NULL, &db_key, &db_data, 0)) != 0) {
		if (err != DB_NOTFOUND) {
			fprintf (stderr, "iiimf-skk: do_search %s\n", db_strerror (err));
		}
		return FALSE;
	}
	/* Found */
	skk_conv_rule_item_destroy ((SkkConvRuleItem*)db_data.data);
	rule->rule_db->del (rule->rule_db, NULL, &db_key, 0);
	return TRUE;
}

gboolean
skk_conv_add_rule(SkkConvRule *rule, SkkConvRuleItem *item)
{
	DBT key, data;
	SkkConvRuleItem *remove;
	if (!rule)
		return FALSE;
	if (!rule->initialized)
		rule_init (rule);
	if (!item || !item->key)
		return FALSE;
	memset(&key, 0, sizeof(DBT));
	memset(&data, 0, sizeof(DBT));
	key.data = (void *)item->key;
	key.size = strlen(item->key) + 1;
	data.data = item;
	data.size = sizeof(SkkConvRuleItem);
	remove = do_search (rule, item->key);
	if (remove) {
		skk_conv_rule_item_destroy (remove);
	}
	rule->rule_db->del(rule->rule_db, NULL, &key, 0);
	rule->rule_db->put(rule->rule_db, NULL, &key, &data, 0);
	return TRUE;
}

static gboolean
rule_init (SkkConvRule *rule)
{
	int err;
	if (!rule)
		return FALSE;
	/* Create Database */
	if ((err = db_create (&(rule->rule_db),NULL,0)) != 0) {
		fprintf (stderr,
				"db_create: %s\n",db_strerror (err));
	}
	rule->rule_db->set_errfile(rule->rule_db,stderr);
	rule->rule_db->set_errpfx(rule->rule_db,"iiimf-skk");
	rule->rule_db->set_flags (rule->rule_db,DB_DUPSORT);
	rule->rule_db->open (rule->rule_db, NULL, 0,DB_BTREE, DB_CREATE, 0644);
	rule_item_init (rule);
	rule->initialized = TRUE;
	return TRUE;
}

static void
rule_item_init (SkkConvRule *rule)
{
	int i;
	DBT key,data;
	if (!rule)
		return;
	for ( i = 0; i < SKKCONV_BASE_RULE_SIZE ; i++) {
		memset (&key,0,sizeof (DBT));
		memset (&data, 0, sizeof (DBT));
		key.data = (void*)baselist[i].key;
		key.size = strlen (baselist[i].key) + 1;
		data.data = &(baselist[i]);
		data.size = sizeof (baselist[0]);
		rule->rule_db->put (rule->rule_db,NULL,&key,&data,0);
	}
}

gchar*
skk_conv_get_hiragana (SkkConvRule *rule, const gchar *key, gchar **append)
{
	SkkConvRuleItem *item;
	gchar *ret;
	if (!rule)
		return NULL;
	if (!rule->initialized)
		rule_init (rule);
	item = do_search (rule, key);
	if (item) {
		switch (item->type)
		{
			case SKKCONV_TYPE_NORMAL:
			case SKKCONV_TYPE_USER:
				ret = g_strdup (item->hira);
				if (!append) return ret;
				if (item->append) {
					*append  = g_strdup (item->append);
				} else {
					*append  = NULL;
				}
				return ret;
				break;
			default:
				return NULL;
				break;
		}
	} else {
		return NULL;
	}
}

gchar*
skk_conv_get_katakana (SkkConvRule *rule, const gchar *key, gchar **append)
{
	SkkConvRuleItem* item;
	gchar *ret;
	if (!rule)
		return NULL;
	if (!rule->initialized)
		rule_init (rule);
	item = do_search (rule, key);
	if (item) {
		switch (item->type)
		{
			case SKKCONV_TYPE_NORMAL:
			case SKKCONV_TYPE_USER:
				ret = g_strdup (item->kata);
				if (!append) return ret;
				if (item->append) {
					*append = g_strdup (item->append);
				} else {
					*append = NULL;
				}
				return ret;
				break;
			default:
				return NULL;
				break;
		}
	}
	else {
		return NULL;
	}
}

gchar*
skk_conv_get_func (SkkConvRule *rule, const gchar *key)
{
	SkkConvRuleItem* item;
	if (!rule)
		return NULL;
	if (!rule->initialized)
		rule_init (rule);
	item = do_search (rule, key);
	if (item) {
		switch (item->type) {
			case SKKCONV_TYPE_FUNC:
			case SKKCONV_TYPE_USER_FUNC:
				if (item->kata) {
					/* Dirty Hack */
					return g_strdup (item->kata);
				} else {
					return NULL;
				}
				break;
			default:
				return NULL;
				break;
		}
	} 
	else
		return NULL;
}

SkkConvRuleType
skk_conv_get_type (SkkConvRule *rule, const gchar *key)
{
	SkkConvRuleItem *item;
	if (!rule)
		return SKKCONV_TYPE_NONE;
	if (!rule->initialized)
		rule_init (rule);
	item = do_search (rule, key);
	if (item)
		return item->type;
	else
		return SKKCONV_TYPE_NONE;
}

/**
 * skk_conv_hiragana_to_katakana :
 * @hiragana: Hiragana
 *
 * translate hiragana to katakana
 *
 * Return: If success, translated katakana string, otherwise NULL. It's up to caller to free
 **/
gchar*
skk_conv_hiragana_to_katakana (const gchar *hiragana)
{
	gchar *ret, *buf;
	gint len;
	gint actual_len = 0;
	if (!hiragana)
		return NULL;
	ret = buf = g_new (gchar, strlen (hiragana) + 1);
	while (hiragana && *hiragana) {
		len = skk_utils_charbytes (hiragana);
		if (skk_utils_is_hiragana (hiragana)) {
			if (!strncmp (hiragana, "", 4)) {
#if 1
				memcpy (buf, "", 2);
#else
				strcpy (buf, "");
#endif
				hiragana += 2;
			} else {
				buf[0] = (hiragana[0] & 0xff) + 0x01;
				buf[1] = (hiragana[1] & 0xff);
			}
		} else {
#if 1
			memcpy (buf, hiragana, len);
#else
			strncpy (buf, hiragana, len);
#endif
		}
		hiragana += len;
		buf += len;
		actual_len += len;
	}
	ret = g_realloc (ret, actual_len + 1);
	ret[actual_len] = '\0';
	return ret;
}

/**
 * skk_conv_katakana_to_hiragana :
 * @katakana: Katakana
 *
 * translate katakana to hiragana
 *
 * Return: If success, translated hiragana string, otherwise NULL. It's up to caller to free
 **/
gchar*
skk_conv_katakana_to_hiragana (const gchar *katakana)
{
	/* NOTICE */
	/* katakana has more characters than hiragana */
	gchar *ret, *buf;
	gint len;
	gint actual_len = 0;
	if (!katakana)
		return NULL;
	/* In the case of ֥, we need strlen * 2 */
	ret = buf = g_new (gchar, (strlen (katakana) * 2) + 1);
	while (katakana && *katakana) {
		len = skk_utils_charbytes (katakana);
		if (skk_utils_is_katakana (katakana)) {
			if ((katakana[1] & 0xff) == 0xf4) { /*  */
				strcpy (buf,"");
				/* increment only 2. not 4 */
				buf += 2;
				actual_len += 2;
			} else if (((katakana[1] & 0xff) == 0xf5) ||
					((katakana[1] & 0xff) == 0xf6)) { /*  */
				/* there is no hiragana, not translate */
				strncpy (buf, katakana, len);
			} else {
				buf[0] = (katakana[0] & 0xff) - 0x01;
				buf[1] = (katakana[1] & 0xff);
			}
		} else {
			strncpy (buf, katakana, len);
		}
		katakana += len;
		buf += len;
		actual_len += len;
	}
	ret = g_realloc (ret, actual_len + 1);
	ret[actual_len] = '\0';
	return ret;
}

SkkConvRuleItem *
skk_conv_rule_item_new (void)
{
	SkkConvRuleItem *ret;
	ret = g_new0 (SkkConvRuleItem, 1);
	ret->type = SKKCONV_TYPE_USER;
	return ret;
}

SkkConvRuleItem *
skk_conv_rule_item_new_with_value (const gchar *key, const gchar *hira, const gchar *kata, const gchar *append)
{
	SkkConvRuleItem *ret;
	ret = g_new0 (SkkConvRuleItem, 1);
	ret->key = (key && *key) ? g_strdup (key) : NULL;
	ret->hira = (hira && *hira) ? g_strdup (hira) : NULL;
	ret->kata = (kata && *kata) ? g_strdup (kata) : NULL;
	ret->append = (append && *append) ? g_strdup (append) : NULL;
	ret->type = SKKCONV_TYPE_USER;
	return ret;
}

void
skk_conv_rule_item_destroy (SkkConvRuleItem *item)
{
	if (!item)
		return;
	if (item->type != SKKCONV_TYPE_USER ||
			item->type != SKKCONV_TYPE_USER_FUNC)
		return;
	if (item->key)
		g_free (item->key);
	if (item->hira)
		g_free (item->hira);
	if (item->kata)
		g_free (item->kata);
	if (item->append)
		g_free (item->append);
	g_free (item);
	g_message ("skk_conv_rule_item_destroy");
	return;
}

void
skk_conv_rule_clear (SkkConvRule *rule)
{
	if (!rule)
		return;
	if (!rule->initialized)
		return;
	rule_item_clear (rule);
	rule_item_init (rule);
}

void
skk_conv_rule_ref (SkkConvRule *rule)
{
	if (!rule)
		return;
	rule->ref_count++;
	return;
}

void
skk_conv_rule_unref (SkkConvRule *rule)
{
	if (!rule)
		return;
	rule->ref_count--;
	if (rule->ref_count == 0)
		skk_conv_rule_destroy (rule);
	return;
}

SkkConvRule *
skk_conv_rule_new (void)
{
	SkkConvRule *ret;
	ret = g_new0 (SkkConvRule, 1);
	ret->ref_count = 1;
	ret->rule_db = NULL;
	ret->initialized = FALSE;
	return ret;
}

void
skk_conv_rule_destroy (SkkConvRule *rule)
{
	if (!rule)
		return;
	rule->ref_count--;
	if (rule->ref_count > 0)
		return;
	rule_destroy (rule);
	g_free (rule);
	g_message ("skk_conv_rule_destroy");
	return;
}

#ifdef SKKCONV_KANA_DEBUG
int
main (void)
{
	gchar *result;
	gint i;
	gint count = 1;
	SkkConvRule *rule;

	rule = skk_conv_rule_new ();

	if (getenv ("SKKPROF")) {
		count = 10000;
	}
	for (i = 0; i < count; i++) {
	printf ("%s\n",skk_conv_get_hiragana (rule, "a",NULL));
	printf ("%s\n",skk_conv_get_katakana (rule, "te",NULL));
	result = skk_conv_get_katakana (rule, "fuga",NULL);
	if (result)
		printf ("%s\n",result);
	else
		printf ("fuga is not found\n");
	printf ("%s\n",skk_conv_get_katakana (rule, "bb",NULL));
	printf ("%d\n",skk_conv_get_type (rule, "a"));
	printf ("%d\n",skk_conv_is_exist (rule, "f"));
	printf ("%d\n",skk_conv_is_exist (rule, "n"));
	printf ("%d\n",skk_conv_is_exist (rule, "fuga"));
	printf ("%d\n",skk_conv_is_exist (rule, "fuga"));
	printf ("%s\n",skk_conv_hiragana_to_katakana ("礦1󤭤Ǥ"));
	printf ("%s\n",skk_conv_hiragana_to_katakana (""));
	printf ("%s\n",skk_conv_katakana_to_hiragana ("1ǥ"));
	printf ("%s\n",skk_conv_katakana_to_hiragana (""));
	printf ("%s\n",skk_conv_katakana_to_hiragana (""));
	printf ("%s\n",skk_conv_katakana_to_hiragana ("˥奦祯"));
	}
	return 0;
}
#endif
